#!/usr/bin/env python3 
import argparse
from collections import OrderedDict
import ctypes
from sys import exit
import os
import glob
import configparser
import platform
import shutil
import subprocess
import sys
import ast
import tempfile
import xml.etree.ElementTree as ET
from xml.dom import minidom


kbdpdk_tools_path = '/usr/local/share/kbdpdk/tools'
dpdk_mode = False
'''
This script is intended to configure the IPVC seamlessly on Linux and Windows.
'''

# HELPER

# Display results in color
class bcolors:
    RED   =      '\033[31m'
    GREEN =      '\033[32m'
    YELLOW=      '\033[33m'
    BLUE  =      '\033[34m'
    RED_BGND =   '\033[37;41m'
    GREEN_BGND = '\033[37;42m'
    ENDC  =      '\033[0m'

def green_str(s):
   if platform.system() == "Windows":
      return s
   else:
      return bcolors.GREEN+s+bcolors.ENDC

def red_str(s):
   if platform.system() == "Windows":
      return s
   else:
      return bcolors.RED+s+bcolors.ENDC

def yellow_str(s):
   if platform.system() == "Windows":
      return s
   else:
      return bcolors.YELLOW+s+bcolors.ENDC

def blue_str(s):
   if platform.system() == "Windows":
      return s
   else:
      return bcolors.BLUE+s+bcolors.ENDC

# Compute the next power of 2 of a 32-bit number 
# https://graphics.stanford.edu/~seander/bithacks.html#RoundUpPowerOf2
def next_pow_of_2(v):
   v -= 1
   for i in (1, 2, 4, 8, 16):
      v |= v >> i
   return v + 1

def is_windows():
   if platform.system() == "Windows":
      return True
   else:
      return False

def get_registry_value(key_path, print_error):
   try:
      with winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, key_path) as key_handle:
         value, _ = winreg.QueryValueEx(key_handle, None)  # None represents the default value
         return value
   except Exception as e:
      if print_error:
        print(f"Error: {e}")
      return None

def is_kbdpdk_installed():
   
   if is_windows():
      return True
   
   if os.path.exists("/usr/lib/libkbapi.so"):
      return True
   else:
      return False

def run_os_specific_func(func):
   def wrapper(*args, **kwargs):
      try:
         os_func = globals()[func.__name__+"_"+platform.system()]
         return os_func(*args, **kwargs)
      except KeyError:
         print("Os not supported")
   return wrapper

def parse_int_range(int_range):
   ids = []
   for x in int_range.split(','):
      x = x.strip()
      if len(x) == 0: continue
      if x.isdigit():
         ids.append(int(x))
      elif x[0] == '<':
         ids.extend(range(0,int(x[1:])+1))
      elif '-' in x:
         xr = x.split('-')
         ids.extend(range(int(xr[0].strip()), int(xr[1].strip())+1))
      else:
         continue
   return ids

# query_yes_no taken from http://code.activestate.com/recipes/577058/
def query_yes_no(question, default="yes"):
   """Ask a yes/no question via raw_input() and return their answer.

   "question" is a string that is presented to the user.
   "default" is the presumed answer if the user just hits <Enter>.
       It must be "yes" (the default), "no" or None (meaning
       an answer is required of the user).

   The "answer" return value is True for "yes" or False for "no".
   """
   valid = {"yes": True, "y": True, "ye": True,
            "no": False, "n": False}
   if default is None:
      prompt = " [y/n] "
   elif default == "yes":
      prompt = " [Y/n] "
   elif default == "no":
      prompt = " [y/N] "
   else:
      raise ValueError("invalid default answer: '%s'" % default)

   while True:
      print(question + prompt)
      choice = input().lower()
      if default is not None and choice == '':
         return valid[default]
      elif choice in valid:
         return valid[choice]
      else:
         sys.stdout.write("Please respond with 'yes' or 'no' "
                          "(or 'y' or 'n').\n")

def query_number(question, default):
   prompt = ' [{}] '.format(default)
   while True:
      choice = input(question + prompt + ':')
      if choice == '':
         return default
      elif choice.isdigit():
         return int(choice)
      else:
         print('Please enter a valid number\n')

# easy_input taken from https://code.activestate.com/recipes/578552-easy-user-input/
def easy_input(question, answer=None, default=None):
   """Ask a question, return an answer.
   
   <question> is a string that is presented to the user.
   <answer> is a list of strings presented as a choice. User may type only first letters
   <default> is the presumed answer if the user just hits <Enter>.

   """
   
   if answer is None :
      answer = ['yes', 'no']
   else : 
      answer = [i.lower() for i in answer]
   
   # if <default> is None or <default> is not an expected answers
   # <default> will be the first of the expected answers
   if default is None or default not in answer :
      default = answer[0]
      
   prompt = '/'.join([
      "\x1b[1;1m{0}\x1b[1;m".format(i.capitalize()) if i == default else i
      for i in answer
   ])
   
   while True :
      choice = input("{0} [{1}]: ".format(question, prompt)).lower()

      if default is not None and choice == '':
         return default
      if choice in answer :
         return choice	
         
      valid_answer = { i[:len(choice)] : i for i in answer }
      
      if len(valid_answer) < len(answer) :
         print(" -- Ambiguous, please use a more detailed answer.")
      elif choice in valid_answer :
         return valid_answer[choice]
      else:
         print(" -- Please answer only with {0} or {1}.".format(", ".join(answer[:-1]), answer[-1]))

def print_title(title,separator='='):
   print('')
   print(yellow_str(title))
   separator_line = ''
   for i in range(0,len(title)):
      separator_line+=separator
   print(blue_str(separator_line))
   
def print_error(error_str):
    print(red_str(f"ERROR : {error_str}"))
    
def prettify_xml(xml_root):
   etstr = (ET.tostring(xml_root)).decode("utf-8")
   xmlstr = minidom.parseString(etstr)
   return f'{end_line()}'.join([line for line in xmlstr.toprettyxml(indent=' '*3).split('\n') if line.strip()])

def end_line():
   if is_windows():
      return '\r\n'
   else:
      return '\n'

# SCRIPT FUNC
import configparser


def read_config_file(file_path):
   try:
      config_parser = configparser.ConfigParser(delimiters='=')
      config_parser.optionxform=str # used to keep key case
      config_parser.read(file_path)
   except FileNotFoundError:
      print_error(f"Config file '{file_path}' not found.")
      return None
   except configparser.MissingSectionHeaderError: # if the file comes without section we add one manually
      with open(file_path) as stream:
         config_parser.read_string(f"[default]{end_line()}" + stream.read()) 
   except configparser.ParsingError as e:
      print_error(f"Parsing config file '{file_path}': {e}")
      return None

   config_dict = {}
   for section in config_parser.sections():
      config_dict[section] = {}
      for key, value in config_parser.items(section):
         try:
            config_dict[section][key] = ast.literal_eval(value)
         except Exception:
            config_dict[section][key] = value

   return config_dict


def write_config_file(file_path, config_dict):
   config_parser = configparser.ConfigParser()
   config_parser.optionxform=str # used to keep key case
   config_parser.read_dict(config_dict)

   try:
      with open(file_path, 'w') as config_file:
         config_file.write(f'# ======================={end_line()}')
         config_file.write(f'# IPVC configuration file{end_line()}')
         config_file.write(f'# ======================={end_line()}')
         config_file.write(f'{end_line()}')
         config_file.write(f'# PLEASE DO NOT EDIT THIS FILE DIRECTLY.{end_line()}')
         config_file.write(f'# This file was automatically generated by ipvc_configure.py{end_line()}{end_line()}')
         config_parser.write(config_file)
   except PermissionError:
      print_error(f"Permission denied to write to file '{file_path}'.")
      return False
   except Exception as e:
      print_error(f"Writing config file '{file_path}': {e}")
      return False
   
   print(f"Config saved to {os.path.abspath(file_path)}")
   
   return True

def get_cpu_topology_Linux():
   sockets = []
   cores = []
   core_map = {}
   base_path = "/sys/devices/system/cpu"
   # Read maximum CPU index allowed in the kernel
   fd = open("{}/kernel_max".format(base_path)) 
   max_cpus = int(fd.read())
   fd.close()
   # Retrieve sockets, core and logical cores
   for cpu in range(max_cpus + 1):
      try:
         fd = open("{}/cpu{}/topology/core_id".format(base_path, cpu))
      except IOError:
         continue
      except:
         break
      core = int(fd.read())
      fd.close()
      fd = open("{}/cpu{}/topology/physical_package_id".format(base_path, cpu))
      socket = int(fd.read())
      fd.close()
      if core not in cores:
         cores.append(core)
      if socket not in sockets:
         sockets.append(socket)
      key = (socket, core)
      if key not in core_map:
         core_map[key] = []
      core_map[key].append(cpu)

   return sockets, cores, core_map
  
def get_cpu_topology_Windows():
   sockets = []
   cores = []
   core_map = {}
   cpu = 0
   for processor in processors :
      sockets.append(processor.SocketDesignation)
      nb_cores = processor.NumberOfCores
      nb_logical_cores = processor.NumberOfLogicalProcessors
      for logical_core_index in range(nb_logical_cores):
         core_index = logical_core_index // (nb_logical_cores // nb_cores)
         key = (processor.SocketDesignation, core_index)
         if key not in core_map:
            core_map[key] = []
         core_map[key].append(cpu)
         if core_index not in cores:
            cores.append(core_index)
         cpu += 1
   return sockets, cores, core_map

@run_os_specific_func
def get_cpu_topology():
   pass
## Display the core topology in a formatted table
#
def disp_cpu_topology(sockets, cores, core_map, reserved_cores):
   col1_head = 'Cores'
   col2_head = 'Logical cores'
   tables = [None] * len(sockets)
   for i in range(len(sockets)):
      s = sockets[i]
      tables[i] = {}
      col1_width = max(len(col1_head), len(str(max(cores))))
      col2_width = max(len(col2_head), max([len(str(v)) for v in core_map.values()]))
      table_width = col1_width + col2_width + 7
      
      title_str     = yellow_str('{:^'+str(table_width)+'}')
      top_line_str  = '+'+'-'*(col1_width+col2_width+5)+'+'
      bot_line_str  = top_line_str
      separator_str = '|'+'-'*(col1_width+col2_width+5)+'|'
      row_str       = '| {:^'+str(col1_width)+'} | {:^'+str(col2_width)+'} |'

      tables[i]['width'] = table_width
      tables[i]['lines'] = []
      tables[i]['lines'].append(title_str.format('Socket '+str(s)))
      tables[i]['lines'].append(top_line_str)
      tables[i]['lines'].append(row_str.format(col1_head, col2_head))
      tables[i]['lines'].append(separator_str)
      for c in cores:
         if (s,c) in core_map:
            col2_w = col2_width
            lcore_str = '['
            for lcore in core_map[(s,c)]:
               if lcore in reserved_cores:
                  lcore_str += '*{}*'.format(lcore)+', '
               else:
                  lcore_str += '{}, '.format(lcore)
            lcore_str = lcore_str[:-2]+']'
            row_str = '| {:^'+str(col1_width)+'} | {:^'+str(col2_w)+'} |'
            tables[i]['lines'].append(row_str.format(c, lcore_str))
      tables[i]['lines'].append(bot_line_str)

   # diplay tables 2 by 2 
   for group in range((len(tables)+1)//2):
      idx = 2*group
      if len(tables) > idx+1:
         line_fmt = '{:<'+str(tables[idx]['width'])+'}    {}'
      else:
         line_fmt = '{}'

      for line in range(len(tables[idx]['lines'])):
         l1 = tables[idx]['lines'][line]
         l2 = tables[idx+1]['lines'][line] if len(tables) > idx+1 else None
         print(line_fmt.format(l1, l2))
   if len(reserved_cores) > 0:
      print('Logical cores currently reserved are shown between asterix : {}.'.format('*core*'))

def is_hardware_ptp_available_Linux(interface_name):
   try:
      # Run the ethtool command to get PTP information
      result = subprocess.run(['ethtool', '-T', interface_name], capture_output=True, text=True)
      output = result.stdout

      # Check if hardware PTP is mentioned in the output
      if ('hardware-raw-clock' in output and
            'hardware-transmit' in output and
            'hardware-receive' in output):
         return True
   except Exception as e:
      print(f"Error: {e}")

def is_hardware_ptp_available_Windows(interface_name):
   return False

@run_os_specific_func
def is_hardware_ptp_available(interface_name):
   pass

def get_interface_name_from_pci(pci_address):
    try:
        # Construct the path to the PCI device
        pci_path = f"/sys/bus/pci/devices/{pci_address}/net/"
        
        # Use glob to find the network interface directory
        interface_paths = glob.glob(pci_path + "*")
        
        # Extract the interface name from the path
        if interface_paths:
            interface_name = os.path.basename(interface_paths[0])
            return interface_name
    except Exception as e:
        print(f"Error: {e}")
    
    return None

def prompt_core_param(param,reserved_cores,multiple_cores=False, forbid_core_0 = False):
   sockets, cores, core_map = get_cpu_topology()
   disp_cpu_topology(sockets,cores,core_map,reserved_cores)
   selected_cores = []
   thread_per_core = len(list(core_map.values())[0])
   lcores_count = len(core_map)*thread_per_core
   
   if forbid_core_0:
      first_core = 1
   else:
      first_core = 0

   while True:
      if multiple_cores :
         print('Please enter logical cores ID(s) (from {} to {})'.format(first_core, lcores_count-1))
         resp = input('You can select multiple logical cores in a comma-separated list of ids or ranges: ')
         selected_cores = parse_int_range(resp)
      else:
         resp = input('Please enter a logical core ID (from {} to {}) : '.format(first_core, lcores_count-1))
         try:
            selected_cores = [int(resp)]
         except ValueError:
            print_error(f'Invalid input "{selected_cores}": please only enter numbers')
            continue
      if param['required'] and len(selected_cores) < 1:
         print_error(f'Invalid input "{selected_cores}": please select at least 1 core')
         continue
      if not all(i>=0 and i<lcores_count for i in selected_cores):
         print_error(f'Invalid input "{selected_cores}": select logical core IDs from {first_core} to {lcores_count-1}')
         continue
      if (forbid_core_0 and 0 in selected_cores) and (param['key'] != "forbidden_cores"):
         print_error(f'Invalid input "{selected_cores}": you cannot select logical core ID 0')
         continue
      if not all(core not in reserved_cores for core in selected_cores):
         print_error(f'Invalid input "{selected_cores}": you cannot select already reserved logical core IDs')
         continue
      # Remove duplicates
      selected_cores = list(OrderedDict.fromkeys(selected_cores))
      # Basic checks
      selected_cores.sort()
      print('You have selected {} logical core{}: {}'.format(len(selected_cores), 's' if len(selected_cores)>1 else '', selected_cores))
      break
      
   for core in selected_cores:
      reserved_cores.add(core)
      
   if len(selected_cores) > 0:
      return selected_cores
   else:
      return None
   
def remove_values(vcsconfig_root,path):
   # remove all values corresponding to path
   parents_path = "/".join(path.split("/")[0:-1])
   parents = vcsconfig_root.findall(parents_path)
   if len(parents) > 1:
      print_error(f"Multiple tags for {parents_path}")
      return False
   if len(parents) == 1:
      for child in parents[0].findall(path.split("/")[-1]):
         parents[0].remove(child)
         
def add_values(vcsconfig_root,path,values_array):
   # find the parent of the element describe by path and create all intermediate node
   current_element = vcsconfig_root
   for path_element in path.split("/")[0:-1]:
      previous_element = current_element
      current_element = current_element.find(path_element)
      if current_element is None:
         previous_element.append(ET.Element(path_element))
         current_element = previous_element.find(path_element)
      
   # add the element once or multiple times
   for value in values_array:
      new_element = ET.Element(path.split("/")[-1])
      new_element.text = str(value)
      current_element.append(new_element)
      
def get_value_from_ipvc_config_dict(config_dict,key):
   if key in config_dict.keys():
      return config_dict[key]
   else:
      return None
      
# OS SPECIFIC SCRIPT FUNC

def check_access_rights_Linux():
   if os.getuid() != 0:
      print("Please run as root")
      exit(1)

def check_access_rights_Windows():
   if ctypes.windll.shell32.IsUserAnAdmin() == 0:
      print("Please run as administrator")
      exit(1)

@run_os_specific_func
def check_access_rights():
   pass

def check_kbdpdk_Linux():
   try:
      sys.path.append('/usr/share/deltacast/kbdpdk/tools')
      global kbtools
      global kbutils
      global kbconfig_setup
      import kbtools
      import kbconfig_setup
      import kbutils
   except ModuleNotFoundError:
      print("KBDPDK is not installed correctly on your system")
      return False
   else:
      return True

def check_kbdpdk_Windows():
   try:
      cwd = os.path.curdir
      os.chdir(get_vcs_path_Windows())
      subprocess.run('kbcheck_config.exe',stdout=subprocess.PIPE,check=True)
      os.chdir(cwd)
   except subprocess.CalledProcessError as e:
      print(e.stdout.decode('utf-8'))
      print(f"KBDPDK is not correctly configured on your system")
   except Exception as e:
      print(f"KBDPDK is not installed correctly on your system - {e}")
      return False
   else:
      return True

@run_os_specific_func
def check_kbdpdk():
   pass

def get_vcs_path_Linux():
   return "/usr/share/deltacast/vcs"

def get_vcs_path_Windows():
   install_location_key = r"SOFTWARE\WOW6432Node\Deltacast.tv\IPVirtualCard"
   # Get the default value of the specified registry key
   value = get_registry_value(install_location_key, False)
   if value is None:
      install_location_key = r"SOFTWARE\Deltacast.tv\IPVirtualCard"
      value = get_registry_value(install_location_key, False)
   if value is None:
        print_error(f"Could not find the registry key {install_location_key}")
        exit(1)
   
   value = os.path.join(value, "vcs")

   return value

@run_os_specific_func
def get_vcs_path():
   pass

def count_substrings(input_string):
    if not input_string:
        return 0
    return len(input_string.split(','))

def print_nics_list(available_nics : list):
   # Print the table header
   print(f"{'Index':<5} {'Available Networks':<20}")
   print("=" * 25)

   # Print each network interface with an index
   for index, network in enumerate(available_nics, start=0):
      print(f"{index:<5} {network:<20}")
   return

def get_nics_list_Linux(nb_v_nics = 0):
   import netifaces
   addresses = netifaces.interfaces()
   available_nics = []
   for intface in addresses:
      if not intface.startswith("veth"):
         available_nics.append(intface)

   if nb_v_nics:
      for i in range(0, nb_v_nics):
         available_nics.append(f"veth{i}")

   return available_nics

def get_nics_list_Windows(nb_v_nics = 0):
   import psutil
   addresses = psutil.net_if_addrs()
   available_nics = []
   for intface, _ in addresses.items():
      if not intface.startswith("veth"):
         available_nics.append(intface)

   if nb_v_nics:
      for i in range(0,nb_v_nics):
         available_nics.append(f"veth{i}")

   return available_nics

@run_os_specific_func
def get_nics_list(nb_v_nics = 0):
   pass

def choose_nic_from_list(nics_list):
   selected = -1
   while True:
      resp = input('Please enter NIC ID (from {} to {}) of device you are receiving PTP on : '.format(0, len(nics_list)-1))
      selected = int(resp) if resp.isdigit() else None
      if selected == None:
         print_error('Invalid input "{}": you must choose one valid NIC Id\n'.format(resp))
         continue
      if selected < 0 or selected >= len(nics_list):
         print_error('Index too high "{}": you must choose one valid NIC Id\n'.format(resp))
         continue
      break
   return selected

# MAIN FUNC

def wizard_config():
   config_dict = {}
   nb_other_nics = 0
   # demande du mode
   print_title('IPVC Configuration Wizard','#')
   ipvc_config_dict = {}
   
   print_title("IPVC mode configuration")
   
   global dpdk_mode
   if is_kbdpdk_installed():
      dpdk_mode = query_yes_no("Do you want to use DPDK with VCS ?","no")
      ipvc_config_dict['dpdk_mode'] = dpdk_mode
   else:
      ipvc_config_dict['dpdk_mode'] = False
      print("KBDPDK is not installed on your system. Configuration will continue for socket mode only.")   

   if dpdk_mode:
      if not check_kbdpdk():
          exit(1)
      if not is_windows():
         # select NIC(s) to use with DPDK
         selected_nics = kbconfig_setup.reserve_nic()
         nb_mellanox_nics = sum(1 for nic in selected_nics if "mellanox" in nic['Vendor_str'].lower())
         nb_other_nics = len(selected_nics) - nb_mellanox_nics
         ipvc_config_dict['nics'] = [nic['Slot'] for nic in selected_nics]

   if dpdk_mode and not is_windows() and nb_mellanox_nics > 0:
       forbid_core_0 = False
   else:
       forbid_core_0 = True

   print_title("IPVC parameters configuration")
   
   # type can be number, core, cores
   params = []
   
   if dpdk_mode:
      if is_windows() or nb_other_nics > 0:
         params.append({
            'name': 'Virtual Interface Polling Core',
            'key': 'virtual_interface_polling_core',
            'type': 'core',
            'required': True,
            'description': 'CPU core that will be reserved and used by VCS to communicate with the Virtual Interface.' + (' This core cannot be core 0.'  if forbid_core_0 else '')
         })
      if not is_windows() and nb_mellanox_nics > 0:
         params.append({
            'name': 'Main DPDK Core',
            'key': 'main_core',
            'type': 'core',
            'required': True,
            'description': 'CPU core that will be reserved for the DPDK master core. This core is mandatory for narrow linear traffic shaping profile.'
         })
      params.append({
         'name': 'Conductor Cores',
         'key': 'conductor_cores',
         'type': 'cores',
         'required': True,
         'description': 'CPU Core(s) that will be able to run Conductors (see documentation about Conductor).'
      })
   params.append({
      'name': 'Forbidden Cores',
      'key': 'forbidden_cores',
      'type': 'cores',
      'required': False,
      'description': 'CPU core(s) that will be ignored by VCS. This is optional.'
   })

   reserved_cores = set()

   for param in params:
      
      print_title(param['name'],'-')
      print(param['description'])
      
      if param['type'] == 'core':
         result = prompt_core_param(param,reserved_cores,multiple_cores=False,forbid_core_0=forbid_core_0)
      if param['type'] == 'cores':
         result = prompt_core_param(param,reserved_cores,multiple_cores=True, forbid_core_0=forbid_core_0)
      if param['type'] == 'number':
         result = query_number("Enter a value",param['default'])
         
      if result is not None:
         ipvc_config_dict[param['key']] = result
      
   # Handle DPDK config generation on windows
   if dpdk_mode and is_windows():
      try:
         tmpdir = tempfile.mkdtemp()
         file_path = os.path.join(tmpdir,"kbtempconf.conf")

         cwd = os.path.curdir
         os.chdir(get_vcs_path_Windows())
         args = [
                  "kbconfig_setup.exe",
                  "--output-file",file_path,
                  "--ipvc-silent"
               ]
         subprocess.run(args,check=True,shell=True)
         os.chdir(cwd)

         kb_config_dict = read_config_file(file_path)
         if kb_config_dict is None:
            raise Exception(f"Could not read {file_path}")
         os.chdir(get_vcs_path())
      except Exception as e:
         print_error(f"Could not generate DPDK config - {e}")
         exit(1)
      finally:
         shutil.rmtree(tmpdir,ignore_errors=True)
         
      config_dict['kbdpdk_config'] = kb_config_dict['default']

   print_title("PTP configuration",'-')
   use_ptp = query_yes_no("Do you want to use PTP with VCS ?","yes")
   if use_ptp:
      #PTP domain
      while True:
         resp = input("Please enter the PTP domain of your PTP grand master (0-127) ? ")
         domain = int(resp) if resp.isdigit() and int(resp) <= 127 else None
         if domain == None:
            print_error(f'PTP domain must be in range of 0 to 127 included')
            continue
         break
      ipvc_config_dict['ptp_domain'] = domain

      print_title("PTP NIC selection",'-')
      multiple_mellanox_nic_selected = False
      
      if dpdk_mode and not is_windows() and nb_mellanox_nics > 0:
         
         print("NIC where you receive PTP. This is mandatory for narrow linear traffic shaping profile with mellanox NICs.")
         nic_list = kbtools.get_nic_list()
         nic_list = [nic for nic in nic_list if "mellanox" in nic['Vendor_str'].lower()]
         kbtools.disp_nic_list(nic_list)
         selected = choose_nic_from_list(nic_list)
         ipvc_config_dict['main_nic'] = nic_list[selected]['Slot']
         ipvc_config_dict['ptp_nic'] = get_interface_name_from_pci(nic_list[selected]['Slot'])
         ipvc_config_dict['timestamp_mode'] = "hardware"
         multiple_mellanox_nic_selected = True

      if not multiple_mellanox_nic_selected:
         nic_list = []
         if is_windows():
            nb_nics = 0
            if 'kbdpdk_config' in config_dict:
               nb_nics = count_substrings(config_dict['kbdpdk_config']['reserved_nics'])
            nics_list = get_nics_list(nb_nics)
         else:
            nics_list = get_nics_list(nb_other_nics)

         print_nics_list(nics_list)
         selected = choose_nic_from_list(nics_list)
         ipvc_config_dict['ptp_nic'] = nics_list[selected]
         if 'veth' not in nics_list[selected]:
            ipvc_config_dict['timestamp_mode'] = "hardware" if is_hardware_ptp_available(nics_list[selected]) else "software"
         else:
            ipvc_config_dict['timestamp_mode'] = "software"

   ipvc_config_dict['use_ptp'] = use_ptp
   config_dict['ipvc_config'] = ipvc_config_dict

   return config_dict
         
         
def apply_config(config_dict):
   # config vcsconfig.xml
   result = True
   isolated_cores = []
   ipvc_config_dict = config_dict['ipvc_config']

   is_dpdk = get_value_from_ipvc_config_dict(ipvc_config_dict,'dpdk_mode')
   if is_dpdk is not None:
      if is_dpdk:
         mode = 'DPDK'
         vcsconfig_name = "vcsconfig_dpdk.xml"
         if not check_kbdpdk():
            return False
      else:
         mode = 'Socket'
         vcsconfig_name = "vcsconfig_socket.xml"
         
      parser = ET.XMLParser(target=ET.TreeBuilder(insert_comments=True))
      vcsconfig_file = ET.parse(os.path.join(get_vcs_path(),vcsconfig_name), parser)
      vcsconfig_root = vcsconfig_file.getroot()
         
      remove_values(vcsconfig_root,"./Core/KernelNetworkMode")
      add_values(vcsconfig_root,"./Core/KernelNetworkMode",[mode])
      
   else:
      print_error("IPVC mode cannot be found in config")
      return False
   
   remove_values(vcsconfig_root,"./Core/DPDK/ConductorCore")
   value = get_value_from_ipvc_config_dict(ipvc_config_dict,'conductor_cores')
   if value is not None:
      add_values(vcsconfig_root,"./Core/DPDK/ConductorCore",value)
      isolated_cores += value
   
   remove_values(vcsconfig_root,"./Core/DPDK/VirtualInterfacePollingCore")   
   value = get_value_from_ipvc_config_dict(ipvc_config_dict,'virtual_interface_polling_core')
   if value is not None:
      add_values(vcsconfig_root,"./Core/DPDK/VirtualInterfacePollingCore",value)
      isolated_cores += value
   
   remove_values(vcsconfig_root,"./Core/DPDK/MainCore")   
   value = get_value_from_ipvc_config_dict(ipvc_config_dict,'main_core')
   if value is not None:
      add_values(vcsconfig_root,"./Core/DPDK/MainCore",value)
      isolated_cores += value
   
   remove_values(vcsconfig_root,"./CPU/ForbiddenCore")   
   value = get_value_from_ipvc_config_dict(ipvc_config_dict,'forbidden_cores')
   if value is not None:
      add_values(vcsconfig_root,"./CPU/ForbiddenCore",value)
   
   remove_values(vcsconfig_root,"./Core/NetworkBufferQueueDepthRX")   
   depth_rx = get_value_from_ipvc_config_dict(ipvc_config_dict,'network_buffer_queue_depth_rx')
   if depth_rx is not None:
      add_values(vcsconfig_root,"./Core/NetworkBufferQueueDepthRX",[depth_rx])
      
   remove_values(vcsconfig_root,"./Core/NetworkBufferQueueDepthTX")
   depth_tx = get_value_from_ipvc_config_dict(ipvc_config_dict,'network_buffer_queue_depth_tx')
   if depth_tx is not None:
      add_values(vcsconfig_root,"./Core/NetworkBufferQueueDepthTX",[depth_tx])
      
   remove_values(vcsconfig_root,"./Network/DPDK/MaxRxQueues")
   queues_rx = get_value_from_ipvc_config_dict(ipvc_config_dict,'max_rx_queues')
   if queues_rx is not None:
      add_values(vcsconfig_root,"./Network/DPDK/MaxRxQueues",[queues_rx])
      
   remove_values(vcsconfig_root,"./Network/DPDK/MaxTxQueues")
   queues_tx = get_value_from_ipvc_config_dict(ipvc_config_dict,'max_tx_queues')
   if queues_tx is not None:
      add_values(vcsconfig_root,"./Network/DPDK/MaxTxQueues",[queues_tx])

   remove_values(vcsconfig_root, "./Ptp")
   if get_value_from_ipvc_config_dict(ipvc_config_dict,'use_ptp'):

      add_values(vcsconfig_root,"./Ptp/LockPtpConfiguration", ["false"])

      remove_values(vcsconfig_root, "./Ptp/DomainNumber")
      ptp_domain = get_value_from_ipvc_config_dict(ipvc_config_dict,'ptp_domain')
      if ptp_domain is not None:
         add_values(vcsconfig_root,"./Ptp/DomainNumber",[ptp_domain])
      
      remove_values(vcsconfig_root, "./Ptp/InterfaceName")
      ptp_nic = get_value_from_ipvc_config_dict(ipvc_config_dict,'ptp_nic')
      if ptp_nic is not None:
         add_values(vcsconfig_root,"./Ptp/InterfaceName",[ptp_nic])

      remove_values(vcsconfig_root, "./Ptp/TimeStamping")
      timestamp_mode = get_value_from_ipvc_config_dict(ipvc_config_dict,'timestamp_mode')
      if timestamp_mode is not None:
         add_values(vcsconfig_root,"./Ptp/TimeStamping",[timestamp_mode])
   else:
      remove_values(vcsconfig_root, "./Ptp")


   remove_values(vcsconfig_root,"./Logger/Debug")
   log_sinks = ''
   if is_windows():
      log_sinks = 'StdOutputLog,DebugApiLog'
   else:
      log_sinks = 'StdOutputLog,SysLog'

   remove_values(vcsconfig_root,"./Logger/Debug")
   add_values(vcsconfig_root,"./Logger/Debug",[log_sinks])
   
   # fix xml file formatting
   xmlstr = prettify_xml(vcsconfig_root)
   
   with open(os.path.join(get_vcs_path(),"vcsconfig.xml"), "w") as f:
      f.write(xmlstr)
      
   # config dpdk
   
   if mode == 'DPDK':
      # apply DPDK config
      if is_windows():
         try:
            kbconfig_dict = {}
            kbconfig_dict['kbdpdk_config'] = config_dict['kbdpdk_config'] # we only want to write the kbdpdk_config section from config_dict
            config_path = os.path.join(get_vcs_path(),'kbconfig.conf')
            write_config_file(config_path,kbconfig_dict)
            # kbdpdk do not support .ini file with section so we remove the section
            with open(config_path, 'r') as f:
               lines = f.readlines()

            with open(config_path, 'w') as f:
               for line in lines:
                  if not line.startswith("["):
                     f.write(line)

         except Exception as e:
            print_error(f"Failed to apply the KBDPDK configuration - {e}")
            result = False
      else:
         selected_nics_pci_id = get_value_from_ipvc_config_dict(ipvc_config_dict,'nics')
         
         nic_list = kbtools.get_nic_list()
         selected_nics = [nic for nic in nic_list if nic['Slot'] in selected_nics_pci_id]
         selected_other_nics_pci_id = tuple(nic['Slot'] for nic in selected_nics if "mellanox" not in nic['Vendor_str'].lower())
         selected_mellanox_nics_pci_id = tuple(nic['Slot'] for nic in selected_nics if "mellanox" in nic['Vendor_str'].lower())
         
         # No huge page are set as KBDPDK now dynamically creates hugepages when needed.
         mem_cfg = []
         nb_numa = len(kbconfig_setup.get_numa_entries())
            
         for numa in range(nb_numa):
            mem_cfg.append({'1G_pages':0, '2M_pages':0})
                        
         tmp_file = tempfile.mktemp()
         kbconfig_setup.save_config((selected_other_nics_pci_id,selected_mellanox_nics_pci_id,('1G',mem_cfg),isolated_cores),tmp_file)
         try:
            subprocess.run(['kbconfig_apply.py', '-s', tmp_file],stdout=subprocess.DEVNULL,check=True)
         except Exception as e:
            print_error(f"Failed to apply the KBDPDK configuration - {e}")
            result = False
         
         print(yellow_str("You need to reboot the system to apply the new configuration"))

         if get_value_from_ipvc_config_dict(ipvc_config_dict,'use_ptp') and len(selected_mellanox_nics_pci_id) > 1:
            # run config_nic_sync to synchronize all mellanox NICs
            ptp_domain = get_value_from_ipvc_config_dict(ipvc_config_dict,'ptp_domain')
            main_nic = get_value_from_ipvc_config_dict(ipvc_config_dict,'main_nic')
            if ptp_domain is not None and main_nic is not None:
               try:
                  command = ["python3",os.path.join(get_vcs_path(),"config_nic_sync.py"),str(ptp_domain),main_nic]
                  command += [nic for nic in selected_mellanox_nics_pci_id if nic != main_nic]
                  subprocess.run(command,stdout=subprocess.DEVNULL,stderr=subprocess.DEVNULL,check=True)
               except Exception as e:
                  print_error(f"Failed to configure mellanox NIC sync - {e}")
                  result = False
   
   if result:
      print("IPVC configured successfully")
   else:
      print_error("Error during IPVC configuration")
   
   return result

# SCRIPT START
result = True

check_access_rights()

parser = argparse.ArgumentParser(description='IPVC configuration script')
parser.add_argument('-c','--config_file',help="Config file path to load")
args = parser.parse_args()

if is_windows():
   from wmi import WMI
   import winreg
   import time
   print("Loading WMI")
   w = WMI()
   processors = w.Win32_Processor()

if args.config_file:
   config_dict = read_config_file(args.config_file)
   if config_dict is None:
      result = False
   else:
      result = apply_config(config_dict)
else:

   config_dict = wizard_config()

   if query_yes_no("Do you want to save the configuration ?"):
      config_default_name = "ipvc_config.cfg"
      result = write_config_file(input(f'Enter filename to save configuration [{config_default_name}]: ') or config_default_name,config_dict)
   if query_yes_no("Do you want to apply the configuration ?"):
      result = apply_config(config_dict)
   print(f"DPDK mode: {dpdk_mode}")
   #Only need to reboot on linux
   if dpdk_mode and not is_windows():
      text="Do you want to reboot the system to apply configuration ?"
   else:
      text="Do you want to restart vcs service to apply configuration ?"
      
   if query_yes_no(text):
      if is_windows():
         command = ['sc','stop','DELTACAST Vcs']
         subprocess.run(command,stdout=subprocess.DEVNULL,stderr=subprocess.DEVNULL,check=False, shell=True)
         service_name= 'DELTACAST Vcs'
         while True:
             status = subprocess.run(['sc', 'query', service_name], stdout=subprocess.PIPE, text=True, shell=True)
             if "STATE" in status.stdout and "STOPPED" in status.stdout:
                 print(f"The {service_name} service has been stopped.")
                 break
             print(f"Waiting for the {service_name} service to be stopped...")
             time.sleep(3)
         
         command = ['sc','start','DELTACAST Vcs']
         subprocess.run(command,stdout=subprocess.DEVNULL,stderr=subprocess.DEVNULL,check=False, shell=True)
      else:
         if dpdk_mode:
            print('System will reboot...')
            command = ['reboot','now']
            subprocess.run(command,stdout=subprocess.DEVNULL,stderr=subprocess.DEVNULL,check=False)
         else:
            print('Restarting vcs service...')
            command = ['systemctl','restart','vcs']
            subprocess.run(command,stdout=subprocess.DEVNULL,stderr=subprocess.DEVNULL,check=False)   
   
if result:
   exit(0)
else:
   exit(1)