#pragma once
/*!
   @file Tools.h
   @brief This file contains some helping functions used in the samples.
*/

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_networkinterface.h>
#include <videomasterip/videomasterip_stream.h>
#include <videomasterip/videomasterip_ptp.h>
#include <videomasterip/videomasterip_cpu.h>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#include <iostream>
#include <string>
#include <vector>

#define PIXELSIZE_10BIT 5/2

/*!
   @brief Structure representing yuv color object.
*/

typedef struct _YUV_10BIT
{
   uint16_t y;
   uint16_t u;
   uint16_t v;
}YUV_10BIT;

/*!
   @brief Structure representing the parameters of an audio stream

   See Also
      <link ESSENCE_PARAM>
*/

typedef struct _AUDIO_PARAM
{
   VMIP_ST2110_30_PACKET_TIME PacketTime; //Packet time
   VMIP_AUDIO_SAMPLING_RATE SamplingRate; //Audio sampling rate
   uint32_t NbChannel; //Number of audio channel
   uint32_t PacketNbPerSlot; // Number of audio packet contain in a lock slot
}AUDIO_PARAM;

/*!
   @brief Structure representing the parameters of a ST2110-22 essence

   See Also
      <link ESSENCE_PARAM>
*/

typedef struct _ST2110_22_PARAM
{
   VMIP_ST2110_22_CODEC Codec; /*Codec. */
   VMIP_VIDEO_STANDARD VideoStandard; /*Video standard. */
   uint32_t FrameSize; //Size of the frame.
}ST2110_22_PARAM;

/*!
   @brief Structure representing the essences parameters.

   See Also
      <link ConfigureStream>
*/

struct ESSENCE_PARAM
{
   VMIP_ESSENCETYPE EssenceType; /*! Essence type. This is used to know which of the following field is used.*/
   union
   {
      VMIP_VIDEO_STANDARD VideoStandard; /*Video standard. Used in ST2110_20 (EssenceType == VMIP_ET_ST2110_20)*/
      AUDIO_PARAM AudioParam; /* Audio Param. Used in ST2110_30 (EssenceType == VMIP_ET_ST2110_30)*/
      VMIP_ANC_SDI_VIDEOSTANDARD AncVideoStandard; /*ANC video standard. Used in ST2110_40 (EssenceType == VMIP_ET_ST2110_40)*/
      ST2110_22_PARAM Param_ST2110_20; /*ST2110-20 Param. Used in ST2110_22 (EssenceType == VMIP_ET_ST2110_22)*/
   };
} ;

/*!
   @brief This function prints the information regarding the detected CPU's. CPU's mentionned as forbidden in the configuration file will not appear.

   @returns The function returns the status of its execution as VMIP_ERRORCODE
*/
VMIP_ERRORCODE PrintCpuCoresInfo(HANDLE VcsContext /*!< [in] Context of the VCS session */);

/*!
   @brief This function prints the information regarding the detected network interfaces. Network interfaces mentionned as forbidden in the configuration file will not appear.

   @returns The function returns the status of its execution as VMIP_ERRORCODE
*/
VMIP_ERRORCODE PrintNicsInfo(HANDLE VcsContext /*!< [in] Context of the VCS session */);

/*!
   @brief The function provides the network interface ID based on its OS name.

   @returns The function returns the status of its execution as VMIP_ERRORCODE
*/
VMIP_ERRORCODE GetNicIdFromName(HANDLE VcsContext /*!< [in] Context of the VCS session */
   , std::string NicName /*!< [in] Name of the network interface */
   , uint64_t* pNicId /*!< [out] ID of the network interface */
);

/*!
   @brief This function provides a string representation of the VMIP_NETWORK_ITF_DUPLEX_MODE enumeration value.

   @returns a string representation of the VMIP_NETWORK_ITF_DUPLEX_MODE enumeration value.
*/
std::string ToString(VMIP_NETWORK_ITF_DUPLEX_MODE DuplexMode /*!< [in] Duplex mode that we want to stringify. */);

/*!
   @brief This function provides a string representation of the VMIP_PTP_STATE enumeration value.

   @returns a string representation of the VMIP_PTP_STATE enumeration value.
*/
std::string ToString(VMIP_PTP_STATE PtpState /*!< [in] PTP state that we want to stringify.*/);

/*!
   @brief This function provides a string representation of the VMIP_CONDUCTOR_AVAILABILITY enumeration value.

   @returns a string representation of the VMIP_CONDUCTOR_AVAILABILITY enumeration value.
*/
std::string ToString(VMIP_CONDUCTOR_AVAILABILITY Value);

/*!
   @brief This function allows to configure a network interface and provides the corresponding ID.

   @detail The call to this function is optional. If it is not called, the OS configuration will not be modified.

   @returns The function returns the status of its execution as VMIP_ERRORCODE
*/
VMIP_ERRORCODE ConfigureNic(HANDLE VcsContext /*!< [in] Context of the VCS session */
   , const std::string NicName /*!< [in] Name of the interface that we want to configure */
   , const uint32_t LocalIpAddress /*!< [in] Ip address that will be configured on the NIC.*/
   , const uint32_t LocalSubnetMask /*!< [in] Subnet mask that will be configured on the NIC.*/
   , uint64_t* pNicId /*!< [out] ID of the NIC*/
);

/*!
   @brief This function manages conductor creation and configuration

   @details
   This core will be busy at 100% during the stream.
   Using the Cpu core 0 should be avoided as interruption on this core can disrupt the traffic shaping.
   In TX, the conductor will be responsible of sending the packets on the network at the correct rate, in order to be compliant with the ST2110-21 norm.
   In RX, the conductor will be responsible of receiving the packets as fast as possible.

   @returns The function returns the status of its execution as VMIP_ERRORCODE
*/
VMIP_ERRORCODE ConfigureConductor(const uint32_t ConductorCpuCoreOsId /*!< [in] Index of the Cpu on which the conductor will run.*/
   , HANDLE VcsContext /*!< [in] Context of the VCS session */
   , uint64_t* pConductorId /*!< [out] Id of the conductor that was created.*/
);

/*!
   @brief This function manages stream creation and configuration

   @returns The function returns the status of its execution as VMIP_ERRORCODE
*/
VMIP_ERRORCODE ConfigureStream(HANDLE VcsContext /*!< [in] Context of the VCS session */
                               , VMIP_STREAMTYPE StreamType /*!< [in] Type of the stream (RX or TX)*/
                               , const std::vector<uint32_t>& rProcessingCpuCoreOsId /*!< [in]  This will be used to know on which core processing job will be launched.
                                                                                                 Putting the conductor core in this list should be avoided.
                                                                                                 If left empty, all the core of the system will be used. */
                               , const std::vector<uint32_t>& rDestinationAddresses /*!< [in] In TX : The packets will be sent toward those addresses.
                                                                                              In RX : This corresponds to the addresses the packets are received on, can be unicast, multicast.
                                                                                              If this is set to 0, wo filtering will be done on the addresses.
                                                                                              The size of the vector indicates the number of different path that will be used. Using multiple paths allows seamless
                                                                                              switching in RX and TX as specified in the norm ST2022-7. The size must correspond to the size of rDestinationUdpPorts and rNicIds */
                               , const std::vector<uint16_t>& rDestinationUdpPorts /*!< [in] In TX : The packets will be sent toward those ports. The size of the table must correspond to NbPathST2022_7.
                                                                                             In RX : Those are the local ports that will be opened.
                                                                                             The size of the vector indicates the number of different path that will be used. Using multiple paths allows seamless
                                                                                             switching in RX and TX as specified in the norm ST2022-7. The size must correspond to the size of rDestinationAddresses and rNicIds */
                               , const uint32_t DestinationSsrc /*!< [in] In TX : The packets will be sent toward this ssrc.
                                                                          In RX : Ssrc of the received packet.
                                                                                  If this is set to 0, wo filtering will be done on the SSRC.    */
                               , const ESSENCE_PARAM EssenceParam /*!< [in] Essence parameters that will be used for the stream.*/
                               , const std::vector<uint64_t>& rNicIds /*!< [in] Ids of the NICs used by the stream
                                                                          The size of the vector indicates the number of different path that will be used. Using multiple paths allows seamless
                                                                          switching in RX and TX as specified in the norm ST2022-7. The size must correspond to the size of rDestinationAddresses and rDestinationUdpPorts */
                               , uint64_t ConductorId /*!< [in] Id of the conductor used by the stream. */
                               , uint32_t ManagementThreadCpuCoreOsId /*!< [in] CPU core on wich the management thread will be pinned. */
                               , HANDLE* pStream /*!< [out] Handle of the created stream */
);

/*!
   @brief This function manages stream creation and configuration based on a SDP

   @returns The function returns the status of its execution as VMIP_ERRORCODE
*/
VMIP_ERRORCODE ConfigureStreamFromSdp(HANDLE VcsContext /*!< [in] Context of the VCS session */
                                      , VMIP_STREAMTYPE StreamType /*!< [in] Type of the stream (RX or TX)*/
                                      , const std::vector<uint32_t>& rProcessingCpuCoreOsId /*!< [in]  This will be used to know on which core processing job will be launched.
                                                                                                        Putting the conductor core in this list should be avoided.
                                                                                                        If left empty, all the core of the system will be used. */
                                      , std::string Sdp /*!< [in] SDP from which informations will be extracted*/
                                      , const std::vector<uint64_t>& rNicIds /*!< [in] Ids of the NICs used by the stream. Multiple NICs are used in case we use multiple paths as specified in the norm ST2022-7 */
                                      , uint64_t ConductorId /*!< [in] Id of the conductor used by the stream. */
                                      , uint32_t ManagementThreadCpuCoreOsId /*!< [in] CPU core on wich the management thread will be pinned. */
                                      , HANDLE* pStream /*!< [out] Handle of the created stream */
);


/*!
   @brief Creates a colorbar pattern that supports only yuv 4:2:2 10bits specific ST2110 format.
*/
void CreateColorBarPatternYUV422_10Bit_ST2110(uint8_t* pUserBuffer /*!< [in] pointer to the user-instantiated buffer that will be filled.*/
   , uint32_t FrameHeight /*!< [in] Frame height. */
   , uint32_t FrameWidth /*!< [in] Frame width.*/
);
/*!
   @brief Draw an horizontal white line in a buffer. Only available for a yuv 4:2:2 10bits specific ST2110 buffer.
*/
void DrawWhiteLineYUV422_10Bit_ST2110(uint8_t* pBuffer /*!< [in] Buffer in which the white line will be drawn*/
   , uint32_t Line /*!< [in] Line position (in pixel)*/
   , uint32_t FrameHeight /*!< [in] Frame height */
   , uint32_t FrameWidth /*!< [in] Frame width*/
   , bool8_t Interlaced /*!< [in] Is the frame interlaced or not.*/
);

void CreateAudio1Khz(uint8_t* pUserBuffer, uint32_t BufferSize, uint32_t NbBytesFormat, uint32_t SamplingRate, uint32_t mNbChannel, uint32_t* pAudioSampleCount);

/*!
   @brief Print the status of the PTP service

   @returns The function returns the status of its execution as VMIP_ERRORCODE
*/
VMIP_ERRORCODE PrintPtpStatus(HANDLE VcsContext /*!< [in] Handle to the VCS context.*/);

/*!
   @brief Stringify the VCS version

   @returns A string containing the formated VCS version as M.m.b, where M is the MAJOR version, m is the MINOR version and b is the BUILD version.
*/
std::string VersionToString(uint64_t Version /*!< [in] Version that will be stringified*/);

/*!
   @brief Convert a sampling rate enum representative value to the associated uint32_t

   @returns A uint32_t representing the sampling rate (e.g VMIP_AUDIO_SAMPLING_RATE_48KHZ -> 48000)
*/
uint32_t VmipSamplingRateToUint32(VMIP_AUDIO_SAMPLING_RATE SamplingRate /*!< [in] Sampling rate*/);

/*!
   @brief Convert an audio format enumeration representative value to the associated number of bytes

   @returns A uint32_t representing the number of bytes associated to an audio format (e.g VMIP_AUDIO_FORMAT_L16 -> 2)
*/
uint32_t VmipFormatToNbBytes(VMIP_AUDIO_FORMAT Format /*!< [in] Audio format*/);

/*!
   @brief This function generates an SDP based on a stream configuration.

   @returns The function returns the status of its execution as VMIP_ERRORCODE
*/
VMIP_ERRORCODE GenerateSdp(HANDLE VcsContext,   /*!< [in] VCS context used by the stream */
                           HANDLE Stream,       /*!< [in] Stream used for the SDP generation */
                           std::string* pSdp    /*!< [out] SDP generated*/
);


/*!
   @brief This function monitor RX stream status
*/
void MonitorRXStreamStatus(HANDLE Stream, bool* RequestStop);

/*!
   @brief This function monitor TX stream status
*/
void MonitorTXStreamStatus(HANDLE Stream, bool* RequestStop);

/*!
   @brief This function print error message with the last error message
*/
void PrintWithLastError(std::string Message);

/*!
   @brief This function returns the default buffer type for a given essence type
*/
uint32_t GetEssenceDefaultBufferType(VMIP_ESSENCETYPE EssenceType);
