#define _USE_MATH_DEFINES
#include <cmath>

#include "tools.h"

#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_networkinterface.h>
#include <videomasterip/videomasterip_cpu.h>
#include <videomasterip/videomasterip_sdp.h>

#include <iostream>
#include <iomanip>
#include <cstring>
#include <vector>
#include <array>
#include <thread>

#define PRINT_TAB_SPACE 10
#define AUDIO_AMPL 3276

VMIP_ERRORCODE PrintCpuCoresInfo(HANDLE VcsContext)
{
   uint64_t* CPUCoresIdList = new uint64_t[MAX_HANDLE_ARRAY_SIZE];
   uint32_t NbCpuCore = 0;
   VMIP_CPUCORE_STATUS CpuStatus;

   VMIP_ERRORCODE Result = VMIP_GetCPUCoresHandlesList(VcsContext, CPUCoresIdList, &NbCpuCore);
   if (Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << NbCpuCore << "  CPU cores detected." << std::endl;
      std::cout << std::setfill(' ')
         << std::setw(PRINT_TAB_SPACE) << "Index"
         << std::setw(PRINT_TAB_SPACE) << "OsId"
         << std::setw(PRINT_TAB_SPACE) << "Usage (%)"
         << std::setw(2 * PRINT_TAB_SPACE) << "MaxSpeed (MHz)"
         << std::setw(PRINT_TAB_SPACE * 2) << "Conductor Avail."
         << std::setw(PRINT_TAB_SPACE) << "Numa" << std::endl;

      for (uint32_t i = 0; i < NbCpuCore; i++)
      {
         Result = VMIP_GetCpuCoreStatus(VcsContext, CPUCoresIdList[i], &CpuStatus);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error while getting the status of CPUCore " + std::to_string(i) + ".[" + std::to_string(Result) + "] ");
            break;
         }
         else
         {
            std::cout << std::setfill(' ')
               << std::setw(PRINT_TAB_SPACE) << i
               << std::setw(PRINT_TAB_SPACE) << CpuStatus.CoreId
               << std::setw(PRINT_TAB_SPACE) << uint32_t(CpuStatus.CoreUsage)
               << std::setw(2 * PRINT_TAB_SPACE) << CpuStatus.MaxSpeed
               << std::setw(PRINT_TAB_SPACE * 2) << ToString(CpuStatus.ConductorAvailability)
               << std::setw(PRINT_TAB_SPACE) << CpuStatus.NumaNode << std::endl;
         }
      }
   }
   else
      PrintWithLastError("Error while getting the number of CPUCore [" + std::to_string(Result) + "]");

   delete[] CPUCoresIdList;

   return Result;
}

VMIP_ERRORCODE PrintNicsInfo(HANDLE VcsContext)
{
   uint64_t* NetworkInterfaceIdList = new uint64_t[MAX_HANDLE_ARRAY_SIZE];
   uint32_t NbNetworkInterface = 0;
   VMIP_NETWORK_ITF_STATUS NetworkItfStatus;

   std::string NicName = "";
   std::string DriverName = "";

   VMIP_ERRORCODE Result = VMIP_GetNetworkInterfaceHandlesList(VcsContext, NetworkInterfaceIdList, &NbNetworkInterface);
   if (Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << NbNetworkInterface << " network interfaces detected." << std::endl;
      std::cout << std::setfill(' ') << std::setw(PRINT_TAB_SPACE) << "Index"
         << std::setfill(' ') << std::setw(2 * PRINT_TAB_SPACE) << "Name"
         << std::setfill(' ') << std::setw(PRINT_TAB_SPACE) << "Numa"
         << std::setfill(' ') << std::setw(PRINT_TAB_SPACE) << "Duplex"
         << std::setfill(' ') << std::setw(2 * PRINT_TAB_SPACE) << "Link Status"
         << std::setfill(' ') << std::setw(2 * PRINT_TAB_SPACE) << "Speed (Mbits/sec)"
         << std::setfill(' ') << std::setw(2 * PRINT_TAB_SPACE) << "MAC"
         << std::setfill(' ') << std::setw(PRINT_TAB_SPACE) << "OsId"
         << std::setfill(' ') << std::setw(2 * PRINT_TAB_SPACE) << "Driver" << std::endl;

      for (uint32_t i = 0; i < NbNetworkInterface; i++)
      {
         Result = VMIP_GetNetworkInterfaceStatus(VcsContext, NetworkInterfaceIdList[i], &NetworkItfStatus);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error while getting the status of Interface " + std::to_string(i) + ".[" + std::to_string(Result) + "] ");
            break;
         }
         else
         {
            NicName = std::string(NetworkItfStatus.pName);
            DriverName = std::string(NetworkItfStatus.pDriverVersion);
            std::cout
               << std::setfill(' ') << std::setw(PRINT_TAB_SPACE) << i
               << std::setfill(' ') << std::setw(2 * PRINT_TAB_SPACE) << NicName
               << std::setfill(' ') << std::setw(PRINT_TAB_SPACE) << NetworkItfStatus.NumaNode
               << std::setfill(' ') << std::setw(PRINT_TAB_SPACE) << ToString(NetworkItfStatus.DuplexMode)
               << std::setfill(' ') << std::setw(2 * PRINT_TAB_SPACE) << uint32_t(NetworkItfStatus.LinkStatus)
               << std::setfill(' ') << std::setw(2 * PRINT_TAB_SPACE) << NetworkItfStatus.LinkSpeed
               << "        " << std::setfill('0') << std::setw(12) << std::hex << NetworkItfStatus.MacAddress << std::dec
               << std::setfill(' ') << std::setw(PRINT_TAB_SPACE) << NetworkItfStatus.OsId
               << std::setfill(' ') << std::setw(2 * PRINT_TAB_SPACE) << DriverName << std::endl;
         }
      }
   }
   else
      PrintWithLastError("Error while getting the number of NetInterfaces [" + std::to_string(Result) + "]");

   delete[] NetworkInterfaceIdList;

   return Result;
}

VMIP_ERRORCODE GetNicIdFromName(HANDLE VcsContext, std::string NicName, uint64_t* pNicId)
{
   uint64_t* NetworkInterfaceIdList = new uint64_t[MAX_HANDLE_ARRAY_SIZE];
   uint32_t NbNetworkInterface = 0;
   bool NicFound = false;
   VMIP_NETWORK_ITF_STATUS NetworkItfStatus;

   if (pNicId == nullptr)
   {
      std::cout << std::endl << "The  pNicId pointer is invalid" << std::endl;
      return VMIPERR_INVALIDPOINTER;
   }

   VMIP_ERRORCODE Result = VMIP_GetNetworkInterfaceHandlesList(VcsContext, NetworkInterfaceIdList, &NbNetworkInterface);
   if (Result == VMIPERR_NOERROR)
   {
      for (uint32_t i = 0; i < NbNetworkInterface; i++)
      {
         Result = VMIP_GetNetworkInterfaceStatus(VcsContext, NetworkInterfaceIdList[i], &NetworkItfStatus);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error while getting the status of Interface " + std::to_string(i) + ".[" + std::to_string(Result) + "] ");
            break;
         }
         else
         {
            if (strcmp(NetworkItfStatus.pName, NicName.c_str()) == 0)
            {
               NicFound = true;
               *pNicId = NetworkInterfaceIdList[i];
               break;
            }
         }
      }
      if (!NicFound)
      {
         std::cout << std::endl << " No NIC with the name " << NicName << " was found." << std::endl;
         Result = VMIPERR_NOTFOUND;
      }
   }
   else
      PrintWithLastError("Error while getting the number of NetInterfaces [" + std::to_string(Result) + "]");

   delete[] NetworkInterfaceIdList;

   return Result;
}

std::string ToString(VMIP_NETWORK_ITF_DUPLEX_MODE DuplexMode)
{
   std::string Return = "Undefined";

   switch (DuplexMode)
   {
   case VMIP_NETWORK_ITF_DUPLEX_MODE_HALF: Return = "Half";
      break;
   case VMIP_NETWORK_ITF_DUPLEX_MODE_FULL: Return = "Full";
      break;
   case VMIP_NETWORK_ITF_DUPLEX_MODE_UNKNOWN: Return = "Unknown";
      break;
   case NB_NETWORK_ITF_DUPLEX_MODE:
   default: break;
   }

   return Return;
}

std::string ToString(VMIP_PTP_STATE PtpState)
{
   std::string Return = "Undefined";

   switch (PtpState)
   {
   case VMIP_PTP_STATE_INITIZALIZING: Return = "Initializing"; break;
   case VMIP_PTP_STATE_FAULTY: Return = "Faulty"; break;
   case VMIP_PTP_STATE_DISABLED: Return = "Disabled"; break;
   case VMIP_PTP_STATE_LISTENING: Return = "Listening"; break;
   case VMIP_PTP_STATE_PREMASTER: Return = "Pre-Master"; break;
   case VMIP_PTP_STATE_MASTER: Return = "Master"; break;
   case VMIP_PTP_STATE_PASSIVE: Return = "Passive"; break;
   case VMIP_PTP_STATE_UNCALIBRATED: Return = "Uncalibrated"; break;
   case VMIP_PTP_STATE_SLAVE: Return = "Slave"; break;
   case NB_VMIP_PTP_STATE:
   default: break;
   }

   return Return;
}

std::string ToString(VMIP_CONDUCTOR_AVAILABILITY Value)
{
   std::string Return = "Undefined";

   switch (Value)
   {
   case VMIP_CONDUCTOR_AVAILABLE: Return = "Available"; break;
   case VMIP_CONDUCTOR_NOT_AVAILABLE: Return = "NotAvail."; break;
   case VMIP_CONDUCTOR_ALREADY_USED: Return = "Alr Used"; break;
   case NB_VMIP_CONDUCTOR_AVAILABILITY: break;
   default:;
   }

   return Return;
}

VMIP_ERRORCODE ConfigureConductor(const uint32_t ConductorCpuCoreOsId, HANDLE VcsContext,
   uint64_t* pConductorId)
{
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;

   if (pConductorId == nullptr)
   {
      std::cout << std::endl << "The  pConductorId pointer is invalid" << std::endl;
      return VMIPERR_INVALIDPOINTER;
   }

   //We create the conductor. The conductor will be responsible of sending the packet at the correct rate to be compliant with the ST2110-20 norm.
   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_CreateConductor(VcsContext, true, pConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when creating conductor");
   }

   //We associate the conductor to a Cpu Core by setting its configuration
   if (Result == VMIPERR_NOERROR)
   {
      VMIP_CONDUCTOR_CONFIG ConductorConfig;

      //We get the cpu Core Id from the Os ID
      Result = VMIP_GetCPUCoreFromOSId(VcsContext, ConductorCpuCoreOsId, &ConductorConfig.CpuCoreId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the CpuId for Os Id = " + std::to_string(ConductorCpuCoreOsId));
      else
      {
         Result = VMIP_SetConductorConfig(VcsContext, *pConductorId, ConductorConfig);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when setting conductor config");
      }
   }

   return Result;
}

VMIP_ERRORCODE ConfigureStream(HANDLE VcsContext, VMIP_STREAMTYPE StreamType,
                               const std::vector<uint32_t>& rProcessingCpuCoreOsId,
                               const std::vector<uint32_t>& rDestinationAddresses,
                               const std::vector<uint16_t>& rDestinationUdpPorts,
                               const uint32_t DestinationSsrc,
                               const ESSENCE_PARAM EssenceParam, 
                               const std::vector<uint64_t>& rNicIds,
                               uint64_t ConductorId, uint32_t ManagementThreadCpuCoreOsId, HANDLE* pStream)
{
   VMIP_STREAM_COMMON_CONFIG StreamCommonConfig;
   VMIP_STREAM_NETWORK_CONFIG StreamNetworkConfig;
   VMIP_STREAM_ESSENCE_CONFIG EssenceConfig;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;

   if (pStream == nullptr)
   {
      std::cout << std::endl << "The pStream pointer is invalid" << std::endl;
      return VMIPERR_INVALIDPOINTER;
   }

   if(rDestinationAddresses.size() != rDestinationUdpPorts.size() || rDestinationAddresses.size() != rNicIds.size())
   {
      std::cout << std::endl << "The sizes of rDestinationAddresses and rDestinationUdpPorts do not correspond" << std::endl;
      return VMIPERR_BAD_CONFIGURATION;
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_CreateStream(VcsContext, StreamType, EssenceParam.EssenceType, pStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when creating stream");
   }

   //We get the stream common config. It is also a way to initialize the structure StreamCommonConfig.
   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetStreamCommonConfig(*pStream, &StreamCommonConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting stream common config");
   }

   // We get the CPU id for the management thread CPU core.
   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetCPUCoreFromOSId(VcsContext, ManagementThreadCpuCoreOsId, &StreamCommonConfig.ManagementThreadCpuCoreId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the CPU id for OS id " + std::to_string(ManagementThreadCpuCoreOsId));
   }

   //We set the stream common config.
   if (Result == VMIPERR_NOERROR)
   {
      //We associate a stream to a conductor.
      StreamCommonConfig.ConductorId = ConductorId;

      //We associate the stream to an array of Cpu core. The processing tasks will run on those Cpu cores.
      uint64_t TmpCpuId = 0;
      StreamCommonConfig.NbCpuCoreForProcessing = 0;

      for (auto CpuOsId : rProcessingCpuCoreOsId)
      {
         if (VMIP_GetCPUCoreFromOSId(VcsContext, CpuOsId, &TmpCpuId) == VMIPERR_NOERROR)
         {
            StreamCommonConfig.pCpuCoreIdxForProcessing[StreamCommonConfig.NbCpuCoreForProcessing] = TmpCpuId;
            StreamCommonConfig.NbCpuCoreForProcessing++;
         }
      }

      Result = VMIP_SetStreamCommonConfig(*pStream, StreamCommonConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when setting stream common config");
   }

   //We get the stream network config. It is also a way to initialize the structure StreamNetworkConfig.
   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetStreamNetworkConfig(*pStream, &StreamNetworkConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting stream network config");
   }

   //Wet set the network configuration.
   if (Result == VMIPERR_NOERROR)
   {
      StreamNetworkConfig.PathParameterSize = rDestinationAddresses.size();
      for(uint32_t i = 0; i < rDestinationAddresses.size(); i++)
      {
         StreamNetworkConfig.pPathParameters[i].DestinationIp = rDestinationAddresses[i];
         StreamNetworkConfig.pPathParameters[i].UdpPort = rDestinationUdpPorts[i];
         //We associate the stream to one or multiples NICs. The packets will be issued on this/those NICs.
         StreamNetworkConfig.pPathParameters[i].InterfaceId = rNicIds[i];
      }
      switch (EssenceParam.EssenceType)
      {
      case VMIP_ET_ST2110_20:
         StreamNetworkConfig.PayloadType = 96;
         break;
      case VMIP_ET_ST2110_22:
         StreamNetworkConfig.PayloadType = 112;
         break;
      case VMIP_ET_ST2110_30:
         StreamNetworkConfig.PayloadType = 97;
         break;
      case VMIP_ET_ST2110_40:
         StreamNetworkConfig.PayloadType = 98;
         break;
      default:
         StreamNetworkConfig.PayloadType = 96;
      }
      StreamNetworkConfig.Ssrc = DestinationSsrc;

      Result = VMIP_SetStreamNetworkConfig(*pStream, StreamNetworkConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when setting stream network config");
   }

   //We get the stream essence config. It is also a way to initialize the structure EssenceConfig.
   if (Result == VMIPERR_NOERROR)
   {
      EssenceConfig.EssenceType = EssenceParam.EssenceType;
      Result = VMIP_GetStreamEssenceConfig(*pStream, &EssenceConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting stream essence config");
   }

   //We set the stream essence config.
   if (Result == VMIPERR_NOERROR)
   {
      switch (EssenceParam.EssenceType)
      {
      case VMIP_ET_ST2110_20:
         EssenceConfig.EssenceType = EssenceParam.EssenceType;
         EssenceConfig.EssenceS2110_20Prop.NetworkBitDepth = VMIP_VIDEO_DEPTH_10BIT;
         EssenceConfig.EssenceS2110_20Prop.NetworkBitSampling = VMIP_VIDEO_SAMPLING_YCBCR_422;
         EssenceConfig.EssenceS2110_20Prop.UserBitDepth = VMIP_VIDEO_DEPTH_10BIT;
         EssenceConfig.EssenceS2110_20Prop.UserBitSampling = VMIP_VIDEO_SAMPLING_YCBCR_422;
         EssenceConfig.EssenceS2110_20Prop.UserBitPadding = VMIP_VIDEO_NO_PADDING;
         EssenceConfig.EssenceS2110_20Prop.UserEndianness = VMIP_VIDEO_BIG_ENDIAN;
         EssenceConfig.EssenceS2110_20Prop.UseDefaultTrOffset = true;
         EssenceConfig.EssenceS2110_20Prop.VideoStandard = EssenceParam.VideoStandard;
         break;
      case VMIP_ET_ST2110_22:
         EssenceConfig.EssenceType = EssenceParam.EssenceType;
         EssenceConfig.EssenceS2110_22Prop.BitDepth = VMIP_VIDEO_DEPTH_10BIT;
         EssenceConfig.EssenceS2110_22Prop.BitSampling = VMIP_VIDEO_SAMPLING_YCBCR_422;
         EssenceConfig.EssenceS2110_22Prop.VideoStandard = EssenceParam.Param_ST2110_20.VideoStandard;
         EssenceConfig.EssenceS2110_22Prop.FrameSize = EssenceParam.Param_ST2110_20.FrameSize;
         EssenceConfig.EssenceS2110_22Prop.Codec = EssenceParam.Param_ST2110_20.Codec;
         break;
      case VMIP_ET_ST2110_30:
         EssenceConfig.EssenceType = EssenceParam.EssenceType;
         EssenceConfig.EssenceS2110_30Prop.Format = VMIP_AUDIO_FORMAT_L24;
         EssenceConfig.EssenceS2110_30Prop.NbChannel = EssenceParam.AudioParam.NbChannel;
         EssenceConfig.EssenceS2110_30Prop.UseFrameBasedMode = false;
         EssenceConfig.EssenceS2110_30Prop.PacketNbPerSlot = EssenceParam.AudioParam.PacketNbPerSlot;
         EssenceConfig.EssenceS2110_30Prop.PacketTime = EssenceParam.AudioParam.PacketTime;
         EssenceConfig.EssenceS2110_30Prop.SamplingRate = EssenceParam.AudioParam.SamplingRate;
         break;
      case VMIP_ET_ST2110_40:
         EssenceConfig.EssenceType = EssenceParam.EssenceType;
         EssenceConfig.EssenceS2110_40Prop.SdiVideoStandard = EssenceParam.AncVideoStandard;
         break;
      default:
         Result = VMIPERR_BADARG;
         break;
      }

      if (Result == VMIPERR_NOERROR)
      {
         Result = VMIP_SetStreamEssenceConfig(*pStream, EssenceConfig);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when setting stream essence config");
      }
      else
         std::cout << "Error in Essence type" << std::endl;
   }

   return Result;
}

VMIP_ERRORCODE ConfigureStreamFromSdp(HANDLE VcsContext, VMIP_STREAMTYPE StreamType,
                                      const std::vector<uint32_t>& rProcessingCpuCoreOsId, std::string Sdp, 
                                      const std::vector<uint64_t>& rNicIds, uint64_t ConductorId,
                                      uint32_t ManagementThreadCpuCoreOsId, HANDLE* pStream)
{
   VMIP_STREAM_COMMON_CONFIG StreamCommonConfig;
   VMIP_STREAM_NETWORK_CONFIG StreamNetworkConfig;
   VMIP_STREAM_ESSENCE_CONFIG EssenceConfig;
   VMIP_SDP_ADDITIONAL_INFORMATION SdpAdditionalInformation;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;

   if (pStream == nullptr)
   {
      std::cout << std::endl << "The  pStream pointer is invalid" << std::endl;
      return VMIPERR_INVALIDPOINTER;
   }

   //Fill the structure with the information contained in the SDP
   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_ReadSDP(Sdp.c_str(), Sdp.size(), &StreamNetworkConfig, &EssenceConfig, &SdpAdditionalInformation);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when reading the SDP");
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_CreateStream(VcsContext, StreamType, EssenceConfig.EssenceType, pStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when creating stream");
   }

   //We get the stream common config. It is also a way to initialize the structure StreamCommonConfig.
   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetStreamCommonConfig(*pStream, &StreamCommonConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting stream common config");
   }

   // We get the CPU id for the management thread CPU core.
   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetCPUCoreFromOSId(VcsContext, ManagementThreadCpuCoreOsId, &StreamCommonConfig.ManagementThreadCpuCoreId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the CPU id for OS id " + std::to_string(ManagementThreadCpuCoreOsId));
   }

   //We set the stream common config.
   if (Result == VMIPERR_NOERROR)
   {
      //We fill parameters that are not present in the SDP.

      //We associate a stream to a conductor.
      StreamCommonConfig.ConductorId = ConductorId;

      //We associate the stream to an array of Cpu core. The processing tasks will run on those Cpu cores.
      uint64_t TmpCpuId = 0;
      StreamCommonConfig.NbCpuCoreForProcessing = 0;
      for (auto CpuOsId : rProcessingCpuCoreOsId)
      {
         if (VMIP_GetCPUCoreFromOSId(VcsContext, CpuOsId, &TmpCpuId) == VMIPERR_NOERROR)
         {
            StreamCommonConfig.pCpuCoreIdxForProcessing[StreamCommonConfig.NbCpuCoreForProcessing] = TmpCpuId;
            StreamCommonConfig.NbCpuCoreForProcessing++;
         }
      }

      Result = VMIP_SetStreamCommonConfig(*pStream, StreamCommonConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when setting stream common config");
   }

   if(rNicIds.size() != StreamNetworkConfig.PathParameterSize)
   {
      std::cout << std::endl << "The number of path configured in the SDP and the number of Nic do not correspond" << std::endl;
      Result = VMIPERR_BAD_CONFIGURATION;
   }

   //Wet set the network configuration.
   if (Result == VMIPERR_NOERROR)
   {
      //We fill parameters that are not present in the SDP.
      StreamNetworkConfig.Ssrc = 0;
      for (uint32_t PathIndex = 0; PathIndex < StreamNetworkConfig.PathParameterSize; PathIndex++)
      {
         StreamNetworkConfig.pPathParameters[PathIndex].InterfaceId = rNicIds[PathIndex];
         StreamNetworkConfig.pPathParameters[PathIndex].IgmpFilteringType = VMIP_FILTERING_BLACKLIST;
         StreamNetworkConfig.pPathParameters[PathIndex].IgmpSourceListSize = 0;
      }

      Result = VMIP_SetStreamNetworkConfig(*pStream, StreamNetworkConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when setting stream network config");
   }

   //We set the stream essence config.
   if (Result == VMIPERR_NOERROR)
   {
      switch (EssenceConfig.EssenceType)
      {
      case VMIP_ET_ST2110_20:
         EssenceConfig.EssenceS2110_20Prop.UserBitDepth = EssenceConfig.EssenceS2110_20Prop.NetworkBitDepth;
         EssenceConfig.EssenceS2110_20Prop.UserBitSampling = EssenceConfig.EssenceS2110_20Prop.NetworkBitSampling;
         EssenceConfig.EssenceS2110_20Prop.UserBitPadding = VMIP_VIDEO_NO_PADDING;
         EssenceConfig.EssenceS2110_20Prop.UserEndianness = VMIP_VIDEO_BIG_ENDIAN;
         EssenceConfig.EssenceS2110_20Prop.UseDefaultTrOffset = true;
         EssenceConfig.EssenceS2110_20Prop.UseDefaultVrxCapacity = true;
         break;
      case VMIP_ET_ST2110_22:
         //Nothing needs to be added to what is present in the SDP
         break;
      case VMIP_ET_ST2110_30:
         EssenceConfig.EssenceS2110_30Prop.UseFrameBasedMode = false;
         EssenceConfig.EssenceS2110_30Prop.PacketNbPerSlot = 40;
         break;
      case VMIP_ET_ST2110_40:
         EssenceConfig.EssenceS2110_40Prop.SdiVideoStandard = VMIP_ANC_SDI_VIDEOSTD_1920X1080P30;
         break;
      case NB_VMIP_ESSENCETYPE:
      default:
         Result = VMIPERR_BADARG;
         break;
      }

      if (Result == VMIPERR_NOERROR)
      {
         Result = VMIP_SetStreamEssenceConfig(*pStream, EssenceConfig);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when setting stream essence config");
      }
      else
         std::cout << "Error in Essence type" << std::endl;
   }

   return Result;
}

void CreateColorBarPatternYUV422_10Bit_ST2110(uint8_t* pUserBuffer, uint32_t FrameHeight, uint32_t FrameWidth)
{
   const YUV_10BIT White75 = {0x2D0,0x200,0x200};
   const YUV_10BIT Yellow75 = {0x2A0,0xB0,0x220};
   const YUV_10BIT Cyan75 = {0x244,0x24c,0xB0};
   const YUV_10BIT Green75 = {0x214,0xfc,0xD0};
   const YUV_10BIT Magenta75 = {0xfc,0x304,0x330};
   const YUV_10BIT Red75 = {0xcc,0x184,0x350};
   const YUV_10BIT Blue75 = {0x70,0x350,0x1e0};
   const YUV_10BIT Black75 = {0x40,0x200,0x200};
   std::vector<YUV_10BIT> ColorList = {White75,Yellow75,Cyan75,Green75,Magenta75,Red75,Blue75,Black75};

   uint32_t ColorId=0;
   int j=0;
   
   if (pUserBuffer == nullptr)
   {
      std::cout << std::endl << "The  pUserBuffer pointer is invalid" << std::endl;
      return;
   }

   for(uint32_t Pixely = 0; Pixely < FrameHeight; Pixely++)
   {
      for (uint32_t Pixelx = 0; Pixelx < FrameWidth; Pixelx+=8)
      {
         ColorId = Pixelx/(FrameWidth/static_cast<uint32_t>(ColorList.size()));
         for(int i=0; i<4; i++)
         {
            pUserBuffer[(j*20)+(i*5)+0] = (ColorList[ColorId].u >> 2);
            pUserBuffer[(j*20)+(i*5)+1] = ((ColorList[ColorId].u & 0x3)<<6) + ((ColorList[ColorId].y & 0x3f0)>>4);
            pUserBuffer[(j*20)+(i*5)+2] = ((ColorList[ColorId].v & 0x3c0)>>6) + ((ColorList[ColorId].y & 0xf)<<4);
            pUserBuffer[(j*20)+(i*5)+3] = ((ColorList[ColorId].v & 0xcf)<<2) + ((ColorList[ColorId].y & 0x300)>>8);
            pUserBuffer[(j*20)+(i*5)+4] = (ColorList[ColorId].y & 0xff);
         }
         j++;
      }
   }
}

void DrawWhiteLineYUV422_10Bit_ST2110(uint8_t* pBuffer, uint32_t Line, uint32_t FrameHeight, uint32_t FrameWidth,
   bool8_t Interlaced_b)
{
   uint8_t* pTemp;
   int j=0;
   const YUV_10BIT White100 = {0x3AC,0x200,0x200};

   if (pBuffer == nullptr)
   {
      std::cout << std::endl << "The  pBuffer pointer is invalid" << std::endl;
      return;
   }

   if (Interlaced_b)
   {
      if (Line % 2 == 0)
         pTemp = (pBuffer + (Line / 2) * FrameWidth * PIXELSIZE_10BIT);
      else
         pTemp = (pBuffer + (((FrameHeight + 1) / 2) + (Line / 2)) * FrameWidth * PIXELSIZE_10BIT);
   }
   else
      pTemp = (pBuffer + Line * FrameWidth * PIXELSIZE_10BIT);

   for (uint32_t Pixelx = 0; Pixelx < FrameWidth; Pixelx+=8)
   {
      for(int i=0; i<4; i++)
      {
         pTemp[(j*20)+(i*5)+0] = (White100.u >> 2);
         pTemp[(j*20)+(i*5)+1] = ((White100.u & 0x3)<<6) + ((White100.y & 0x3f0)>>4);
         pTemp[(j*20)+(i*5)+2] = ((White100.v & 0x3c0)>>6) + ((White100.y & 0xf)<<4);
         pTemp[(j*20)+(i*5)+3] = ((White100.v & 0xcf)<<2) + ((White100.y & 0x300)>>8);
         pTemp[(j*20)+(i*5)+4] = (White100.y & 0xff);
      }
      j++;
   }
}

void CreateAudio1Khz(uint8_t* pUserBuffer, uint32_t BufferSize, uint32_t NbBytesFormat, uint32_t SamplingRate, uint32_t mNbChannel, uint32_t* pAudioSampleCount)
{
   uint32_t NbOfAudioSamples = BufferSize / (NbBytesFormat * mNbChannel);
   uint16_t Temp = 0;
   for (uint32_t i = 0; i < NbOfAudioSamples; i++)
   {
      Temp = (uint16_t)(AUDIO_AMPL * sin(static_cast<double>(2 * M_PI * 1000 * (*pAudioSampleCount)++) / SamplingRate));
      for (uint32_t j = 0; j < mNbChannel; j++)
      {
         pUserBuffer[i * (mNbChannel * NbBytesFormat) + (j * NbBytesFormat)] = 0;
         pUserBuffer[i * (mNbChannel * NbBytesFormat) + (j * NbBytesFormat) + 1] = (uint8_t)(Temp & 0xff);
         pUserBuffer[i * (mNbChannel * NbBytesFormat) + (j * NbBytesFormat) + 2] = (uint8_t)((Temp & 0xff00) >> 8);
      }
   }
}

VMIP_ERRORCODE PrintPtpStatus(HANDLE VcsContext)
{
   VMIP_ERRORCODE Result;
   VMIP_PTP_STATUS PtpStatus = {};

   Result = VMIP_GetPTPStatus(VcsContext, &PtpStatus);
   if (Result != VMIPERR_NOERROR)
      PrintWithLastError("Error when getting the PTP status");
   else
   {
      std::cout << "State = " << ToString(PtpStatus.PortDS.PortState) << " (Offset : " << PtpStatus.CurrentDS.OffsetFromMaster
         << " seconds)        \r" << std::flush;
   }
   return Result;
}

std::string VersionToString(uint64_t Version /*!< [in] Version that will be stringified*/)
{
   std::string MajorVersion_str = std::to_string((Version & 0xffff000000000000) >> 48);
   std::string MinorVesion_str = std::to_string((Version & 0xffff00000000) >> 32);
   std::string BuildVesion_str = std::to_string((Version & 0xfffffff0) >> 4);
   std::string CustomVersion_str = std::to_string(Version & 0xf);

   return MajorVersion_str + "." + MinorVesion_str + "." + BuildVesion_str + "." + CustomVersion_str;
}

uint32_t VmipSamplingRateToUint32(VMIP_AUDIO_SAMPLING_RATE SamplingRate)
{
   switch (SamplingRate)
   {
   case VMIP_AUDIO_SAMPLING_RATE_48KHZ:
      return 48000;
   case VMIP_AUDIO_SAMPLING_RATE_96KHZ:
      return 96000;
   default:
      return 0;
   }
}

uint32_t VmipFormatToNbBytes(VMIP_AUDIO_FORMAT Format)
{
   switch (Format)
   {
   case VMIP_AUDIO_FORMAT_L16:
      return 2;
   case VMIP_AUDIO_FORMAT_L24:
      return 3;
   default:
      return 0;
   }
}

VMIP_ERRORCODE GenerateSdp(HANDLE VcsContext, HANDLE Stream, std::string* pSdp)
{
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;

   VMIP_STREAM_NETWORK_CONFIG StreamNetworkConfig;
   VMIP_STREAM_ESSENCE_CONFIG EssenceConfig;
   VMIP_SDP_ADDITIONAL_INFORMATION SdpAdditionalInformation;

   if (pSdp == nullptr)
   {
      std::cout << std::endl << "The  pSdp pointer is invalid" << std::endl;
      return VMIPERR_INVALIDPOINTER;
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetStreamNetworkConfig(Stream, &StreamNetworkConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the stream network config");
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetStreamEssenceConfig(Stream, &EssenceConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the stream essence config");
   }

   if (Result == VMIPERR_NOERROR)
   {
      VMIP_NETWORK_ITF_CONFIG NicConfig;
      Result = VMIP_GetNetworkInterfaceConfig(VcsContext,StreamNetworkConfig.pPathParameters[0].InterfaceId, &NicConfig);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the network interface config");
      else
         StreamNetworkConfig.pPathParameters[0].SourceIp = NicConfig.InterfaceIpAddress;
   }

   if (Result == VMIPERR_NOERROR)
   {
      VMIP_NETWORK_ITF_STATUS NicStatus;
      Result = VMIP_GetNetworkInterfaceStatus(VcsContext,StreamNetworkConfig.pPathParameters[0].InterfaceId, &NicStatus);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the network interface status");
      else
         SdpAdditionalInformation.TrafficShapingProfile = NicStatus.TrafficShapingProfile;
   }

   if (Result == VMIPERR_NOERROR)
   {
      SdpAdditionalInformation.MediaClockType = VMIP_MEDIA_CLOCK_DIRECT;
      SdpAdditionalInformation.TimestampMode = VMIP_TS_MODE_NEW;
      SdpAdditionalInformation.ReferenceClock.ReferenceClockType = VMIP_REF_CLOCK_PTP_GRANDMASTER_ID;
      SdpAdditionalInformation.ReferenceClock.Grandmaster.Domain = 127;
      const std::array<uint8_t, 8> MasterId = { 0xd0, 0xcb, 0x07, 0xfe, 0xff, 0x94, 0xa7, 0x39 };
      memcpy(SdpAdditionalInformation.ReferenceClock.Grandmaster.pId, MasterId.data(), MasterId.size());
      SdpAdditionalInformation.HasTimestampDelay = true;
      SdpAdditionalInformation.TimestampDelay = 1000;
      snprintf(SdpAdditionalInformation.pSessionName, MAX_CHAR_ARRAY_SIZE, "Sample Media Stream");
      snprintf(SdpAdditionalInformation.pSessionDescription, MAX_CHAR_ARRAY_SIZE, "media stream created with Deltacast IPVirtualCard");
      if (EssenceConfig.EssenceType == VMIP_ET_ST2110_30)
         snprintf(SdpAdditionalInformation.pAudioChannelOrder, MAX_CHAR_AUDIO_CHANNEL_ORDER_SIZE, "SMPTE2110.(M,M,M,M,ST,U02)");

      const uint32_t MaxSdpSize = 65536;
      uint32_t SdpSize = MaxSdpSize;
      char pSdp_char[MaxSdpSize];
      Result = VMIP_WriteSDP(StreamNetworkConfig, EssenceConfig, SdpAdditionalInformation, pSdp_char, &SdpSize);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when writing the SDP");
      else
         *pSdp = std::string(pSdp_char);
   }

   return Result;
}

using namespace std::chrono_literals;
void MonitorRXStreamStatus(HANDLE Stream, bool* RequestStop)
{
   VMIP_STREAM_COMMON_STATUS StreamCommonStatus;
   VMIP_STREAM_NETWORK_STATUS StreamNetworkStatus;

   while(!*RequestStop)
   {
      VMIP_GetStreamCommonStatus(Stream, &StreamCommonStatus);
      VMIP_GetStreamNetworkStatus(Stream, &StreamNetworkStatus);

      std::cout << "SlotCount: " << StreamCommonStatus.SlotCount << " - SlotFilling: " << StreamCommonStatus.ApplicativeBufferQueueFilling << " - SlotDropped: " << StreamNetworkStatus.SlotDropped << " - PacketLost:  " << StreamNetworkStatus.PacketLost << "                      \r" << std::flush;

      std::this_thread::sleep_for(100ms);
   }
}

void MonitorTXStreamStatus(HANDLE Stream, bool* RequestStop)
{
   VMIP_STREAM_COMMON_STATUS StreamCommonStatus;
   VMIP_STREAM_NETWORK_STATUS StreamNetworkStatus;

   while(!*RequestStop)
   {
      VMIP_GetStreamCommonStatus(Stream, &StreamCommonStatus);
      VMIP_GetStreamNetworkStatus(Stream, &StreamNetworkStatus);

      std::cout << "SlotCount: " << StreamCommonStatus.SlotCount << " - SlotFilling: " << StreamCommonStatus.ApplicativeBufferQueueFilling << " - PacketUnderrun:  " << StreamNetworkStatus.PacketUnderrun << " - FrameUnderrun:  " << StreamNetworkStatus.FrameUnderrun << " - PacketDrop:  " << StreamNetworkStatus.PacketDrop << "                      \r" << std::flush;

      std::this_thread::sleep_for(100ms);
   }

   std::cout << std::endl;
}

void PrintWithLastError(std::string Message)
{
   const size_t MessageSize = 4096;
   char* pErrorMessage = new char[MessageSize];

   VMIP_GetLastErrorMessage(pErrorMessage, MessageSize);
   std::cout << Message << " : " << pErrorMessage << std::endl;

   if(pErrorMessage)
      delete[] pErrorMessage;
}

uint32_t GetEssenceDefaultBufferType(VMIP_ESSENCETYPE EssenceType)
{
   uint32_t BufferType = 0;
   switch (EssenceType)
   {
   case VMIP_ET_ST2110_20: 
      BufferType = VMIP_ST2110_20_BT_VIDEO;
      break;
   case VMIP_ET_ST2110_30:      
      BufferType = VMIP_ST2110_30_BT_AUDIO;
      break;
   case VMIP_ET_ST2110_40: 
      BufferType = VMIP_ST2110_40_BT_ANC_Y_1;
      break;
   case VMIP_ET_ST2110_22:
      BufferType = VMIP_ST2110_22_BT_FULL_FRAME;
      break;
   case NB_VMIP_ESSENCETYPE:
   default:
      BufferType = 0;
   }

   return BufferType;
}