/*
   IP Virtual Card
   ------------
   SYNCED TRANSMISSION SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a synchronized transmission stream use case.

   The application opens a Vcs context handle, a conductor, two streams and a synchronized stream handles, configures them, and
   enters in a transmission loop.
   One stream is configured to receive ST2110-20 video data and the other to receive ST2110-30 audio data.
   The synchronized stream allows the user to get data from the two streams in a synchronized way.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream_sync.h inclusion file
   and to VMIP library file must be properly configured.

*/

/*** HEADER INCLUSION ********************************************************/

#include <iostream>
#include <vector>
#include <string>
#include <cstring>
#include <thread>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>

#include "videomasterip/videomasterip_stream_sync.h"


int main(int argc, char* argv[])
{
   //User parameters
   const std::string NicName = "eth0";  //Streaming network interface controller name
   const uint32_t ConductorCpuCoreOsId = 1; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index
   const uint32_t DestinationAddress = 0xef1a0192; //IP destination address
   const uint16_t DestinationUdpPortVideo = 50020, DestinationUdpPortAudio = 50030, DestinationUdpPortAnc = 50040; //UDP destination port
   const uint32_t DestinationSsrc = 0; //SSRC destination
   const VMIP_VIDEO_STANDARD VideoStandard = VMIP_VIDEO_STANDARD_1920X1080P30; //Streaming video standard
   const AUDIO_PARAM AudioParam = { VMIP_ST2110_30_PACKET_TIME_1MS ,VMIP_AUDIO_SAMPLING_RATE_48KHZ,8,10 };
   const VMIP_ANC_SDI_VIDEOSTANDARD AncVideoStandard = VMIP_ANC_SDI_VIDEOSTD_1920X1080P30;

   HANDLE VcsContext = nullptr, StreamSync = nullptr, VideoStream = nullptr, AudioStream = nullptr, AncStream = nullptr, StreamSyncSlot = nullptr, AncSlot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   uint64_t NicId = (uint64_t)-1;
   VMIP_STREAMTYPE StreamType = VMIP_ST_TX;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;
   uint32_t FrameHeight = 0, FrameWidth = 0, FrameRate = 0;
   bool8_t Interlaced = false, IsUs = false;

   ESSENCE_PARAM VideoEssenceParam = {};
   VideoEssenceParam.EssenceType = VMIP_ET_ST2110_20;
   VideoEssenceParam.VideoStandard = VideoStandard;

   ESSENCE_PARAM AudioEssenceParam = {};
   AudioEssenceParam.EssenceType = VMIP_ET_ST2110_30;
   AudioEssenceParam.AudioParam = AudioParam;

   ESSENCE_PARAM AncEssenceParam = {};
   AncEssenceParam.EssenceType = VMIP_ET_ST2110_40;
   AncEssenceParam.AncVideoStandard = AncVideoStandard;

   uint32_t AudioSampleCount = 0;

   uint8_t* pVideoBuffer = nullptr, *pAudioBuffer = nullptr;
   uint32_t VideoBufferSize = 0, AudioBufferSize = 0, Index = 0, Line = 0;
   uint8_t* pVideoPatternBuffer = nullptr;
   std::string Sdp;
   VMIP_ANC_PACKET_CONTENT* pAncPacket = nullptr;

   init_keyboard();

   std::cout << "IP VIRTUAL CARD SYNCED ST2110-20 AND ST2110-30 TRANSMISSION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the VCS status");
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context");


   if (Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);
      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if (Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used 100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting conductor");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = GetNicIdFromName(VcsContext, NicName, &NicId);

   }

   if (Result == VMIPERR_NOERROR)
   {
      //Video Stream creation and configuration
      Result = ConfigureStream(VcsContext, StreamType, ProcessingCpuCoreOsId,
         { DestinationAddress }, { DestinationUdpPortVideo }, DestinationSsrc, VideoEssenceParam, { NicId }, ConductorId, ManagementThreadCpuCoreOsId, &VideoStream);
   }

   if (Result == VMIPERR_NOERROR)
   {
      //Audio Stream creation and configuration
      Result = ConfigureStream(VcsContext, StreamType, ProcessingCpuCoreOsId,
         { DestinationAddress }, { DestinationUdpPortAudio }, DestinationSsrc, AudioEssenceParam, { NicId }, ConductorId, ManagementThreadCpuCoreOsId, &AudioStream);
   }

   if (Result == VMIPERR_NOERROR)
   {
      //Anc Stream creation and configuration
      Result = ConfigureStream(VcsContext, StreamType, ProcessingCpuCoreOsId,
         { DestinationAddress }, { DestinationUdpPortAnc }, DestinationSsrc, AncEssenceParam, { NicId }, ConductorId, ManagementThreadCpuCoreOsId, &AncStream);
   }

   if (Result == VMIPERR_NOERROR)
   {

      Result = VMIP_CreateStreamSync(StreamType, 100, &StreamSync);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when creating the stream");
      }
   }


   if (Result == VMIPERR_NOERROR)
   {

      Result = VMIP_StreamSync_SetMainStream(StreamSync, VideoStream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when setting the main stream for the SyncStreams.");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {

      Result = VMIP_StreamSync_AddSecondaryStream(StreamSync, AudioStream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when adding audio secondary stream for the SyncStreams.");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {

      Result = VMIP_StreamSync_AddSecondaryStream(StreamSync, AncStream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when adding ANC secondary stream for the SyncStreams.");
      }
   }

   //Generate the SDP Video
   if (Result == VMIPERR_NOERROR)
      Result = GenerateSdp(VcsContext, VideoStream, &Sdp);

   if (Result == VMIPERR_NOERROR)
      std::cout << "VIDEO SDP:" << std::endl << Sdp << std::endl;

   //Generate the SDP Audio
   if (Result == VMIPERR_NOERROR)
      Result = GenerateSdp(VcsContext, AudioStream, &Sdp);

   if (Result == VMIPERR_NOERROR)
      std::cout << "AUDIO SDP:" << std::endl << Sdp << std::endl;

   //Generate the SDP ANC
   if (Result == VMIPERR_NOERROR)
      Result = GenerateSdp(VcsContext, AncStream, &Sdp);

   if (Result == VMIPERR_NOERROR)
      std::cout << "ANC SDP:" << std::endl << Sdp << std::endl;

   if(Result == VMIPERR_NOERROR)
   {
      //Video pattern buffer

      //Extract details from video standard
      Result = VMIP_GetVideoStandardInfo(VideoStandard, &FrameWidth, &FrameHeight, &FrameRate,
         &Interlaced, &IsUs);
      if (Result != VMIPERR_NOERROR)
      {
         std::cout << "Could not get the video standard information." << std::endl;
      }

      if(Result == VMIPERR_NOERROR)
      {
         //Video pattern generation
         pVideoPatternBuffer = new uint8_t[FrameWidth * FrameHeight * PIXELSIZE_10BIT];
         CreateColorBarPatternYUV422_10Bit_ST2110(pVideoPatternBuffer, FrameHeight, FrameWidth);
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStreamSync(StreamSync);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting the stream");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Reception started, press any key to stop..." << std::endl;

      //Transmission loop
      while (1)
      {

         if (_kbhit())
         {
            _getch();
            break;
         }
         Result = VMIP_StreamSync_LockSlot(StreamSync, &StreamSyncSlot);
         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               PrintWithLastError("Time out at slot " + std::to_string(Index));
               continue;
            }
            PrintWithLastError("Error when locking slot " + std::to_string(Index));
            break;
         }

         std::cout << "Slot " << Index << " locked";

         Result = VMIP_StreamSync_GetSlotBuffer(StreamSync, VideoStream, StreamSyncSlot, VMIP_ST2110_20_BT_VIDEO, &pVideoBuffer, &VideoBufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error in GetSlotBuffer for video stream " + std::to_string(Index));
            break;
         }
         std::cout << "Video buffer size: " << VideoBufferSize;

         Result = VMIP_StreamSync_GetSlotBuffer(StreamSync, AudioStream, StreamSyncSlot, VMIP_ST2110_30_BT_AUDIO, &pAudioBuffer, &AudioBufferSize);
         if(Result == VMIPERR_NOTFOUND)
            std::cout << "No audio data found synchronized with the video slot.";
         else if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error in GetSlotBuffer for audio stream " + std::to_string(Index));
            break;
         }
         else
         {
            std::cout << " - Audio buffer size: " << AudioBufferSize ;
         }

         //Copy test pattern buffer to slot
         if(pVideoPatternBuffer)
            std::memcpy(pVideoBuffer, pVideoPatternBuffer, VideoBufferSize);

         //Draw white line
         DrawWhiteLineYUV422_10Bit_ST2110(pVideoBuffer, Line, FrameHeight, FrameWidth, Interlaced);

         //Create audio at 1Khz
         CreateAudio1Khz(pAudioBuffer
            , AudioBufferSize
            , VmipFormatToNbBytes(VMIP_AUDIO_FORMAT_L24)
            , VmipSamplingRateToUint32(AudioEssenceParam.AudioParam.SamplingRate)
            , AudioEssenceParam.AudioParam.NbChannel
            , &AudioSampleCount);


         //ANC data
         Result = VMIP_StreamSync_GetAncSlot(StreamSync, AncStream, StreamSyncSlot, &AncSlot);
         if (Result != VMIPERR_NOERROR)
            std::cout << "Could not get ANC slot.";

         if (Result == VMIPERR_NOERROR)
            Result = VMIP_SlotAncAllocatePacket(AncStream, AncSlot, 8, &pAncPacket);

         if (Result == VMIPERR_NOERROR)
         {
            pAncPacket->DataID = 0x41;
            pAncPacket->SecondaryDataID = 0x05;
            pAncPacket->InHANC = false;

            /* Fill in the packet UDW with some data */
            pAncPacket->pUserDataWords[0] = 0x0244;
            for (int UDW = 1; UDW < pAncPacket->DataCount; UDW++)
            {
               pAncPacket->pUserDataWords[UDW] = 0x0200;
            }

            /* Insert the packet in C sample stream of video on the 9th line */
            Result = VMIP_SlotAncInsertPacket(AncStream, AncSlot, 9, VMIP_ST2110_40_BT_ANC_C_1, -1, pAncPacket);
            if (Result != VMIPERR_NOERROR)
               PrintWithLastError("ERROR : Cannot insert ANC packet. Result = " + std::to_string(Result));
         }
         else
            PrintWithLastError("ERROR : Cannot allocate ANC packet. Result = " + std::to_string(Result));

         std::cout << "\r" << std::flush;

         Result = VMIP_StreamSync_UnlockSlot(StreamSync, StreamSyncSlot);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking slot " + std::to_string(Index));
            break;
         }

         Index++;
      }

      Result = VMIP_StopStreamSync(StreamSync);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the stream " + std::to_string(Result));
   }

   if (AncStream)
   {
      Result = VMIP_DestroyStream(AncStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the anc stream " + std::to_string(Result));
   }

   if (AudioStream)
   {
      Result = VMIP_DestroyStream(AudioStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the audio stream " + std::to_string(Result));
   }

   if (VideoStream)
   {
      Result = VMIP_DestroyStream(VideoStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the video stream " + std::to_string(Result));
   }

   if (StreamSync)
   {
      Result = VMIP_DestroyStreamSync(StreamSync);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the syncstream " + std::to_string(Result));
   }

   if (ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if (VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context " + std::to_string(Result));
   }

   if (pVideoPatternBuffer)
      delete[] pVideoPatternBuffer;

   close_keyboard();

   return 0;
}
