/*
   IP Virtual Card
   ------------
   TRANSMISSION STREAM SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a transmission stream use case.

   The application opens a Vcs context handle, a conductor and a stream handles, configures them, and
   enters in a transmission loop.
   It also allows the configuration of a network interface.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream.h inclusion file
   and to VMIP library file must be properly configured.

*/

/*** HEADER INCLUSION ********************************************************/

#include <iostream>
#include <vector>
#include <string>
#include <iomanip>
#include <thread>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>

std::string Sdp = R"(v=0
o=- 1648826085 3835850490 IN IP4 10.1.1.2
s=Sample Media Stream
i=media stream created with Deltacast IPVirtualCard
t=0 0
m=video 1025 RTP/AVP 98
c=IN IP4 224.1.1.7
a=rtpmap:98 raw/90000
a=fmtp:98 exactframerate=30; SSN=ST2110-40:2018; TSMODE=NEW; TSDELAY=1000;
a=ts-refclk:ptp=IEEE1588-2008:39-A7-94-FF-FE-07-CB-D0:127
a=mediaclk:direct=0)";

int main(int argc, char* argv[])
{
   //User parameters
   const std::string NicName = "eth0";  //Streaming network interface controller name
   const uint32_t ConductorCpuCoreOsId = 1; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index

   HANDLE VcsContext = nullptr, Stream = nullptr, Slot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   uint64_t NicId = (uint64_t)-1;
   VMIP_STREAMTYPE StreamType = VMIP_ST_RX;
   VMIP_STREAM_COMMON_STATUS StreamStatus;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;
   
   uint32_t NbAncPackets = 0;
   VMIP_ANC_PACKET_CONTENT* pAncPacket = nullptr;
   uint32_t Index = 0;

   //get the initial print formating
   std::ios InitialFormat(NULL);
   InitialFormat.copyfmt(std::cout);

   init_keyboard();

   std::cout << "IP VIRTUAL CARD ST2110-40 RECEPTION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the VCS status");
      }
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context");

   if(Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);

      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if(Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used 100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting conductor");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = GetNicIdFromName(VcsContext, NicName, &NicId);

      if(Result == VMIPERR_NOERROR)
      {
         //Stream creation and configuration
         Result = ConfigureStreamFromSdp(VcsContext, StreamType, ProcessingCpuCoreOsId,
                                         Sdp, {NicId}, ConductorId, ManagementThreadCpuCoreOsId, &Stream);
      }
   }   

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(Stream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting stream");
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Reception started, press any key to stop..." << std::endl;

      //Transmission loop
      while (1)
      {
         if (_kbhit())
         {
            _getch();
            break;
         }

         //Try to lock the next slot.
         Result = VMIP_LockSlot(Stream, &Slot);

         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               PrintWithLastError("Time out at slot " + std::to_string(Index));
               continue;
            }
            PrintWithLastError("Error when locking slot at slot " + std::to_string(Index));
            break;
         }

         /* Get ANC content on the first 10 lines */
         for (int LineNumber = 1; LineNumber < 11; LineNumber++)
         {
            /* First check C samples stream */
            NbAncPackets = 0;
            Result = VMIP_SlotAncGetNbPackets(Stream, Slot, LineNumber, VMIP_ST2110_40_BT_ANC_C_1, &NbAncPackets);
            if (Result == VMIPERR_NOERROR)
            {
               if (NbAncPackets > 0)
               {
                  std::cout << std::endl << "Line " << LineNumber << " at slot " << Index << " - C ANC : " << NbAncPackets << " ANC packets" << std::endl;

                  /* Get all packets */
                  for (int32_t AncPacketIndex = 0; AncPacketIndex < (int32_t)NbAncPackets; AncPacketIndex++)
                  {
                     pAncPacket = nullptr;
                     Result = VMIP_SlotAncGetPacket(Stream, Slot, LineNumber, VMIP_ST2110_40_BT_ANC_C_1, AncPacketIndex, &pAncPacket);
                     if (Result == VMIPERR_NOERROR)
                     {
                        std::cout << "  ANC Packet " << AncPacketIndex << " : Did=" << std::hex << uint32_t(pAncPacket->DataID)
                           << " - SDid=" << uint32_t(pAncPacket->SecondaryDataID) << " - DC=" << std::dec << uint32_t(pAncPacket->DataCount) << std::endl;

                        std::cout << "    UDW : ";
                        for (int UDW = 0; UDW < pAncPacket->DataCount; UDW++)
                           std::cout << std::setfill('0') << std::setw(3) << std::hex << pAncPacket->pUserDataWords[UDW] << "."; //Format the print
                        std::cout << std::endl;
                        std::cout.copyfmt(InitialFormat); //reset the print formating
                     }
                     else
                     {
                        PrintWithLastError("Error when getting ANC packet " + std::to_string(AncPacketIndex) + " on line " + std::to_string(LineNumber) + " at slot " + std::to_string(Index) + " - C samples. Result = " + std::to_string(Result));
                        break;
                     }
                  }

                  if (Result != VMIPERR_NOERROR)
                     break;
               }
            }
            else
            {
               PrintWithLastError("Error when getting number of packets on line " + std::to_string(LineNumber) + " at slot " + std::to_string(Index) + " - C samples. Result = " + std::to_string(Result));
               break;
            }
         }

         
         //Unlock the slot. pBuffer wont be available anymore
         Result = VMIP_UnlockSlot(Stream, Slot);

         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking slot at slot " + std::to_string(Index));
            break;
         }

         Index++;
      }

      //Stop the stream
      Result = VMIP_StopStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the stream " + std::to_string(Result));
   }
   
   if(Stream)
   {
      Result = VMIP_DestroyStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the stream " + std::to_string(Result));
   }

   if(ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context : " + std::to_string(Result));
   }

   close_keyboard();

   return 0;
}