/*
   IP Virtual Card
   ------------
   TRANSMISSION STREAM SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a transmission stream use case.

   The application opens a Vcs context handle, a conductor and a stream handles, configures them, and
   enters in a transmission loop.
   It also allows the configuration of a network interface.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream.h inclusion file
   and to VMIP library file must be properly configured.

*/

/*** HEADER INCLUSION ********************************************************/

#include <iostream>
#include <vector>
#include <string>
#include <iomanip>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>
#include <videomasterip/videomasterip_anc_processing.h>

std::string Sdp = R"(v=0
o=- 1648826085 3835850490 IN IP4 10.1.1.2
s=Sample Media Stream
i=media stream created with Deltacast IPVirtualCard
t=0 0
m=video 1025 RTP/AVP 98
c=IN IP4 224.1.1.7
a=rtpmap:98 raw/90000
a=fmtp:98 exactframerate=30; SSN=ST2110-40:2018; TSMODE=NEW; TSDELAY=1000;
a=ts-refclk:ptp=IEEE1588-2008:39-A7-94-FF-FE-07-CB-D0:127
a=mediaclk:direct=0)";

int main(int argc, char* argv[])
{
   //User parameters
   const std::string NicName = "eth0";  //Streaming network interface controller name
   const uint32_t ConductorCpuCoreOsId = 1; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index

   HANDLE VcsContext = nullptr, Stream = nullptr, Slot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   uint64_t NicId = (uint64_t)-1;
   VMIP_STREAMTYPE StreamType = VMIP_ST_RX;
   VMIP_STREAM_COMMON_STATUS StreamStatus;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;
   VMIP_OP47_SLOT Op47Slot;
   char pText[40];
   
   uint32_t Index = 0;

   //get the initial print formating
   std::ios InitialFormat(NULL);
   InitialFormat.copyfmt(std::cout);

   init_keyboard();

   std::cout << "IP VIRTUAL CARD ST2110-40 RECEPTION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
      {
         std::cout << "Error when getting the VCS status" << std::endl;
      }
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      std::cout << "Error when creating the context." << std::endl;

   if(Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);

      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if(Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used � 100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            std::cout << "Error when starting conductor" << std::endl;
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = GetNicIdFromName(VcsContext, NicName, &NicId);

      if(Result == VMIPERR_NOERROR)
      {
         //Stream creation and configuration
         Result = ConfigureStreamFromSdp(VcsContext, StreamType, ProcessingCpuCoreOsId,
                                         Sdp, {NicId}, ConductorId, ManagementThreadCpuCoreOsId, &Stream);
      }
   }   

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(Stream);
      if (Result != VMIPERR_NOERROR)
      {
         std::cout << "Error when starting stream." << std::endl;
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Reception started, press any key to stop..." << std::endl;

      //Transmission loop
      while (1)
      {
         if (_kbhit())
         {
            _getch();
            break;
         }

         //Try to lock the next slot.
         Result = VMIP_LockSlot(Stream, &Slot);

         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               std::cout << "Time out at slot " << Index << std::endl;
               continue;
            }
            std::cout << std::endl << "Error when locking slot at slot " << Index << "." << std::endl;
            break;
         }

         Result = VMIP_SlotExtractOP47(Stream, Slot, &Op47Slot, 15);
         if (Result != VMIPERR_NOERROR)
         {
            std::cout << std::endl << "Error when extracting op 47 code at slot " << Index << "." << std::endl;
            break;
         }

         for (uint32_t i = 0;i < Op47Slot.NbPackets; i++)
         {
            char* pTemp = pText;
            for (uint32_t j = 0; j < 40; j++)
            {
               /* Print only ascii char */
               if ((Op47Slot.pVbiPacket[i].pData[j] & 0x7f) >= 32)
                  sprintf(pTemp++, "%c", Op47Slot.pVbiPacket[i].pData[j] & 0x7f);
            }

            printf("OP47 received packet Magazine : %d  Packet Number : %d  Line Number : %d  Field Number : %d  Text : %s\n", Op47Slot.pVbiPacket[i].MagazineNumber, Op47Slot.pVbiPacket[i].PacketNumber, Op47Slot.pVbiPacket[i].LineNumber, Op47Slot.pVbiPacket[i].FieldNumber, pText);

         }

         printf("\n");

         //Unlock the slot. pBuffer wont be available anymore
         Result = VMIP_UnlockSlot(Stream, Slot);

         if (Result != VMIPERR_NOERROR)
         {
            std::cout << std::endl << "Error when unlocking slot at slot " << Index << std::endl;
            break;
         }

         VMIP_GetStreamCommonStatus(Stream, &StreamStatus);
         std::cout << "Index : " << Index << ". SlotCount : " << StreamStatus.SlotCount << ". SlotFilling : " << StreamStatus.ApplicativeBufferQueueFilling << "                      \r" << std::flush;

         Index++;
      }

      //Stop the stream
      Result = VMIP_StopStream(Stream);
      if (Result != VMIPERR_NOERROR)
         std::cout << "Error when stopping the stream : " << Result << "." << std::endl;
   }
   
   if(Stream)
   {
      Result = VMIP_DestroyStream(Stream);
      if (Result != VMIPERR_NOERROR)
         std::cout << "Error when destroying the stream : " << Result << "." << std::endl;
   }

   if(ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         std::cout << "Error when stopping the conductor : " << Result << "." << std::endl;

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         std::cout << "Error when destroying the conductor : " << Result << "." << std::endl;
   }

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         std::cout << "Error when destroying the context : " << Result << "." << std::endl;
   }

   close_keyboard();

   return 0;
}