/*
   IP Virtual Card
   ------------
   TRANSMISSION STREAM SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a transmission stream use case.

   The application opens a Vcs context handle, a conductor and a stream handles, configures them, and
   enters in a transmission loop.
   It also allows the configuration of a network interface.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream.h inclusion file
   and to VMIP library file must be properly configured.

*/

/*** HEADER INCLUSION ********************************************************/

#include <iostream>
#include <vector>
#include <string>
#include <cstring>
#include <thread>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>

int main(int argc, char* argv[])
{
   //User parameters
   const std::string NicName = "eth0";  //Streaming network interface controller name
   const uint32_t ConductorCpuCoreOsId = 1; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index
   const uint32_t DestinationAddress = 0xe0010107; //IP destination address
   const uint16_t DestinationUdpPort = 1025; //UDP destination port
   const uint32_t DestinationSsrc = 0x12345600; //SSRC destination
   AUDIO_PARAM AudioParam = { VMIP_ST2110_30_PACKET_TIME_1MS ,VMIP_AUDIO_SAMPLING_RATE_48KHZ,8,10 };

   HANDLE VcsContext = nullptr, Stream = nullptr, Slot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   uint64_t NicId = (uint64_t)-1;
   VMIP_STREAMTYPE StreamType = VMIP_ST_TX;
   VMIP_STREAM_COMMON_STATUS StreamStatus;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;

   ESSENCE_PARAM AudioEssenceParam = {};
   AudioEssenceParam.EssenceType = VMIP_ET_ST2110_30;
   AudioEssenceParam.AudioParam = AudioParam;

   uint32_t AudioSampleCount = 0;
   uint8_t* pBuffer = nullptr;
   uint32_t BufferSize = 0, Index = 0;
   std::string Sdp;

   init_keyboard();

   std::cout << "IP VIRTUAL CARD ST2110-30 TRANSMISSION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the VCS status");
      }
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context.");

   if(Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);

      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if(Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used 100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting conductor");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = GetNicIdFromName(VcsContext, NicName, &NicId);

      if(Result == VMIPERR_NOERROR)
      {
         //Stream creation and configuration
         Result = ConfigureStream(VcsContext, StreamType, ProcessingCpuCoreOsId,
                                  {DestinationAddress}, {DestinationUdpPort}, DestinationSsrc, AudioEssenceParam, {NicId}, ConductorId, ManagementThreadCpuCoreOsId, &Stream);
      }
   }   

   //Generate the SDP
   if (Result == VMIPERR_NOERROR)
      Result = GenerateSdp(VcsContext, Stream, &Sdp);

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(Stream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting stream");
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Generated Sdp : " << std::endl << Sdp << std::endl;
      std::cout << std::endl << "Transmission started, press any key to stop..." << std::endl;

      bool StopMonitoring = false;
      std::thread Monitoringthread (MonitorTXStreamStatus, Stream, &StopMonitoring);
      //Transmission loop
      while (1)
      {
         if (_kbhit())
         {
            _getch();
            break;
         }

         //Try to lock the next slot.
         Result = VMIP_LockSlot(Stream, &Slot);

         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when locking slot at slot " + std::to_string(Index));
            break;
         }

         //Get the Audio buffer associated to the slot.
         Result = VMIP_GetSlotBuffer(Stream, Slot, VMIP_ST2110_30_BT_AUDIO, &pBuffer, &BufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when getting slot buffer at slot " + std::to_string(Index));
         }

         //Do TX data processing here on pBuffer

         //Create audio at 1Khz
         CreateAudio1Khz(pBuffer
            , BufferSize
            , VmipFormatToNbBytes(VMIP_AUDIO_FORMAT_L24)
            , VmipSamplingRateToUint32(AudioEssenceParam.AudioParam.SamplingRate)
            , AudioEssenceParam.AudioParam.NbChannel
            , &AudioSampleCount);
         
         //Unlock the slot. pBuffer wont be available anymore
         Result = VMIP_UnlockSlot(Stream, Slot);

         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking slot at slot " + std::to_string(Index));
            break;
         }
         
         Index++;
      }

      StopMonitoring = true;
      Monitoringthread.join();

      //Stop the stream
      Result = VMIP_StopStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the stream " + std::to_string(Result));
   }
   
   if(Stream)
   {
      Result = VMIP_DestroyStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the stream " + std::to_string(Result));
   }

   if(ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context : " + std::to_string(Result));
   }

   close_keyboard();

   return 0;
}