/*
   IP Virtual Card
   ------------
   TRANSMISSION STREAM SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a transmission stream use case.
   This sample also uses IntoPix's SDK to encode data and generate the JPEG-XS Video Support box and
   Color Specification box.

   The application opens a Vcs context handle, a conductor and a stream handles, configures them, and
   enters in a transmission loop. For each frame, the data are encoded using the IntoPix's SDK.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream.h inclusion file
   and to VMIP library file must be properly configured.

*/

/*** HEADER INCLUSION ********************************************************/

#include <iostream>
#include <vector>
#include <string>
#include <cstring>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"
#include "../tools_jpeg_xs.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>


int main(int argc, char* argv[])
{
   //User parameters
   const std::string NicName = "eth0";  //Streaming network interface controller name
   const uint32_t ConductorCpuCoreOsId = 1; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index
   const uint32_t DestinationAddress = 0xe0010107; //IP destination address
   const uint16_t DestinationUdpPort = 1025; //UDP destination port
   const uint32_t DestinationSsrc = 0x12345600; //SSRC destination
   const VMIP_VIDEO_STANDARD VideoStandard = VMIP_VIDEO_STANDARD_1920X1080P30; //Streaming video standard
   const uint32_t CompressedVideoSize = 777600; // Corresponds to 3bits per pixel for this video standard. View https://support.intopix.com/compression/.

   jxs_video_information_box_t jxs_video_information_box;
   jxs_profile_and_level_box_t jxs_profile_and_level_box;
   colour_specification_box_t colour_specification_box;

   size_t JPEG_XS_Boxes_size = InitJpegXsBoxes(&jxs_video_information_box, &jxs_profile_and_level_box, &colour_specification_box);

   HANDLE VcsContext = nullptr, Stream = nullptr, Slot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   uint64_t NicId = (uint64_t)-1;
   VMIP_STREAMTYPE StreamType = VMIP_ST_TX;
   VMIP_STREAM_COMMON_STATUS StreamStatus;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;
   ipxcpucodec_ret_code_t EncodeError;
   char EncodeErrorMessage[2048];

   uint32_t FrameHeight = 0, FrameWidth = 0, FrameRate = 0;
   bool8_t Interlaced = false, IsUs = false;

   ESSENCE_PARAM VideoEssenceParam = {};
   VideoEssenceParam.EssenceType = VMIP_ET_ST2110_22;
   VideoEssenceParam.Param_ST2110_20.Codec = VMIP_ST2110_22_CODEC_JPEG_XS;
   VideoEssenceParam.Param_ST2110_20.VideoStandard = VideoStandard;
   VideoEssenceParam.Param_ST2110_20.FrameSize = CompressedVideoSize + JPEG_XS_Boxes_size;

   uint8_t* pBuffer = nullptr;
   uint32_t BufferSize = 0, VideoPatternBufferSize = 0, Index = 0, Line = 0;
   uint8_t* pVideoPatternBuffer = nullptr, *pColorBarBuffer = nullptr;
   std::string Sdp;

   init_keyboard();

   std::cout << "IP VIRTUAL CARD ST2110-20 TRANSMISSION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   ipxcpucodec_encoder_t* encoder = nullptr;
   ipxcpucodec_image_geometry_t* image_geometry = nullptr;
   ipxcpucodec_image_format_cfg_t* image_format_cfg = nullptr;
   ipxcpucodec_ctx_options_t* ctx_options = nullptr;
   ipxcpucodec_encode_options_t* encode_options = nullptr;

   if (!InitJpegXsEncoder(CompressedVideoSize, &encoder, &image_geometry, &image_format_cfg, &ctx_options, &encode_options))
   {
      close_keyboard();
      return -1;
   }

   char msg[4096];
   ipxcpucodec_print_image_geometry(msg, image_geometry);
   printf("%s\n", msg);

   ipxcpucodec_print_image_format_cfg(msg, image_format_cfg);
   printf("%s\n", msg);

   uint32_t input_buffer_size = (ipxcpucodec_get_uncompressed_buffer_size(image_geometry, image_format_cfg) + 255U) & (~255U);

   std::cout << "input_buffer_size" << input_buffer_size << std::endl;


   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the VCS status");
      }
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context");


   if(Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);

      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if(Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used  100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting conductor");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = GetNicIdFromName(VcsContext, NicName, &NicId);

      if(Result == VMIPERR_NOERROR)
      {
         //Stream creation and configuration
         Result = ConfigureStream(VcsContext, StreamType, ProcessingCpuCoreOsId,
                                  {DestinationAddress}, {DestinationUdpPort}, DestinationSsrc, VideoEssenceParam, {NicId}, ConductorId, ManagementThreadCpuCoreOsId, &Stream);
      }
   }
   
   //Generate the SDP
   if (Result == VMIPERR_NOERROR)
      Result = GenerateSdp(VcsContext, Stream, &Sdp);

   if(Result == VMIPERR_NOERROR)
   {
      //Video pattern buffer

      //Extract details from video standard
      Result = VMIP_GetVideoStandardInfo(VideoStandard, &FrameWidth, &FrameHeight, &FrameRate,
         &Interlaced, &IsUs);
      if (Result != VMIPERR_NOERROR)
      {
         std::cout << "Could not get the video standard information." << std::endl;
      }

      if(Result == VMIPERR_NOERROR)
      {
         //Video pattern generation
         VideoPatternBufferSize = FrameWidth * FrameHeight * PIXELSIZE_10BIT;
         pVideoPatternBuffer = new uint8_t[VideoPatternBufferSize];
         pColorBarBuffer = new uint8_t[VideoPatternBufferSize];
         CreateColorBarPatternYUV422_10Bit_ST2110(pColorBarBuffer, FrameHeight, FrameWidth);
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(Stream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting stream" + std::to_string(Result));
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Generated Sdp : " << std::endl << Sdp << std::endl;
      std::cout << std::endl << "Transmission started, press any key to stop..." << std::endl;
      
      //Transmission loop
      while (1)
      {
         if (_kbhit())
         { 
            _getch();
            break;
         }

         //Try to lock the next slot.
         Result = VMIP_LockSlot(Stream, &Slot);
         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               PrintWithLastError("Timeout at slot " + std::to_string(Index));
               continue;
            }
            PrintWithLastError("Error when locking slot " + std::to_string(Index));
            break;
         }

         //Get the video buffer associated to the slot.
         Result = VMIP_GetSlotBuffer(Stream, Slot, VMIP_ST2110_22_BT_FULL_FRAME, &pBuffer, &BufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when getting slot buffer at slot " + std::to_string(Index));
         }

         //Write the JPEG-XS information boxes
         jxs_video_information_box.tcod.frame_count++;
         JPEG_XS_Boxes_size = write_rtp_boxes(&jxs_video_information_box, &jxs_profile_and_level_box, &colour_specification_box, pBuffer);

         std::memcpy(pVideoPatternBuffer, pColorBarBuffer, VideoPatternBufferSize);

         //Draw white line
         DrawWhiteLineYUV422_10Bit_ST2110(pVideoPatternBuffer, Line, FrameHeight, FrameWidth, Interlaced);

         //Increment line position
         Line++;
         if (Line > FrameHeight - 1) Line = 0;

         //Compress data in the buffer
         EncodeError = ipxcpucodec_encode(encoder, encode_options, image_geometry, image_format_cfg, VideoPatternBufferSize,
                                     pVideoPatternBuffer, BufferSize - JPEG_XS_Boxes_size,
                                     pBuffer + JPEG_XS_Boxes_size);
         if (EncodeError != IPXCPUCODEC_SUCCESS)
         {
            ipxcpucodec_ret_code_to_str(EncodeError, EncodeErrorMessage);
            std::cout << "Encoding error : " << EncodeErrorMessage;
            break;
         }

         //Unlock the slot. pBuffer wont be available anymore
         Result = VMIP_UnlockSlot(Stream, Slot);

         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking slot at slot " + std::to_string(Index));
            break;
         }

         VMIP_GetStreamCommonStatus(Stream, &StreamStatus);
         std::cout << "Index : " << Index << ". SlotCount : " << StreamStatus.SlotCount << ". SlotFilling : " << StreamStatus.ApplicativeBufferQueueFilling << "                      \r" << std::flush;

         Index++;
      }
      
      Result = VMIP_StopStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the stream " + std::to_string(Result));
   }
   
   if(Stream)
   {
      Result = VMIP_DestroyStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the stream " + std::to_string(Result));
   }

   if(ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context " + std::to_string(Result));
   }

   if(pVideoPatternBuffer)
      delete[] pVideoPatternBuffer;

   if (pColorBarBuffer)
      delete[] pColorBarBuffer;

   ipxcpucodec_release_ctx_options(ctx_options);
   ipxcpucodec_release_encoder(encoder);
   ipxcpucodec_release_image_geometry(image_geometry);
   ipxcpucodec_release_image_format_cfg(image_format_cfg);
   ipxcpucodec_release_encode_options(encode_options);

   close_keyboard();

   return 0;
}