/*
   IP Virtual Card
   ------------
   RECEPTION STREAM SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a ST2110-22 reception stream use case.
   This sample also uses IntoPix's SDK to decode data and interpret the JPEG-XS Video Support box
   and Color Specification box.

   The application opens a Vcs context handle, a conductor and a stream handles, configures them, and
   enters in a reception loop. For each frame, the data are decoded using the IntoPix's SDK.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream.h inclusion file
   and to VMIP library file must be properly configured.

*/

/*** HEADER INCLUSION ********************************************************/

#include <iostream>
#include <vector>
#include <string>
#include <cstring>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"
#include "../tools_jpeg_xs.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>

std::string Sdp = R"(v=0
o=- 1664441208 3834278216 IN IP4 127.0.0.1
s=Sample Media Stream
i=media stream created with Deltacast IPVirtualCard
t=0 0
m=video 1025 RTP/AVP 112
c=IN IP4 127.0.0.1
b=AS:186625
a=rtpmap:112 jxsv/90000
a=fmtp:112 sampling=YCbCr-4:2:2; width=1920; height=1080; exactframerate=30; depth=10; TCS=SDR; colorimetry=BT709; TP=2110TPW; packetmode=0; TSMODE=NEW; TSDELAY=1000;
a=ts-refclk:ptp=IEEE1588-2008:39-A7-94-FF-FE-07-CB-D0:127
a=mediaclk:direct=0
)";


int main(int argc, char* argv[])
{
   //User parameters
   const std::string NicName = "eth0";  //Streaming network interface controller name
   const uint32_t ConductorCpuCoreOsId = 3; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index

   jxs_video_information_box_t jxs_video_information_box;
   jxs_profile_and_level_box_t jxs_profile_and_level_box;
   colour_specification_box_t colour_specification_box;
   uint32_t JPEG_XS_Boxes_size = 0;

   HANDLE VcsContext = nullptr, Stream = nullptr, Slot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   uint64_t NicId = (uint64_t)-1;
   VMIP_STREAMTYPE StreamType = VMIP_ST_RX;
   VMIP_STREAM_COMMON_STATUS StreamStatus;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;
   ipxcpucodec_ret_code_t DecodeError;
   char DecodeErrorMessage[2048];

   uint32_t FrameHeight = 1080, FrameWidth = 1920;

   //Buffer that will be created and filled by the API
   uint8_t* pBuffer = nullptr;
   uint32_t BufferSize = 0, Index = 0;;

   //Buffer in which the data will be decoded
   uint32_t DecodedBufferSize = 0;
   uint8_t* pDecodedBuffer = nullptr;
   

   init_keyboard();

   std::cout << "IP VIRTUAL CARD ST2110-20 TRANSMISSION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   ipxcpucodec_decoder_t* decoder = nullptr;
   ipxcpucodec_image_geometry_t* image_geometry = nullptr;
   ipxcpucodec_image_format_cfg_t* image_format_cfg = nullptr;
   ipxcpucodec_ctx_options_t* ctx_options = nullptr;
   ipxcpucodec_decode_options_t* decode_options = nullptr;

   if (!InitJpegXsDecoder(&decoder, &image_geometry, &image_format_cfg, &ctx_options, &decode_options))
   {
      close_keyboard();
      return -1;
   }

   DecodedBufferSize = FrameWidth * FrameHeight * PIXELSIZE_10BIT;
   pDecodedBuffer = new uint8_t[DecodedBufferSize];

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   if (Result == VMIPERR_NOERROR)
      Result = VMIP_CreateVCSContext("", &VcsContext);

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the VCS status");
      }
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context.");


   if(Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);

      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if(Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used  100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting conductor");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = GetNicIdFromName(VcsContext, NicName, &NicId);

      if(Result == VMIPERR_NOERROR)
      {
         //Stream creation and configuration
         Result = ConfigureStreamFromSdp(VcsContext, StreamType, ProcessingCpuCoreOsId,
                                         Sdp, {NicId}, ConductorId, ManagementThreadCpuCoreOsId, &Stream);
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(Stream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting stream");
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Reception started, press any key to stop..." << std::endl;
      
      //Transmission loop
      while (1)
      {
         if (_kbhit())
         { 
            _getch();
            break;
         }

         //Try to lock the next slot
         Result = VMIP_LockSlot(Stream, &Slot);

         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               PrintWithLastError("Time out at slot " + std::to_string(Index));
               continue;
            }
            PrintWithLastError("Error when locking slot " + std::to_string(Index));
            break;
         }


         //Get the video buffer associated to the slot.
         Result = VMIP_GetSlotBuffer(Stream, Slot, VMIP_ST2110_22_BT_FULL_FRAME, &pBuffer, &BufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when getting slot buffer at slot " + std::to_string(Index));
         }

         // Read JPEG-XS information boxes
         JPEG_XS_Boxes_size = get_boxes_size(pBuffer);
         parse_rtp_boxes(pBuffer, &jxs_video_information_box, &jxs_profile_and_level_box, &colour_specification_box);

         // Decode video data
         DecodeError = ipxcpucodec_decode(decoder, decode_options, image_format_cfg,
            BufferSize - JPEG_XS_Boxes_size, pBuffer + JPEG_XS_Boxes_size, DecodedBufferSize, pDecodedBuffer, NULL, NULL);
         if (DecodeError) {
            ipxcpucodec_ret_code_to_str(DecodeError, DecodeErrorMessage);
            std::cout << "Decoding error : " << DecodeErrorMessage;
            break;
         }

         
         //Unlock the slot. pBuffer wont be available anymore
         Result = VMIP_UnlockSlot(Stream, Slot);

         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking slot " + std::to_string(Index));
            break;
         }

         VMIP_GetStreamCommonStatus(Stream, &StreamStatus);
         std::cout << "Index : " << Index << ". SlotCount : " << StreamStatus.SlotCount << ". SlotFilling : " << StreamStatus.ApplicativeBufferQueueFilling << "                      \r" << std::flush;

         Index++;
      }
      
      Result = VMIP_StopStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the stream " + std::to_string(Result));
   }
   
   if(Stream)
   {
      Result = VMIP_DestroyStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the stream " + std::to_string(Result));
   }

   if(ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context " + std::to_string(Result));
   }

   if(pDecodedBuffer)
      delete[] pDecodedBuffer;

   ipxcpucodec_release_ctx_options(ctx_options);
   ipxcpucodec_release_decoder(decoder);
   ipxcpucodec_release_image_geometry(image_geometry);
   ipxcpucodec_release_image_format_cfg(image_format_cfg);
   ipxcpucodec_release_decode_options(decode_options);

   close_keyboard();

   return 0;
}