/*
   IP Virtual Card
   ------------
   TRANSMISSION STREAM SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a transmission stream use case.

   The application opens a Vcs context handle, a conductor and a stream handles, configures them, and
   enters in a transmission loop.
   It also allows the configuration of a network interface.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream.h inclusion file
   and to VMIP library file must be properly configured.

*/

/*** HEADER INCLUSION ********************************************************/

#include <iostream>
#include <vector>
#include <string>
#include <cstring>
#include <thread>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>



int main(int argc, char* argv[])
{
   //User parameters
   const std::vector<std::string> NicNames = { "eth0", "eth1" };  //Streaming network interface controller names
   const uint32_t ConductorCpuCoreOsId = 1; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index
   const std::vector<uint32_t> DestinationAddresses = { 0xe0010107, 0xe0010108}; //IP destination addresses
   const std::vector<uint16_t> DestinationUdpPorts = { 1025, 1026}; //UDP destination ports
   const uint32_t DestinationSsrc = 0x12345600; //SSRC destination
   const VMIP_VIDEO_STANDARD VideoStandard = VMIP_VIDEO_STANDARD_1920X1080P30; //Streaming video standard

   HANDLE VcsContext = nullptr, Stream = nullptr, Slot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   std::vector<uint64_t> NicIds;
   VMIP_STREAMTYPE StreamType = VMIP_ST_TX;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;
   uint32_t FrameHeight = 0, FrameWidth = 0, FrameRate = 0;
   bool8_t Interlaced = false, IsUs = false;

   ESSENCE_PARAM VideoEssenceParam = {};
   VideoEssenceParam.EssenceType = VMIP_ET_ST2110_20;
   VideoEssenceParam.VideoStandard = VideoStandard;

   uint8_t* pBuffer = nullptr;
   uint32_t BufferSize = 0, Index = 0, Line = 0;
   uint8_t* pVideoPatternBuffer = nullptr;
   std::string Sdp;

   init_keyboard();

   std::cout << "IP VIRTUAL CARD ST2110-20 TRANSMISSION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the VCS status");
      }
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context.");


   if(Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);

      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if(Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used 100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting conductor");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      for(std::string NicName : NicNames)
      {
         uint64_t NicId;
         Result = GetNicIdFromName(VcsContext, NicName, &NicId);
         if (Result != VMIPERR_NOERROR)
            break;
         NicIds.push_back(NicId);
      }

      if(Result == VMIPERR_NOERROR)
      {
         //Stream creation and configuration
         Result = ConfigureStream(VcsContext, StreamType, ProcessingCpuCoreOsId,
                                  DestinationAddresses, DestinationUdpPorts, DestinationSsrc, VideoEssenceParam, NicIds, ConductorId, ManagementThreadCpuCoreOsId, &Stream);
      }
   }
   
   //Generate the SDP
   if (Result == VMIPERR_NOERROR)
      Result = GenerateSdp(VcsContext, Stream, &Sdp);

   if(Result == VMIPERR_NOERROR)
   {
      //Video pattern buffer

      //Extract details from video standard
      Result = VMIP_GetVideoStandardInfo(VideoStandard, &FrameWidth, &FrameHeight, &FrameRate,
         &Interlaced, &IsUs);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Could not get the video standard information.");
      }

      if(Result == VMIPERR_NOERROR)
      {
         //Video pattern generation
         pVideoPatternBuffer = new uint8_t[FrameWidth * FrameHeight * PIXELSIZE_10BIT];
         CreateColorBarPatternYUV422_10Bit_ST2110(pVideoPatternBuffer, FrameHeight, FrameWidth);
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(Stream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting stream");
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Generated Sdp : " << std::endl << Sdp << std::endl;
      std::cout << std::endl << "Transmission started, press any key to stop..." << std::endl;
            
      bool StopMonitoring = false;
      std::thread Monitoringthread (MonitorTXStreamStatus, Stream, &StopMonitoring);
      //Transmission loop
      while (1)
      {
         if (_kbhit())
         { 
            _getch();
            break;
         }

         //Try to lock the next slot.
         Result = VMIP_LockSlot(Stream, &Slot);

         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when locking slot at slot " + std::to_string(Index));
            break;
         }

         //Get the video buffer associated to the slot.
         Result = VMIP_GetSlotBuffer(Stream, Slot, VMIP_ST2110_20_BT_VIDEO, &pBuffer, &BufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when getting slot buffer at slot " + std::to_string(Index));
         }

         //Do TX data processing here on pBuffer

         //Copy test pattern buffer to slot
         if(pVideoPatternBuffer)
            std::memcpy(pBuffer, pVideoPatternBuffer, BufferSize);

         //Draw white line
         DrawWhiteLineYUV422_10Bit_ST2110(pBuffer, Line, FrameHeight, FrameWidth, Interlaced);

         //Increment line position
         Line++;
         if (Line > FrameHeight - 1) Line = 0;
         
         //Unlock the slot. pBuffer wont be available anymore
         Result = VMIP_UnlockSlot(Stream, Slot);

         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking slot at slot " + std::to_string(Index));
            break;
         }         

         Index++;
      }

      StopMonitoring = true;
      Monitoringthread.join();
      
      Result = VMIP_StopStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the stream " + std::to_string(Result));
   }
   
   if(Stream)
   {
      Result = VMIP_DestroyStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the stream " + std::to_string(Result));
   }

   if(ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context " + std::to_string(Result));
   }

   if(pVideoPatternBuffer)
      delete[] pVideoPatternBuffer;

   close_keyboard();

   return 0;
}