/*
   IP Virtual Card
   ------------
   RECEPTION STREAM SAMPLE APPLICATION WITH VIEWER
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a reception stream use case.

   This application also demonstrate the use of the video-viewer library to display the received video stream. 
   The video viewer code can be found in this github repository : https://github.com/deltacasttv/video-viewer

   The application opens a Vcs context handle, a conductor and a stream handles, configures them, and
   enters in a reception loop feeding the video-viewer with received frames.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream.h inclusion file
   and to VMIP library file must be properly configured.
*/

#include <iostream>
#include <vector>
#include <string>
#include <thread>
#include <cstring>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>

#include "videoviewer/videoviewer.hpp"

std::string Sdp = R"(v=0
o=- 1648821363 1453711395 IN IP4 10.1.1.2
s=Sample Media Stream
i=media stream created with Deltacast IPVirtualCard
t=0 0
m=video 1025 RTP/AVP 96
c=IN IP4 224.1.1.7
a=rtpmap:96 raw/90000
a=fmtp:96 sampling=YCbCr-4:2:2; width=1920; height=1080; exactframerate=30; depth=10; TCS=UNSPECIFIED; colorimetry=BT709; PM=2110GPM; TP=2110TPW; SSN=ST2110-20:2017; TSMODE=NEW; TSDELAY=1000;
a=ts-refclk:ptp=IEEE1588-2008:39-A7-94-FF-FE-07-CB-D0:127
a=mediaclk:direct=0)";

void RenderVideo(Deltacast::VideoViewer& Monitor, int WindowWidth, int WindowHeight, const char* WindowTitle, int TextureWidth, int TextureHeight, Deltacast::VideoViewer::InputFormat InputFormat, int FrameRateInMs)
{
   if (Monitor.init(WindowWidth, WindowHeight, WindowTitle, TextureWidth, TextureHeight, InputFormat))
   {
      Monitor.render_loop(FrameRateInMs);
      Monitor.release();
   }
   else
      std::cout << "VideoViewer initialization failed" << std::endl;

}

int main(int argc, char* argv[])
{
   //User parameters
   const std::string NicName = "eth0";  //Streaming network interface controller name
   const uint32_t ConductorCpuCoreOsId = 1; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index

   HANDLE VcsContext = nullptr, Stream = nullptr, Slot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   uint64_t NicId = (uint64_t)-1;
   VMIP_STREAMTYPE StreamType = VMIP_ST_RX;
   VMIP_STREAM_COMMON_STATUS StreamStatus;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;

   //Buffer that will be created and filled by the API
   uint8_t* pBuffer = nullptr;
   uint32_t BufferSize = 0, Index = 0;

   //viewer variables
   Deltacast::VideoViewer Monitor;
   std::thread* ViewerThread = nullptr;
   uint32_t Height = 0, Width = 0, FrameRate = 0;
   bool8_t Interlaced = false, IsUs = false;
   VMIP_STREAM_ESSENCE_CONFIG EssenceConfig;
   uint8_t *pMonitorData = nullptr;
   uint64_t MonitorDataSize = 0;

   init_keyboard();

   std::cout << "IP VIRTUAL CARD ST2110-20 RECEPTION SAMPLE APPLICATION WITH VIEWER\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the VCS status");
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context");

   if(Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);

      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if(Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used 100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting conductor");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = GetNicIdFromName(VcsContext, NicName, &NicId);

      if(Result == VMIPERR_NOERROR)
      {
         //Stream creation and configuration
         Result = ConfigureStreamFromSdp(VcsContext, StreamType, ProcessingCpuCoreOsId,
                                         Sdp, {NicId}, ConductorId, ManagementThreadCpuCoreOsId, &Stream);
      }
   }

   if(Result == VMIPERR_NOERROR)
   {

      Result = VMIP_GetStreamEssenceConfig(Stream,&EssenceConfig);
      if(Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting stream essence config");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVideoStandardInfo(EssenceConfig.EssenceS2110_20Prop.VideoStandard, &Width, &Height, &FrameRate, &Interlaced, &IsUs);
      if(Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting video standard info");
      }
      else
         ViewerThread = new std::thread(RenderVideo, std::ref(Monitor), 800, 600, "Deltacast.tv Video viewer", Width, Height, Deltacast::VideoViewer::InputFormat::ycbcr_422_10_be, FrameRate);
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(Stream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting stream.");
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Reception started, press any key to stop..." << std::endl;

      bool StopMonitoring = false;
      std::thread MonitoringThread (MonitorRXStreamStatus, Stream, &StopMonitoring);
      //Transmission loop
      while (1)
      {
         if (_kbhit())
         {
            _getch();
            break;
         }

         if(Monitor.window_request_close())
         {
            break;
         }

         //Try to lock the next slot.
         Result = VMIP_LockSlot(Stream, &Slot);
         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               PrintWithLastError("Time out at slot " + std::to_string(Index));
               continue;
            }
            PrintWithLastError("Error when locking slot " + std::to_string(Index));
            break;
         }

         //Get the video buffer associated to the slot.
         Result = VMIP_GetSlotBuffer(Stream, Slot, VMIP_ST2110_20_BT_VIDEO, &pBuffer, &BufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when getting slot buffer " + std::to_string(Index));
         }

         //Set data to viewer
         if (Monitor.lock_data(&pMonitorData, &MonitorDataSize)) 
         {
            if (MonitorDataSize == BufferSize && pBuffer)
            {
               if (pMonitorData)
               {
                  memcpy(pMonitorData, pBuffer, MonitorDataSize);
               }
            }

            Monitor.unlock_data();
         }


         //Unlock the slot. pBuffer wont be available anymore
         Result = VMIP_UnlockSlot(Stream, Slot);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking slot " + std::to_string(Index));
            break;
         }

         Index++;

      }
      
      StopMonitoring = true;
      MonitoringThread.join();

      if(Monitor.window_request_close())
      {
         std::cout << std::endl << "Viewer window closed : shutting down the application" << std::endl;
      }

      Monitor.stop();
      ViewerThread->join();
      delete ViewerThread;

      Result = VMIP_StopStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the stream " + std::to_string(Result));
   }
   
   if(Stream)
   {
      Result = VMIP_DestroyStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the stream " + std::to_string(Result));
   }

   if(ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context " + std::to_string(Result));
   }

   close_keyboard();

   return 0;
}