/*
   IP Virtual Card
   ------------
   RECEPTION STREAM SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a reception stream use case.

   The application opens a Vcs context handle, a conductor and a stream handles, configures them, and
   enters in a transmission loop.
   It also allows the configuration of a network interface.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream.h inclusion file
   and to VMIP library file must be properly configured.
*/

#include <iostream>
#include <vector>
#include <string>
#include <thread>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>

std::string Sdp = R"(v=0
o=- 1676459060 1131603683 IN IP4 10.1.1.2
s=Sample Media Stream
i=media stream created with Deltacast IPVirtualCard
t=0 0
a=group:DUP 0 1
m=video 1025 RTP/AVP 96
c=IN IP4 224.1.1.7
a=rtpmap:96 raw/90000
a=fmtp:96 sampling=YCbCr-4:2:2; width=1920; height=1080; exactframerate=30; depth=10; TCS=SDR; colorimetry=BT709; PM=2110GPM; TP=2110TPW; SSN=ST2110-20:2017; TSMODE=NEW; TSDELAY=1000;
a=mid:0
a=ts-refclk:ptp=IEEE1588-2008:D0-CB-07-FE-FF-94-A7-39:127
a=mediaclk:direct=0
m=video 1026 RTP/AVP 96
c=IN IP4 224.1.1.8
a=rtpmap:96 raw/90000
a=fmtp:96 sampling=YCbCr-4:2:2; width=1920; height=1080; exactframerate=30; depth=10; TCS=SDR; colorimetry=BT709; PM=2110GPM; TP=2110TPW; SSN=ST2110-20:2017; TSMODE=NEW; TSDELAY=1000;
a=mid:1
a=ts-refclk:ptp=IEEE1588-2008:D0-CB-07-FE-FF-94-A7-39:127
a=mediaclk:direct=0)";

int main(int argc, char* argv[])
{
   //User parameters
   const std::vector<std::string> NicNames = { "eth0", "eth1"};  //Streaming network interface controller name
   const uint32_t ConductorCpuCoreOsId = 1; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index

   HANDLE VcsContext = nullptr, Stream = nullptr, Slot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   std::vector<uint64_t> NicIds;
   VMIP_STREAMTYPE StreamType = VMIP_ST_RX;
   VMIP_STREAM_COMMON_STATUS StreamStatus;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;

   //Buffer that will be created and filled by the API
   uint8_t* pBuffer = nullptr;
   uint32_t BufferSize = 0, Index = 0;

   init_keyboard();

   std::cout << "IP VIRTUAL CARD ST2110-20 RECEPTION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the VCS status");
      }
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context.");

   if(Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);

      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if(Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used 100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting conductor");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      for (std::string NicName : NicNames)
      {
         uint64_t NicId;
         Result = GetNicIdFromName(VcsContext, NicName, &NicId);
         if (Result != VMIPERR_NOERROR)
            break;
         NicIds.push_back(NicId);
      }

      if(Result == VMIPERR_NOERROR)
      {
         //Stream creation and configuration
         Result = ConfigureStreamFromSdp(VcsContext, StreamType, ProcessingCpuCoreOsId,
                                         Sdp, NicIds, ConductorId, ManagementThreadCpuCoreOsId, &Stream);
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(Stream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting stream");
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Reception started, press any key to stop..." << std::endl;

      bool StopMonitoring = false;
      std::thread Monitoringthread (MonitorRXStreamStatus, Stream, &StopMonitoring);
      //Transmission loop
      while (1)
      {
         if (_kbhit())
         {
            _getch();
            break;
         }

         //Try to lock the next slot.
         Result = VMIP_LockSlot(Stream, &Slot);
         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               PrintWithLastError("Time out at slot " + std::to_string(Index));
               continue;
            }
            PrintWithLastError("Error when locking slot " + std::to_string(Index));
            break;
         }

         //Get the video buffer associated to the slot.
         Result = VMIP_GetSlotBuffer(Stream, Slot, VMIP_ST2110_20_BT_VIDEO, &pBuffer, &BufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when getting slot buffer " + std::to_string(Index));
         }

         //Do RX data processing here on pBuffer

         //Unlock the slot. pBuffer wont be available anymore
         Result = VMIP_UnlockSlot(Stream, Slot);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking slot " + std::to_string(Index));
            break;
         }

         Index++;
      }

      StopMonitoring = true;
      Monitoringthread.join();

      Result = VMIP_StopStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the stream " + std::to_string(Result));
   }
   
   if(Stream)
   {
      Result = VMIP_DestroyStream(Stream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the stream " + std::to_string(Result));
   }

   if(ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context " + std::to_string(Result));
   }

   close_keyboard();

   return 0;
}