/*
   IP Virtual Card
   ------------
   SYNCED RECEPTION SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK in a synchronized reception stream use case.

   The application opens a Vcs context handle, a conductor, two streams and a synchronized stream handles, configures them, and
   enters in a transmission loop.
   One stream is configured to receive ST2110-20 video data and the other to receive ST2110-30 audio data.
   The synchronized stream allows the user to get data from the two streams in a synchronized way.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream_sync.h inclusion file
   and to VMIP library file must be properly configured.

*/

/*** HEADER INCLUSION ********************************************************/

#include <iostream>
#include <vector>
#include <string>
#include <cstring>
#include <iomanip>
#include <thread>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>

#include "videomasterip/videomasterip_stream_sync.h"


int main(int argc, char* argv[])
{
   //User parameters
   const std::string NicName = "eth0";  //Streaming network interface controller name
   const uint32_t ConductorCpuCoreOsId = 1; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index
   const uint32_t DestinationAddress = 0xef1a0192; //IP destination address
   const uint16_t DestinationUdpPortVideo = 50020, DestinationUdpPortAudio = 50030, DestinationUdpPortAnc = 50040; //UDP destination port
   const uint32_t DestinationSsrc = 0; //SSRC destination
   const VMIP_VIDEO_STANDARD VideoStandard = VMIP_VIDEO_STANDARD_1920X1080P30; //Streaming video standard
   const AUDIO_PARAM AudioParam = { VMIP_ST2110_30_PACKET_TIME_1MS ,VMIP_AUDIO_SAMPLING_RATE_48KHZ,8,10 };
   const VMIP_ANC_SDI_VIDEOSTANDARD AncVideoStandard = VMIP_ANC_SDI_VIDEOSTD_1920X1080P30;

   HANDLE VcsContext = nullptr, StreamSync = nullptr, VideoStream = nullptr, AudioStream = nullptr, AncStream = nullptr, StreamSyncSlot = nullptr, AncSlot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   uint64_t NicId = (uint64_t)-1;
   VMIP_STREAMTYPE StreamType = VMIP_ST_RX;
   uint64_t ConductorId = (uint64_t)-1;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;
   uint32_t FrameHeight = 0, FrameWidth = 0, FrameRate = 0;
   bool8_t Interlaced = false, IsUs = false;

   ESSENCE_PARAM VideoEssenceParam = {};
   VideoEssenceParam.EssenceType = VMIP_ET_ST2110_20;
   VideoEssenceParam.VideoStandard = VideoStandard;

   ESSENCE_PARAM AudioEssenceParam = {};
   AudioEssenceParam.EssenceType = VMIP_ET_ST2110_30;
   AudioEssenceParam.AudioParam = AudioParam;

   ESSENCE_PARAM AncEssenceParam = {};
   AncEssenceParam.EssenceType = VMIP_ET_ST2110_40;
   AncEssenceParam.AncVideoStandard = AncVideoStandard;

   uint8_t* pVideoBuffer = nullptr, *pAudioBuffer = nullptr;
   uint32_t VideoBufferSize = 0, AudioBufferSize = 0, Index = 0, Line = 0;
   uint8_t* pVideoPatternBuffer = nullptr;

   uint32_t NbAncPackets = 0;
   VMIP_ANC_PACKET_CONTENT* pAncPacket = nullptr;

   //get the initial print formating
   std::ios InitialFormat(NULL);
   InitialFormat.copyfmt(std::cout);

   init_keyboard();

   std::cout << "IP VIRTUAL CARD SYNCED ST2110-20 AND ST2110-30 RECEPTION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
  if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when getting the VCS status");
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context");


   if (Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);
      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(ConductorCpuCoreOsId, VcsContext, &ConductorId);
      if (Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used 100%
         Result = VMIP_StartConductor(VcsContext, ConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting conductor");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = GetNicIdFromName(VcsContext, NicName, &NicId);

   }

   if (Result == VMIPERR_NOERROR)
   {
      //Video Stream creation and configuration
      Result = ConfigureStream(VcsContext, StreamType, ProcessingCpuCoreOsId,
         { DestinationAddress }, { DestinationUdpPortVideo }, DestinationSsrc, VideoEssenceParam, { NicId }, ConductorId, ManagementThreadCpuCoreOsId, &VideoStream);
   }

   if (Result == VMIPERR_NOERROR)
   {
      //Audio Stream creation and configuration
      Result = ConfigureStream(VcsContext, StreamType, ProcessingCpuCoreOsId,
         { DestinationAddress }, { DestinationUdpPortAudio }, DestinationSsrc, AudioEssenceParam, { NicId }, ConductorId, ManagementThreadCpuCoreOsId, &AudioStream);
   }

   if (Result == VMIPERR_NOERROR)
   {
      //Anc Stream creation and configuration
      Result = ConfigureStream(VcsContext, StreamType, ProcessingCpuCoreOsId,
         { DestinationAddress }, { DestinationUdpPortAnc }, DestinationSsrc, AncEssenceParam, { NicId }, ConductorId, ManagementThreadCpuCoreOsId, &AncStream);
   }

   if (Result == VMIPERR_NOERROR)
   {

      Result = VMIP_CreateStreamSync(StreamType, 100, &StreamSync);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when creating the stream");
      }
   }


   if (Result == VMIPERR_NOERROR)
   {

      Result = VMIP_StreamSync_SetMainStream(StreamSync, VideoStream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when setting the main stream for the SyncStreams.");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {

      Result = VMIP_StreamSync_AddSecondaryStream(StreamSync, AudioStream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when adding audio secondary stream for the SyncStreams.");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {

      Result = VMIP_StreamSync_AddSecondaryStream(StreamSync, AncStream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when adding ANC secondary stream for the SyncStreams.");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStreamSync(StreamSync);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting the stream");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Reception started, press any key to stop..." << std::endl;

      //Transmission loop
      while (1)
      {

         if (_kbhit())
         {
            _getch();
            break;
         }
         Result = VMIP_StreamSync_LockSlot(StreamSync, &StreamSyncSlot);
         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               PrintWithLastError("Time out at slot " + std::to_string(Index));
               continue;
            }
            PrintWithLastError("Error when locking slot " + std::to_string(Index));
            break;
         }

         std::cout << "Slot " << Index << " locked" << std::endl;

         Result = VMIP_StreamSync_GetSlotBuffer(StreamSync, VideoStream, StreamSyncSlot, VMIP_ST2110_20_BT_VIDEO, &pVideoBuffer, &VideoBufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error in GetSlotBuffer for video stream " + std::to_string(Index));
            break;
         }
         std::cout << "Video buffer size: " << VideoBufferSize << std::endl;
         std::cout << "Video first data: " << std::hex << *((uint32_t*)pVideoBuffer) << std::dec << std::endl;

         Result = VMIP_StreamSync_GetSlotBuffer(StreamSync, AudioStream, StreamSyncSlot, VMIP_ST2110_30_BT_AUDIO, &pAudioBuffer, &AudioBufferSize);
         if(Result == VMIPERR_NOTFOUND)
            std::cout << "No audio data found synchronized with the video slot." << std::endl;
         else if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error in GetSlotBuffer for audio stream " + std::to_string(Index));
            break;
         }
         else
         {
            std::cout << "Audio buffer size: " << AudioBufferSize << std::endl;
            std::cout << "Audio first data: " <<std::hex << *((uint32_t*)pAudioBuffer) << std::dec << std::endl;
         }

         //ANC data
         Result = VMIP_StreamSync_GetAncSlot(StreamSync, AncStream, StreamSyncSlot, &AncSlot);
         if (Result != VMIPERR_NOERROR)
            std::cout << "Could not get ANC slot.";

         /* Get ANC content on the first 10 lines */
         for (int LineNumber = 1; LineNumber < 11; LineNumber++)
         {
            /* First check C samples stream */
            NbAncPackets = 0;
            Result = VMIP_SlotAncGetNbPackets(AncStream, AncSlot, LineNumber, VMIP_ST2110_40_BT_ANC_C_1, &NbAncPackets);
            if (Result == VMIPERR_NOERROR)
            {
               if (NbAncPackets > 0)
               {
                  std::cout << std::endl << "Line " << LineNumber << " at slot " << Index << " - C ANC : " << NbAncPackets << " ANC packets" << std::endl;

                  /* Get all packets */
                  for (int32_t AncPacketIndex = 0; AncPacketIndex < (int32_t)NbAncPackets; AncPacketIndex++)
                  {
                     pAncPacket = nullptr;
                     Result = VMIP_SlotAncGetPacket(AncStream, AncSlot, LineNumber, VMIP_ST2110_40_BT_ANC_C_1, AncPacketIndex, &pAncPacket);
                     if (Result == VMIPERR_NOERROR)
                     {
                        std::cout << "  ANC Packet " << AncPacketIndex << " : Did=" << std::hex << uint32_t(pAncPacket->DataID)
                           << " - SDid=" << uint32_t(pAncPacket->SecondaryDataID) << " - DC=" << std::dec << uint32_t(pAncPacket->DataCount) << std::endl;

                        std::cout << "    UDW : ";
                        for (int UDW = 0; UDW < pAncPacket->DataCount; UDW++)
                           std::cout << std::setfill('0') << std::setw(3) << std::hex << pAncPacket->pUserDataWords[UDW] << "."; //Format the print
                        std::cout << std::endl;
                        std::cout.copyfmt(InitialFormat); //reset the print formating
                     }
                     else
                     {
                        PrintWithLastError("Error when getting ANC packet " + std::to_string(AncPacketIndex) + " on line " + std::to_string(LineNumber) + " at slot " + std::to_string(Index) + " - C samples. Result = " + std::to_string(Result));
                        break;
                     }
                  }

                  if (Result != VMIPERR_NOERROR)
                     break;
               }
            }
            else
            {
               PrintWithLastError("Error when getting number of packets on line " + std::to_string(LineNumber) + " at slot " + std::to_string(Index) + " - C samples. Result = " + std::to_string(Result));
               break;
            }
         }

         Result = VMIP_StreamSync_UnlockSlot(StreamSync, StreamSyncSlot);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking slot " + std::to_string(Index));
            break;
         }
         std::cout << std::endl << "---------------------------------------------------------------------------" << std::endl << std::endl;

         Index++;
      }

      Result = VMIP_StopStreamSync(StreamSync);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the stream " + std::to_string(Result));
   }

   if (AncStream)
   {
      Result = VMIP_DestroyStream(AncStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the ANC stream " + std::to_string(Result));
   }

   if (AudioStream)
   {
      Result = VMIP_DestroyStream(AudioStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the audio stream " + std::to_string(Result));
   }

   if (VideoStream)
   {
      Result = VMIP_DestroyStream(VideoStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the video stream " + std::to_string(Result));
   }

   if (StreamSync)
   {
      Result = VMIP_DestroyStreamSync(StreamSync);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the sync stream " + std::to_string(Result));
   }

   if (ConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, ConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if (VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context " + std::to_string(Result));
   }

   if (pVideoPatternBuffer)
      delete[] pVideoPatternBuffer;

   close_keyboard();

   return 0;
}
