/*
   IP Virtual Card
   ------------
   RECEPTION AND EMISSION STREAM SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK to receive, modify then transmit data.

   This sample can be used for any ST2110 essence type. The configuration of the stream is done by modifying the SDP string.

   The application opens a Vcs context handle, a conductor and two stream handles, configures them, and
   enters in a loop. In the loops, data are received then transmitted.

   In order to compile this application, path to videomasterip_core.h, videomasterip_cpu.h
   videomasterip_networkinterface.h, videomasterip_conductor.h and videomasterip_stream.h inclusion file
   and to VMIP library file must be properly configured.
*/

#include <iostream>
#include <vector>
#include <string>
#include <thread>
#include <cstring>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_conductor.h>
#include <videomasterip/videomasterip_stream.h>

std::string Sdp = R"(v=0
o=- 1648821363 1453711395 IN IP4 10.1.1.2
s=Sample Media Stream
i=media stream created with Deltacast IPVirtualCard
t=0 0
m=video 50020 RTP/AVP 96
c=IN IP4 239.26.1.146
a=rtpmap:96 raw/90000
a=fmtp:96 sampling=YCbCr-4:2:2; width=1280; height=720; exactframerate=25; depth=10; TCS=UNSPECIFIED; colorimetry=BT709; PM=2110GPM; TP=2110TPW; SSN=ST2110-20:2017; TSMODE=NEW; TSDELAY=1000;
a=ts-refclk:ptp=IEEE1588-2008:39-A7-94-FF-FE-07-CB-D0:127
a=mediaclk:direct=0)";

int main(int argc, char* argv[])
{
   //User parameters
   const std::string NicName = "ens2f1";  //Streaming network interface controller name
   const uint32_t RXConductorCpuCoreOsId = 1, TXConductorCpuCoreOsId = 4; //Conductor CPU core index
   const std::vector<uint32_t> ProcessingCpuCoreOsId = { 2 }; //Processing CPU core indexes list
   const uint32_t ManagementThreadCpuCoreOsId = 3; //Management Thread CPU core index
   const uint32_t DestinationAddress = 0xe0010107; //IP destination address
   const uint16_t DestinationUdpPort = 1025; //UDP destination port

   HANDLE VcsContext = nullptr, RXStream = nullptr, TXStream = nullptr, RXSlot = nullptr, TXSlot = nullptr;
   VMIP_VCS_STATUS VcsStatus;
   VMIP_STREAM_COMMON_CONFIG TXCommonConfig;
   VMIP_STREAM_NETWORK_CONFIG TXNetworkConfig;
   VMIP_STREAM_ESSENCE_CONFIG StreamEssenceConfig;
   uint64_t NicId = (uint64_t)-1;
   uint64_t RXConductorId = (uint64_t)-1, TXConductorId = (uint64_t)-1;;
   VMIP_ERRORCODE Result = VMIPERR_NOERROR;

   //Buffer that will be created and filled by the API
   uint8_t* pRXBuffer = nullptr, *pTXBuffer = nullptr;
   uint32_t RXBufferSize = 0, TXBufferSize = 0, Index = 0;

   init_keyboard();

   std::cout << "IP VIRTUAL CARD ST2110-20 RECEPTION AND EMISSION SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the VCS status");
      }
      else
         std::cout << "Connection established to VCS " << VcsStatus.VersionString << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context.");

   if(Result == VMIPERR_NOERROR)
   {
      PrintCpuCoresInfo(VcsContext);

      PrintNicsInfo(VcsContext);

      //Conductor creation and configuration.
      //The conductor will be responsible of receiving the packets at the fastest possible rate.
      Result = ConfigureConductor(RXConductorCpuCoreOsId, VcsContext, &RXConductorId);
      if(Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used 100%
         Result = VMIP_StartConductor(VcsContext, RXConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting RX conductor");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = ConfigureConductor(TXConductorCpuCoreOsId, VcsContext, &TXConductorId);
      if (Result == VMIPERR_NOERROR)
      {
         //Conductor start. At this point, the associated CPU core will be used 100%
         Result = VMIP_StartConductor(VcsContext, TXConductorId);
         if (Result != VMIPERR_NOERROR)
            PrintWithLastError("Error when starting TX conductor");
      }
   }

   if(Result == VMIPERR_NOERROR)
      Result = GetNicIdFromName(VcsContext, NicName, &NicId);

   if (Result == VMIPERR_NOERROR)
   {
      //RX Stream creation and configuration
      Result = ConfigureStreamFromSdp(VcsContext, VMIP_ST_RX, ProcessingCpuCoreOsId,
         Sdp, { NicId }, RXConductorId, ManagementThreadCpuCoreOsId, &RXStream);
   }

   if (Result == VMIPERR_NOERROR)
   {
      //TX Stream creation and configuration.
      // We configure the same way as the RX stream but we then modify the destination IP address and port.
      Result = ConfigureStreamFromSdp(VcsContext, VMIP_ST_TX, ProcessingCpuCoreOsId,
         Sdp, { NicId }, TXConductorId, ManagementThreadCpuCoreOsId, &TXStream);
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetStreamNetworkConfig(TXStream, &TXNetworkConfig);
      if (Result != VMIPERR_NOERROR)
      {
          PrintWithLastError("Error when getting the TX stream network config");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      TXNetworkConfig.pPathParameters[0].DestinationIp = DestinationAddress;
      TXNetworkConfig.pPathParameters[0].UdpPort = DestinationUdpPort;

      Result = VMIP_SetStreamNetworkConfig(TXStream, TXNetworkConfig);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when setting the TX stream network config");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetStreamEssenceConfig(TXStream, &StreamEssenceConfig);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the TX stream essence config");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      //We modify the start time of the TX stream to 0.5 second in the future to be able to bufferize a few frames before starting the emission.
      StreamEssenceConfig.HasStartTime = true;
      StreamEssenceConfig.StartTime = std::chrono::duration_cast<std::chrono::nanoseconds>(std::chrono::system_clock::now().time_since_epoch() + std::chrono::seconds(StreamEssenceConfig.TaiUtcDelta) + std::chrono::milliseconds(500)).count();
      Result = VMIP_SetStreamEssenceConfig(TXStream, StreamEssenceConfig);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when setting the TX stream essence config");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetStreamCommonConfig(TXStream, &TXCommonConfig);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the TX stream essence config");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      // We increase the buffer queue depth to be able to bufferize the data corresponding to the 0.5 second delay.
      TXCommonConfig.ApplicativeBufferQueueDepth = 10;
      Result = VMIP_SetStreamCommonConfig(TXStream, TXCommonConfig);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when setting the TX stream essence config");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(RXStream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting RX stream");
      }
   }

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_StartStream(TXStream);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when starting TX stream");
      }
   }

   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Reception and emission started, press any key to stop..." << std::endl;

      bool StopMonitoring = false;
      std::thread Monitoringthread (MonitorTXStreamStatus, TXStream, &StopMonitoring);
      //Transmission loop
      while (1)
      {
         if (_kbhit())
         {
            _getch();
            break;
         }

         //Try to lock the next slot.
         Result = VMIP_LockSlot(RXStream, &RXSlot);
         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               PrintWithLastError("Time out at RX slot " + std::to_string(Index));
               continue;
            }
            PrintWithLastError("Error when locking RX slot " + std::to_string(Index));
            break;
         }

         Result = VMIP_LockSlot(TXStream, &TXSlot);
         if (Result != VMIPERR_NOERROR)
         {
            if (Result == VMIPERR_TIMEOUT)
            {
               PrintWithLastError("Time out at TX slot " + std::to_string(Index));
               continue;
            }
            PrintWithLastError("Error when locking TX slot " + std::to_string(Index));
            break;
         }
         //Get the video buffer associated to the slot.
         Result = VMIP_GetSlotBuffer(RXStream, RXSlot, GetEssenceDefaultBufferType(StreamEssenceConfig.EssenceType), &pRXBuffer, &RXBufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when getting RX slot buffer " + std::to_string(Index));
            continue;
         }

         Result = VMIP_GetSlotBuffer(TXStream, TXSlot, GetEssenceDefaultBufferType(StreamEssenceConfig.EssenceType), &pTXBuffer, &TXBufferSize);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when getting RX slot buffer " + std::to_string(Index));
            continue;
         }


         if (RXBufferSize != TXBufferSize)
         {
            PrintWithLastError("RX buffer size (" + std::to_string(RXBufferSize) + ") is not the same as the TX buffer size (" + std::to_string(TXBufferSize) + ").");
            break;
         }

         //Copy RX buffer to TX buffer
         if (pRXBuffer && pTXBuffer)
            std::memcpy(pTXBuffer, pRXBuffer, std::min(RXBufferSize, TXBufferSize));

         //Eventual processing of the buffer could be done here.

         //Unlock the slot. pBuffer wont be available anymore
         Result = VMIP_UnlockSlot(RXStream, RXSlot);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking RX slot " + std::to_string(Index));
            break;
         }

         Result = VMIP_UnlockSlot(TXStream, TXSlot);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when unlocking TX slot " + std::to_string(Index));
            break;
         }

         Index++;
      }

      StopMonitoring = true;
      Monitoringthread.join();

      Result = VMIP_StopStream(TXStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the TX stream " + std::to_string(Result));

      Result = VMIP_StopStream(RXStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the RX stream " + std::to_string(Result));
   }
   
   if(RXStream)
   {
      Result = VMIP_DestroyStream(RXStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the RX stream " + std::to_string(Result));
   }

   if (TXStream)
   {
      Result = VMIP_DestroyStream(TXStream);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the TX stream " + std::to_string(Result));
   }

   if(RXConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, RXConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, RXConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if (TXConductorId != (uint64_t)-1)
   {
      Result = VMIP_StopConductor(VcsContext, TXConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when stopping the conductor " + std::to_string(Result));

      Result = VMIP_DestroyConductor(VcsContext, TXConductorId);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the conductor " + std::to_string(Result));
   }

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context " + std::to_string(Result));
   }

   close_keyboard();

   return 0;
}