/*
   IP Virtual Card
   ------------
   PTP SAMPLE APPLICATION
   (c) DELTACAST
   --------------------------------------------------------

   This application demonstrates the use of the IP Virtual Card and of the
   VideoMasterIP SDK to configure and get the state of the PTP service.

   The application opens a Vcs context handle, configures the PTP service and
   then get the PTP status.

   In order to compile this application, path to VideoMasterIP_Core.h, VideoMasterIP_PTP.h
    and VideoMasterIP_NetworkInterface.h inclusion file
   and to VMIP library file must be properly configured.
*/

#include <iostream>
#include <string>
#include <thread>
#include <chrono>

#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#if defined (__linux__) || defined (__APPLE__)
#include "../keyboard.h"
#else
#include <conio.h>
#define init_keyboard()
#define close_keyboard()
#endif

#include "../tools.h"

#include <videomasterip/videomasterip_core.h>
#include <videomasterip/videomasterip_ptp.h>

using namespace std::chrono_literals;

int main(int argc, char* argv[])
{
   VMIP_PTP_CONFIG PtpConfig = {};

   //User parameters
   const std::string NicName = "eth0";  //Network interface controller name on which the PTP packets are received

   VMIP_ERRORCODE Result = VMIPERR_NOERROR;

   HANDLE VcsContext = nullptr;
   VMIP_VCS_STATUS VcsStatus;

   init_keyboard();

   std::cout << "IP VIRTUAL CARD PTP SAMPLE APPLICATION\n(c) DELTACAST\n--------------------------------------------------------" << std::endl << std::endl;

   //Creates the context in which the stream will be created. This handle will be needed for all following calls.
   Result = VMIP_CreateVCSContext("", &VcsContext);
   if(Result == VMIPERR_NOERROR)
   {
      Result = VMIP_GetVCSStatus(VcsContext, &VcsStatus);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when getting the VCS status");
      }
      else
         std::cout << "Connection established to VCS " << VersionToString(VcsStatus.VcsVersion) << std::endl;
   }
   else
      PrintWithLastError("Error when creating the context");

   if (Result == VMIPERR_NOERROR)
   {
      Result = VMIP_CreatePTPDSFromProfile(VMIP_PTP_PROFILE_ST2059_2, &PtpConfig.DefaultDS, &PtpConfig.PortDS);
      if (Result != VMIPERR_NOERROR)
      {
         PrintWithLastError("Error when creating the PTP profile");
      }
   }
   if(Result == VMIPERR_NOERROR)
   {
      //Prints the information regarding the different NICs of the system.
      PrintNicsInfo(VcsContext);

      //Retrive the ID of the NIC and configure the PTP service
      Result = GetNicIdFromName(VcsContext, NicName, &PtpConfig.NetworkInterfaceId);
      
      if(Result == VMIPERR_NOERROR)
      {
         PtpConfig.Size = sizeof(VMIP_PTP_CONFIG);

         Result = VMIP_SetPTPConfig(VcsContext, PtpConfig);
         if (Result != VMIPERR_NOERROR)
         {
            PrintWithLastError("Error when setting the PTP configuration");
         }
      }
   }
   
   if(Result == VMIPERR_NOERROR)
   {
      std::cout << std::endl << "Status started, press any key to stop..." << std::endl;

      //PTP Status loop
      while (1)
      {
         if (_kbhit())
         {
            _getch();
            break;
         }

         PrintPtpStatus(VcsContext);

         std::this_thread::sleep_for(1s);
      }
   }     

   if(VcsContext)
   {
      Result = VMIP_DestroyVCSContext(VcsContext);
      if (Result != VMIPERR_NOERROR)
         PrintWithLastError("Error when destroying the context: " + std::to_string(Result));
   }

   close_keyboard();

   return 0;
}