/*!
   @file
   @brief This file gathers video related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/

#ifndef _VIDEOMASTERIP_VIDEO_H_
#define _VIDEOMASTERIP_VIDEO_H_

#include "videomasterip_core.h"

/*!
   @brief VideoMaster supported video standard

   The VMIP_VIDEO_STANDARD enumeration lists all the supported video standard.

   @see VMIP_STREAM_ESSENCE_CONFIG_ST2110_20 VMIP_STREAM_ESSENCE_CONFIG_ST2110_22
*/
typedef enum _VMIP_VIDEO_STANDARD
{
   VMIP_VIDEO_STANDARD_720X480I59,     /*!< 720x480i @ 59Hz */
   VMIP_VIDEO_STANDARD_720X487I59,     /*!< 720x487i @ 59Hz */
   VMIP_VIDEO_STANDARD_720X576I50,     /*!< 720x576i @ 50Hz */
   VMIP_VIDEO_STANDARD_1280X720P50,    /*!< 1280x720p @ 50Hz */
   VMIP_VIDEO_STANDARD_1280X720P59,    /*!< 1280x720p @ 59Hz */
   VMIP_VIDEO_STANDARD_1280X720P60,    /*!< 1280x720p @ 60Hz */
   VMIP_VIDEO_STANDARD_1920X1080I50,   /*!< 1920x1080i @ 50Hz */
   VMIP_VIDEO_STANDARD_1920X1080I59,   /*!< 1920x1080i @ 59Hz */
   VMIP_VIDEO_STANDARD_1920X1080I60,   /*!< 1920x1080i @ 60Hz */
   VMIP_VIDEO_STANDARD_1920X1080P23,   /*!< 1920x1080p @ 23Hz */
   VMIP_VIDEO_STANDARD_1920X1080P24,   /*!< 1920x1080p @ 24Hz */
   VMIP_VIDEO_STANDARD_1920X1080P25,   /*!< 1920x1080p @ 25Hz */
   VMIP_VIDEO_STANDARD_1920X1080P29,   /*!< 1920x1080p @ 29Hz */
   VMIP_VIDEO_STANDARD_1920X1080P30,   /*!< 1920x1080p @ 30Hz */
   VMIP_VIDEO_STANDARD_1920X1080P50,   /*!< 1920x1080p @ 50Hz */
   VMIP_VIDEO_STANDARD_1920X1080P59,   /*!< 1920x1080p @ 59Hz */
   VMIP_VIDEO_STANDARD_1920X1080P60,   /*!< 1920x1080p @ 60Hz */
   VMIP_VIDEO_STANDARD_2048X1080P23,   /*!< 2048x1080p @ 23Hz */
   VMIP_VIDEO_STANDARD_2048X1080P24,   /*!< 2048x1080p @ 24Hz */
   VMIP_VIDEO_STANDARD_2048X1080P25,   /*!< 2048x1080p @ 25Hz */
   VMIP_VIDEO_STANDARD_2048X1080P29,   /*!< 2048x1080p @ 29Hz */
   VMIP_VIDEO_STANDARD_2048X1080P30,   /*!< 2048x1080p @ 30Hz */
   VMIP_VIDEO_STANDARD_2048X1080P47,   /*!< 2048x1080p @ 47Hz */
   VMIP_VIDEO_STANDARD_2048X1080P48,   /*!< 2048x1080p @ 48Hz */
   VMIP_VIDEO_STANDARD_2048X1080P50,   /*!< 2048x1080p @ 50Hz */
   VMIP_VIDEO_STANDARD_2048X1080P59,   /*!< 2048x1080p @ 59Hz */
   VMIP_VIDEO_STANDARD_2048X1080P60,   /*!< 2048x1080p @ 60Hz */
   VMIP_VIDEO_STANDARD_3840X2160P23,   /*!< 3840x2160p @ 23Hz */
   VMIP_VIDEO_STANDARD_3840X2160P24,   /*!< 3840x2160p @ 24Hz */
   VMIP_VIDEO_STANDARD_3840X2160P25,   /*!< 3840x2160p @ 25Hz */
   VMIP_VIDEO_STANDARD_3840X2160P29,   /*!< 3840x2160p @ 29Hz */
   VMIP_VIDEO_STANDARD_3840X2160P30,   /*!< 3840x2160p @ 30Hz */
   VMIP_VIDEO_STANDARD_3840X2160P50,   /*!< 3840x2160p @ 50Hz */
   VMIP_VIDEO_STANDARD_3840X2160P59,   /*!< 3840x2160p @ 59Hz */
   VMIP_VIDEO_STANDARD_3840X2160P60,   /*!< 3840x2160p @ 60Hz */
   VMIP_VIDEO_STANDARD_4096X2160P23,   /*!< 4096x2160p @ 23Hz */
   VMIP_VIDEO_STANDARD_4096X2160P24,   /*!< 4096x2160p @ 24Hz */
   VMIP_VIDEO_STANDARD_4096X2160P25,   /*!< 4096x2160p @ 25Hz */
   VMIP_VIDEO_STANDARD_4096X2160P29,   /*!< 4096x2160p @ 29Hz */
   VMIP_VIDEO_STANDARD_4096X2160P30,   /*!< 4096x2160p @ 30Hz */
   VMIP_VIDEO_STANDARD_4096X2160P47,   /*!< 4096x2160p @ 47Hz */
   VMIP_VIDEO_STANDARD_4096X2160P48,   /*!< 4096x2160p @ 48Hz */
   VMIP_VIDEO_STANDARD_4096X2160P50,   /*!< 4096x2160p @ 50Hz */
   VMIP_VIDEO_STANDARD_4096X2160P59,   /*!< 4096x2160p @ 59Hz */
   VMIP_VIDEO_STANDARD_4096X2160P60,   /*!< 4096x2160p @ 60Hz */
   VMIP_VIDEO_STANDARD_1280X720P23,    /*!< 1280x720p @ 23Hz */
   VMIP_VIDEO_STANDARD_1280X720P24,    /*!< 1280x720p @ 24Hz */
   VMIP_VIDEO_STANDARD_1280X720P25,    /*!< 1280x720p @ 25Hz */
   VMIP_VIDEO_STANDARD_1280X720P29,    /*!< 1280x720p @ 29Hz */
   VMIP_VIDEO_STANDARD_1280X720P30,    /*!< 1280x720p @ 30Hz */
   NB_VMIP_VIDEO_STANDARD
} VMIP_VIDEO_STANDARD;

/*!
   @brief VideoMaster supported video bit depth.

   The VMIP_VIDEO_BIT_DEPTH enumeration lists all the video supported bit depth.

   @see VMIP_STREAM_ESSENCE_CONFIG_ST2110_20 VMIP_STREAM_ESSENCE_CONFIG_ST2110_22
*/
typedef enum _VMIP_VIDEO_BIT_DEPTH
{
   VMIP_VIDEO_DEPTH_8BIT,     /*!< 8-bit sample depth */
   VMIP_VIDEO_DEPTH_10BIT,    /*!< 10-bit sample depth */
   VMIP_VIDEO_DEPTH_12BIT,    /*!< 12-bit sample depth */
   NB_VMIP_VIDEO_DEPTH
} VMIP_VIDEO_BIT_DEPTH;

/*!
   @brief VideoMaster supported video bit sampling type.

   The VMIP_VIDEO_BIT_SAMPLING enumeration lists all the video supported bit sampling type.

   @see VMIP_STREAM_ESSENCE_CONFIG_ST2110_20 VMIP_STREAM_ESSENCE_CONFIG_ST2110_22
*/
typedef enum _VMIP_VIDEO_BIT_SAMPLING
{
   VMIP_VIDEO_SAMPLING_YCBCR_422,      /*!< YCbCr 4:2:2 sampling */
   VMIP_VIDEO_SAMPLING_YCBCR_444,      /*!< YCbCr 4:4:4 sampling */
   VMIP_VIDEO_SAMPLING_ICTCP_422,      /*!< ICtCp 4:2:2 sampling */
   VMIP_VIDEO_SAMPLING_ICTCP_444,      /*!< ICtCp 4:4:4 sampling */
   VMIP_VIDEO_SAMPLING_RGB,            /*!< RGB (linear or non-linear) sampling */
   VMIP_VIDEO_SAMPLING_XYZ,            /*!< XYZ sampling */
   NB_VMIP_VIDEO_SAMPLING
} VMIP_VIDEO_BIT_SAMPLING;

/*!
   @brief VideoMaster supported video bit padding type.

   The VMIP_VIDEO_BIT_PADDING enumeration lists all the supported ST2110-20 bit padding type.

   @see VMIP_STREAM_ESSENCE_CONFIG_ST2110_20 VMIP_STREAM_ESSENCE_CONFIG_ST2110_22
*/
typedef enum _VMIP_VIDEO_BIT_PADDING
{
   VMIP_VIDEO_NO_PADDING,        /*!< No padding*/
   VMIP_VIDEO_16_BITS_PADDING,   /*!< Padding is made on 16 bits */
   VMIP_VIDEO_32_BITS_PADDING,   /*!< Padding is made on 32 bits */
   NB_VMIP_VIDEO_BIT_PADDING
} VMIP_VIDEO_BIT_PADDING;

/*!
   @brief VideoMaster supported endianness.

   The VMIP_VIDEO_ENDIANNESS enumeration lists all the supported ST2110-20 bit padding type.

   @see VMIP_STREAM_ESSENCE_CONFIG_ST2110_20 VMIP_STREAM_ESSENCE_CONFIG_ST2110_22
*/
typedef enum _VMIP_VIDEO_ENDIANNESS
{
   VMIP_VIDEO_BIG_ENDIAN,        /*!< Big Endian*/
   VMIP_VIDEO_LITTLE_ENDIAN,     /*!< Little Endian*/
   NB_VMIP_VIDEO_ENDIANNESS
} VMIP_VIDEO_ENDIANNESS;

/*!
   @brief VideoMaster supported video colorimetry.

   The VMIP_VIDEO_COLORIMETRY enumeration lists all the supported ST2110-20 colorimetry.

   @see VMIP_STREAM_ESSENCE_CONFIG_ST2110_20 VMIP_STREAM_ESSENCE_CONFIG_ST2110_22
*/
typedef enum _VMIP_VIDEO_COLORIMETRY
{
   VMIP_VIDEO_COLORIMETRY_BT601,         /*!< BT601 colorimetry.*/
   VMIP_VIDEO_COLORIMETRY_BT709,         /*!< BT709 colorimetry.*/
   VMIP_VIDEO_COLORIMETRY_BT2020,        /*!< BT2020 colorimetry.*/
   VMIP_VIDEO_COLORIMETRY_BT2100,        /*!< BT2100 colorimetry.*/
   VMIP_VIDEO_COLORIMETRY_ST2065_1,      /*!< ST2065-1 colorimetry.*/
   VMIP_VIDEO_COLORIMETRY_ST2065_3,      /*!< ST2065-3 colorimetry.*/
   VMIP_VIDEO_COLORIMETRY_UNSPECIFIED,   /*!< Unspecified colorimetry.*/
   VMIP_VIDEO_COLORIMETRY_XYZ,           /*!< XYZ colorimetry.*/
   NB_VMIP_VIDEO_COLORIMETRY
} VMIP_VIDEO_COLORIMETRY;

/*!
   @brief VideoMaster supported TCS.

   The VMIP_VIDEO_TCS enumeration lists all the supported Transport Characteristic System.

   @see VMIP_STREAM_ESSENCE_CONFIG_ST2110_20 VMIP_STREAM_ESSENCE_CONFIG_ST2110_22
*/
typedef enum _VMIP_VIDEO_TCS
{
   VMIP_VIDEO_TCS_SDR,           /*!< Standard Dynamic Range TCS. */
   VMIP_VIDEO_TCS_PQ,            /*!< Perceptual Quantization TCS. */
   VMIP_VIDEO_TCS_HLG,           /*!< Hybrid Log-Gamma TCS. */
   VMIP_VIDEO_TCS_LINEAR,        /*!< TCS for linear encoded floating-point samples. */
   VMIP_VIDEO_TCS_BT2100LINPQ,   /*!< Linear encoded floating point samples normalized from PQ. */
   VMIP_VIDEO_TCS_BT2100LINHLG,  /*!< Linear encoded floating point samples normalized from HLG. */
   VMIP_VIDEO_TCS_ST2065_1,      /*!< linear encoded floating-point samples as specified in SMPTE ST 2065-1. */
   VMIP_VIDEO_TCS_ST428_1,       /*!< TCS as definied in ST428-1. */
   VMIP_VIDEO_TCS_DENSITY,       /*!< TCS for density encoded samples, such as those defined in SMPTE ST 2065-3. */
   VMIP_VIDEO_TCS_UNSPECIFIED,   /*!< Unspecified TCS.*/
   VMIP_VIDEO_TCS_ST2115LOGS3,   /*!< TCS for high dynamic range video that utilize the "Camera Log S3" transfer characteristic specified in SMPTE ST 2115.*/
   NB_VMIP_VIDEO_TCS
} VMIP_VIDEO_TCS;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

   /*!
      @brief Gets information regarding a video standard

      Retrieves the information associated to the given video standard

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The video standard is not supported;
      - One of the pointer is null.

      @see VMIP_VIDEO_STANDARD
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetVideoStandardInfo(VMIP_VIDEO_STANDARD VidStd /*!< [in] Video standard to get information from*/
      , uint32_t* pFrameWidth                                                                                 /*!< [out] Pointer to a caller\-allocated variable receiving the frame Width*/
      , uint32_t* pFrameHeight                                                                                /*!< [out] Pointer to a caller\-allocated variable receiving the frame height*/
      , uint32_t* pFramerate                                                                                  /*!< [out]Pointer to a caller\-allocated variable receiving the frame rate*/
      , bool8_t* pInterlaced                                                                                  /*!< [out] Pointer to a caller\-allocated variable receiving the interlaced information*/
      , bool8_t* pIsUs                                                                                        /*!< [out] Pointer to a caller\-allocated variable telling if the video standard is US or not*/
   );

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS

#endif //_VIDEOMASTERIP_VIDEO_H_