/*!
   @file videomasterip_stream_sync.h
   @brief This file contains the VideoMasterIP API for the stream synchronizer.

   More details concerning the multi-streams synchronization can be found in the \ref stream-sync "documentation".

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/
#ifndef _VIDEOMASTERIP_ST2110_STREAMSYNC_
#define _VIDEOMASTERIP_ST2110_STREAMSYNC_

#include "videomasterip_stream.h"

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

   /*!
      @brief VideoMaster StreamSync handle creation
   
      This function creates a new object used to synchronize multiple streams. See the \ref stream_sync_handling "documentation" for more information.
   
      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream type is not allowed;
   
      @see VMIP_DestroyStreamSync, VMIP_STREAMTYPE, \ref stream_sync_handling
   
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_CreateStreamSync(VMIP_STREAMTYPE StreamType                   /*!< [in] Type of StreamSync to open, must be a value of the VMIP_STREAMTYPE enumeration*/
      , const uint64_t ResyncWindow                                                                       /*!< [in] Timeout to get the data for all the essences. (RX only) */
      , HANDLE* pStreamSyncHandle                                                                        /*!< [out] Pointer to a caller\-allocated variable receiving the StreamSync handle*/
   );
   
   /*!
      @brief VideoMaster StreamSync handle deletion
   
      This destroys the handle of a specified stream
   
      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The StreamSync handle is invalid.
   
      @see VMIP_CreateStream, \ref stream_sync_handling
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_DestroyStreamSync(const HANDLE StreamSyncHandle /*!< [in] Handle of the StreamSync to destroy */);
   
   
   
   /*!
      @brief VideoMaster StreamSync main stream configuration

      This allows the user to set the main stream of the StreamSync.
      Only one main stream can be set for a given StreamSync.

      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The StreamSync or the main stream handle is invalid;

      @see \ref stream_sync_handling
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StreamSync_SetMainStream(const HANDLE StreamSyncHandle         /*!< [in] Handle of the StreamSync */
      , const HANDLE MainStreamHandle                                                                    /*!< [in] Handle of the main stream */
   );
   
   
   /*!
      @brief VideoMaster StreamSync secondary stream configuration

      This allows the user to set a secondary stream of the StreamSync.
      Multiple secondary stream can be added to a given StreamSync by calling the function multiple times/

      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The StreamSync or the secondary stream handle is invalid;

      @see \ref stream_sync_handling
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StreamSync_AddSecondaryStream(const HANDLE StreamSyncHandle         /*!< [in] Handle of the StreamSync */
      , const HANDLE SecondaryStreamHandle                                                                    /*!< [in] Handle of a secondary stream */
   );

   /*!
      @brief VideoMaster StreamSync start

      This function starts the StreamSync. All associated streams will be started.

      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The StreamSync handle is invalid;
      - The configuration of the associated streams is not compatible (see \ref stream-sync-association);
      - The mofiication of the configuration of an associated stream failed (see \ref stream-sync-start);
      - Starting one of the associated streams fails.

      @see VMIP_StopStreamSync
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StartStreamSync(HANDLE StreamSyncHandle /*!< [in] The handle of the StreamSync to start*/ );


   /*!
      @brief VideoMaster StreamSync stop

      This function stops the StreamSync. All associated streams will be stopped.

      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The StreamSync handle is invalid;
      - Stopping one of the associated streams fails.

      @see VMIP_StartStreamSync
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StopStreamSync(HANDLE StreamSyncHandle /*!< [in] The handle of the StreamSync to stop*/);

   /*!
      @brief VideoMaster StreamSync lock query
   
      This function locks a slot on the Stream Synchronizer, and provides a handle to the locked slot.

      In **RX**, a slot on the main stream is locked and the corresponding timeframe is computed.
      Then, for each secondary stream, slots are locked (and eventually unlocked) until slots with corresponding timeframe are locked.

      In **TX**, a slot is locked for the main stream and each secondary stream.

      Every locked slot must be unlocked using VMIP_StreamSync_UnlockSlot

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - Locking the data-pipe fails for the main stream.

      @see VMIP_StreamSync_UnlockSlot
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StreamSync_LockSlot(HANDLE StreamSyncHandle /*!< [in] The handle of the StreamSync to stop*/
      , HANDLE* pSlotHandle                                                           /*!< [out] The handle of the locked slot*/
   );   
   
   /*!
      @brief VideoMaster StreamSync lock release

      This function unlocks a locked slot. Every locked slot must be unlocked using this function.

      All corresponding slots from the associated streams are unlocked

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The pointer is null;
      - Locking the data-pipe fails for one of the associated stream.

      @see VMIP_StreamSync_LockSlot
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StreamSync_UnlockSlot(HANDLE StreamSyncHandle /*!< [in] The handle of the StreamSync to stop*/
      , HANDLE SlotHandle                                                               /*!< [in] The handle of the locked slot*/
   );
   
   
   /*!
      @brief VideoMasterIP StreamSync slot buffer query

      This function provides pointer to the specified buffer from the given locked slot handle for an associated stream.

      In RX, if a secondary stream is too delayed/advanced compared to the main stream and the Resynchronization Window, no data will be retrieved.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - the StreamSync is invalid;
      - The EssenceStream handle is invalid or not associated with the StreamSync;
      - The slot handle is invalid;
      - The BufferType is invalid;
      - One of the pointer is null;
      - An error occurs when getting the data buffer;
      - Data could not be synchronized (RX only).

      @see VMIP_StreamSync_LockSlot
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StreamSync_GetSlotBuffer(const HANDLE StreamSyncHandle   /*!< [in] Handle of the StreamSync to operate on */
       , const HANDLE EssenceStreamHandle                                                          /*!< [in] Handle of the essence stream to operate on */
       , const HANDLE SlotHandle                                                                   /*!< [in] Handle of the slot to operate on */
       , const uint32_t BufferType                                                                 /*!< [in] Type of the buffer that will be retrieved */
       , uint8_t** ppDataBuffer                                                                    /*!< [out] Pointer to a caller\-allocated variable receiving the buffer pointer */
       , uint32_t* pDataBufferSize                                                                 /*!< [out] Pointer to a caller\-allocated variable receiving the buffer size */
   );
   
   /*!
      @brief VideoMasterIP StreamSync slot timestamps configuration
   
      __TX Only__
      This function sets the timestamps associated to the given slot for the given associated stream.
   
      _Note_ : This does not change the time at which packets will be emitted.
      This only changes the content of the RTP timestamps in the network packet.
   
      More information is available in the \ref slot-timestamp "documentation".
   
      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - the StreamSync is invalid;
      - The EssenceStream handle is invalid or not associated with the StreamSync;
      - The slot handle is invalid;
      - The BufferType is invalid;
      - One of the pointer is null.

      @see VMIP_TIMESTAMP
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StreamSync_SetSlotTimestamps(const HANDLE StreamSyncHandle   /*!< [in] Handle of the multi-stream to operate on */
      , const HANDLE EssenceStreamHandle                                                               /*!< [in] Handle of the essence stream to operate on */
      , const HANDLE SlotHandle                                                                        /*!< [in] Handle of the slot to operate on */
      , const uint32_t BufferType                                                                      /*!< [in] Type of the buffer that will be retrieved */
      , const VMIP_TIMESTAMP pTimestamp[]                                                              /*!< [in] Array containing the RTP timestamp information */
      , const uint32_t TimestampNb                                                                     /*!< [in] number of timestamp in the array */
   );
   
   /*!
      @brief VideoMasterIP StreamSync slot timestamps query
   
      This function retrieves the RTP timestamps associated to the given slot for the given associated stream.
   
      More information is available in the \ref slot-timestamp "documentation".
   
      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - the StreamSync is invalid;
      - The EssenceStream handle is invalid or not associated with the StreamSync;
      - The slot handle is invalid;
      - The BufferType is invalid;
      - One of the pointer is null;
      - Data could not be synchronized (RX only).

      @see VMIP_TIMESTAMP
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StreamSync_GetSlotTimestamps(const HANDLE StreamSyncHandle /*!< [in] Handle of the multi-stream to operate on  */
      , const HANDLE EssenceStreamHandle                                                             /*!< [in] Handle of the essence stream to operate on */
      , const HANDLE SlotHandle                                                                      /*!< [in] Handle of the slot to operate on */
      , const uint32_t BufferType                                                                    /*!< [in] Type of the buffer that will be retrieved */
      , VMIP_TIMESTAMP pTimestamp[]                                                                  /*!< [out] Pointer to a caller\-allocated array receiving the RTP timestamp information */
      , uint32_t* pTimestampNb                                                                       /*!< [in,out] Pointer to a caller\-allocated variable.
                                                                                                                   As input, this gives the size of the ppTimestamp_X array.
                                                                                                                   As output, it receives the number of timestamp retrieved */
   );
   
   /*!
      @brief VideoMasterIP StreamSync slot lost data information query
   
      This function retrieves the lost data information associated to the given slot for the given associated stream.
   
      More information is available in the \ref slot-lost-data "documentation".
   
      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - the StreamSync is invalid;
      - The EssenceStream handle is invalid or not associated with the StreamSync;
      - The slot handle is invalid;
      - The BufferType is invalid;
      - One of the pointer is null;
      - Data could not be synchronized (RX only).

      @see VMIP_LOST_DATA
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StreamSync_GetSlotLostDataInfo(const HANDLE StreamSyncHandle /*!< [in] Handle of the multi-stream to operate on  */
      , const HANDLE EssenceStreamHandle                                                               /*!< [in] Handle of the essence stream to operate on */
      , const HANDLE SlotHandle                                                                        /*!< [in] Handle of the slot to operate on */
      , const uint32_t BufferType                                                                      /*!< [in] Type of the buffer that will be retrieved */
      , VMIP_LOST_DATA pLostData[]                                                                     /*!< [out] Pointer to a caller\-allocated array receiving the lost data information */
      , uint32_t* pLostDataNb                                                                          /*!< [in,out] Pointer to a caller\-allocated variable.
                                                                                                                     As input, this gives the size of the pLostData_X array.
                                                                                                                     As output, it receives the number of lost data segments retrieved */
   );
   
   /*!
      @brief VideoMasterIP ANC slot retrieval
   
      This function retrieves the dedicated ANC slot for a associated ANC stream.
      This slot can then be uased to manupalate the ANC data using functions defined in videomasterip_anc.h and videomasterip_anc_processing.h.
   
      More information is available in the \ref ANC_data_handling "documentation".
   
      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - the StreamSync is invalid;
      - The EssenceStream handle is invalid or not associated with the StreamSync;
      - The EssenceStream handle correspond to a stream which is not an ANC stream;
      - Data could not be synchronized (RX only).

      @see \ref ANC_data_handling
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StreamSync_GetAncSlot(const HANDLE StreamSyncHandle /*!< [in] Handle of the StreamSync to operate on  */
      , const HANDLE EssenceStreamHandle                                                      /*!< [in] Handle of the essence stream to operate on */
      , const HANDLE SyncSlotHandle                                                           /*!< [in] Handle of the slot to operate on */
      , HANDLE* pAncSlotHandle                                                                /*!< [out] The handle of the corresponding ANC slot*/
   );

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS

#endif // _VIDEOMASTERIP_ST2110_STREAMSYNC_
