/*!
   @file
   @brief This file gathers stream and slot related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/

#ifndef _VIDEOMASTERIP_ST2110_STREAM_
#define _VIDEOMASTERIP_ST2110_STREAM_

#include "videomasterip_core.h"
#include "videomasterip_st2110_20.h"
#include "videomasterip_st2110_22.h"
#include "videomasterip_st2110_30.h"
#include "videomasterip_st2110_40.h"

/*!
   @brief VideoMaster supported essence type

   The VMIP_ESSENCETYPE enumeration lists all the supported essence type.

   @see VMIP_CreateStream
*/
typedef enum _VMIP_ESSENCETYPE
{
   VMIP_ET_ST2110_20,  /*!< Essence type corresponding to a ST2110-20 stream. */
   VMIP_ET_ST2110_30,  /*!< Essence type corresponding to a ST2110-30 stream. */
   VMIP_ET_ST2110_40,  /*!< Essence type corresponding to a ST2110-40 stream. */
   VMIP_ET_ST2110_22,  /*!< Essence type corresponding to a ST2110-22 stream. */
   NB_VMIP_ESSENCETYPE
}VMIP_ESSENCETYPE;

/*!
   @brief VideoMaster supported stream type

   The VMIP_STREAMTYPE enumeration lists all the supported stream type.

   @see VMIP_CreateStream
*/
typedef enum _VMIP_STREAMTYPE
{
   VMIP_ST_RX,  /*!< RX stream type. */
   VMIP_ST_TX,  /*!< TX stream type. */
   NB_VMIP_STREAMTYPE
}VMIP_STREAMTYPE;

/*!
   @brief VideoMaster supported IGMP filtering type

   The VMIP_IGMP_FILTERING_TYPE enumeration lists all the IGMP filtering type.

   @see VMIP_STREAM_NETWORK_CONFIG
*/
typedef enum _VMIP_IGMP_FILTERING_TYPE
{
   VMIP_FILTERING_WHITELIST,  /*!< Whitelist filtering type. */
   VMIP_FILTERING_BLACKLIST,  /*!< Blacklist filtering type. */
   NB_VMIP_FILTERING_TYPE
}VMIP_IGMP_FILTERING_TYPE;

/*!
   @brief Timestamp information structure

   The VMIP_TIMESTAMP structure abstracts a timestamp and its position.

   @see VMIP_GetSlotTimestamps and VMIP_SetSlotTimestamps
*/
typedef struct _VMIP_TIMESTAMP
{
   uint32_t SamplePosition;/*!< Position of the sample referenced by the timestamp. (null if there is only one element in the slot). */
   uint32_t TimeStamp;     /*!< Value of the timestamp */
}VMIP_TIMESTAMP;

/*!
   @brief Lost data information structure

   The VMIP_LOST_DATA structure abstracts information regarding lost data.

   @see VMIP_GetSlotLostDataInfo
*/
typedef struct _VMIP_LOST_DATA
{
   uint32_t SamplePosition;/*!< Position of the lost sample. */
   uint32_t LostDataSize;  /*!< Size of the lost data (in bytes). */
}VMIP_LOST_DATA;

/*!
   @brief VideoMasterIP stream common configuration.

   The VMIP_STREAM_COMMON_CONFIG structure contains information regarding the stream common configuration.

   @see VMIP_SetStreamCommonConfig and VMIP_GetStreamCommonConfig
*/
typedef struct _VMIP_STREAM_COMMON_CONFIG
{
   uint32_t ApplicativeBufferQueueDepth;                      /*!< Stream buffers queue depth, in number of slots (minimum is 2, maximum is 32, default is 4) (RX/TX)*/
   uint32_t Timeout;                                          /*!< Slot locking time-out value, in milliseconds (default is 100 msec) (RX/TX)*/
   uint64_t pCpuCoreIdxForProcessing[MAX_HANDLE_ARRAY_SIZE];  /*!< Array containing the Id's of the CPU core on which the processing will be done for the stream. (RX/TX)  
                                                                   For more information, please refer to the \ref processing page in the documentation. */
   uint64_t NbCpuCoreForProcessing;                           /*!< Number of core that will be used for the processing.  
                                                                   If this is set to 0, then all the available cores (meaning, cores non-excluded via the VCS configuration file and not used for conductors) will be used for processing. (RX/TX)*/
   uint64_t ConductorId;                                      /*!< Id of the conductor that will be used by the stream. (RX/TX)*/
   uint64_t ManagementThreadCpuCoreId;                        /*!< Id of the CPU core that will be used by the stream management thread. (RX/TX).  
                                                                   If this is set to 0, then the management thread affinity is the same as VCS affinity.  
                                                                   For more information, please refer to the \ref stream-management page in the documentation.  */
}VMIP_STREAM_COMMON_CONFIG;

/*!
   @brief VideoMasterIP path parameters.

   The VMIP_PATH_PARAMETERS structure contains information regarding an input/output port.

   @see VMIP_STREAM_NETWORK_CONFIG
*/
typedef struct _VMIP_PATH_PARAMETERS
{
   uint16_t UdpPort;                                 /*!< The Port the packets are addressed to.   
                                                               For RX streams, this is the local port that will be opened.  
                                                               For TX streams, this is the port the packets must be received at destination. */
   uint32_t SourceIp;                                /*!< The IP address that is set as source of the packets.  
                                                               For RX streams, this corresponds to the address the packets are sent from, only unicast or 0 if unknown.  
                                                               Regarding stream configuration, this field is only used in DPDK for hardware filtering and is applicable only on RX stream and will be ignored on TX stream and non DPDK streams.  
                                                               It is also used when reading/writing SDP as the source address of the 'origin' field.*/
   uint32_t DestinationIp;                           /*!< The IP address that is set as destination of the packets.  
                                                               For RX streams, this corresponds to the address the packets are received on, can be unicast, multicast, or 0 if unknown. If the address is multicast, the group will be joined.  
                                                               For TX streams, this corresponds to the address the packets are emitted to, can be unicast or multicast but cannot be 0. */
   VMIP_IGMP_FILTERING_TYPE IgmpFilteringType;       /*!< The mode of the multicast filtering if destination IP is multicast. (RX only)*/
   uint32_t pIgmpSourceList[MAX_HANDLE_ARRAY_SIZE];  /*!< The sources that must be excluded or included depending on the multicast mode. To accept all sources, place mode on exclusion and don't add any source in this field. (RX only)*/
   uint32_t IgmpSourceListSize;                      /*!< Number of Igmp sources present in the pIgmpSourceList. (RX only)*/
   uint64_t InterfaceId;                             /*!< The id of the network Interface the port must be opened on; can be 0 if destination is not multicast, the system then chooses the best interface based on usual mechanisms.*/
   uint8_t IpTtl_u8;                                 /*!< The Time To Live of the IP packets. If set to 0, 64 will be set as default value, as recommended by the RFC1700. (TX only)*/
}VMIP_PATH_PARAMETERS;

/*!
   @brief VideoMasterIP stream network configuration.

   The VMIP_STREAM_NETWORK_CONFIG structure contains information regarding the stream network configuration.

   @see VMIP_SetStreamNetworkConfig and VMIP_GetStreamNetworkConfig
*/
typedef struct _VMIP_STREAM_NETWORK_CONFIG
{
   uint32_t SocketBufferSize;                                    /*!< Size of the socket buffer. If 0 is used, a default value will be used by VCS.*/
   uint32_t Ssrc;                                                /*!< The value of the SSRC field in the RTP header.  
                                                                           In RX streams, if this field is not 0, the packets are filtered so that only matching SSRC packets are considered belonging to the stream.  */
   uint8_t PayloadType;                                          /*!< The value of the payload type RTP-header field of this stream.  
                                                                           In RX streams, if this field is not 0, the packets are filtered so that only matching payload type packets are considered belonging to the stream.  
                                                                           In TX, the value chosen will be present in the RTP packets and its value should be set according to industry standards. */
   VMIP_PATH_PARAMETERS pPathParameters[MAX_HANDLE_ARRAY_SIZE];  /*!< The parameters of the different ports that will be opened. Using multiples ports allows seamless switching in RX and TX as specified in the norm ST2022-7.*/
   uint32_t PathParameterSize;                                   /*!< Number of port parameters present in pPortParameters.*/
}VMIP_STREAM_NETWORK_CONFIG;


/*!
   @brief VideoMasterIP stream essence configuration.

   The VMIP_STREAM_ESSENCE_CONFIG structure contains information regarding the stream essence configuration.

   @see VMIP_SetStreamEssenceConfig and VMIP_GetStreamEssenceConfig
*/
typedef struct _VMIP_STREAM_ESSENCE_CONFIG
{
   VMIP_ESSENCETYPE EssenceType;                                    /*!< Essence type. This is used to know which of the following field is used.*/
   bool HasStartTime;                                               /*!< If the stream has a start time.*/
   uint64_t StartTime;                                              /*!< The start time of the stream. This time is expressed in nanoseconds since the epoch time of the clock TAI (TX only)*/
   uint32_t TaiUtcDelta;                                            /*!< Delta between the UTC clock (used by the system time) and the TAI clock. This delta is expressed in seconds (Read-only).
                                                                         The value comes from the VCS configuration file (vcsconfig.xml). See <http://leapsecond.com/java/gpsclock.htm> */
   union
   {
      VMIP_STREAM_ESSENCE_CONFIG_ST2110_20 EssenceS2110_20Prop;     /*!< Configuration of a ST2110-20 essence.*/
      VMIP_STREAM_ESSENCE_CONFIG_ST2110_22 EssenceS2110_22Prop;     /*!< Configuration of a ST2110-22 essence.*/
      VMIP_STREAM_ESSENCE_CONFIG_ST2110_30 EssenceS2110_30Prop;     /*!< Configuration of a ST2110-30 essence.*/
      VMIP_STREAM_ESSENCE_CONFIG_ST2110_40 EssenceS2110_40Prop;     /*!< Configuration of a ST2110-40 essence.*/
   };
}VMIP_STREAM_ESSENCE_CONFIG;

/*!
   @brief VideoMasterIP stream common status.

   The VMIP_STREAM_COMMON_STATUS structure contains information regarding the stream common status.

   @see VMIP_GetStreamCommonStatus
*/
typedef struct _VMIP_STREAM_COMMON_STATUS
{
   uint32_t ApplicativeBufferQueueFilling; /*!< Current filling level of stream buffer queue, in number of slots. */
   uint32_t SlotCount;                     /*!< Counts the number of slots transferred since the stream has been started. */
   uint32_t SlotDropped;                   /*!< DEPRECATED. Used VMIP_STREAM_NETWORK_STATUS::SlotDropped instead */
}VMIP_STREAM_COMMON_STATUS;

/*!
   @brief VideoMasterIP stream network status.

   The VMIP_STREAM_NETWORK_STATUS structure contains information regarding the stream network status.

   @see VMIP_GetStreamNetworkStatus
*/
typedef struct _VMIP_STREAM_NETWORK_STATUS
{
   uint32_t Ssrc;                   /*!< Detected SSRC associated to the current stream. (RX only)*/
   uint8_t PayloadType;             /*!< Detected payload type associated to the current stream. (RX only)*/
   uint64_t PacketCount;            /*!< In RX, number of packets that have been received on the NIC. In TX, number of packets that have been transmitted to the NIC*/
   uint64_t ProcessedPacketCount;   /*!< Number of packets that have been processed. */
   uint64_t PacketUnderrun;         /*!< The number of packets that should have been sent but that were not because no data were available (TX only). */
   uint64_t PacketDrop;             /*!< The number of packets dropped before sending because sending accumulated too much delay. (TX only) */
   uint64_t PacketLost;             /*!< The number of packets that were not received on the network. (RX only)*/
   uint64_t FrameUnderrun;          /*!< The number of frames that should have been sent but that were not because no data were available (TX only). */
   uint32_t SlotDropped;            /*!< Counts the number of slots dropped since the stream has been started (RX only).*/
}VMIP_STREAM_NETWORK_STATUS;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

   /*!
      @brief VideoMaster stream handle creation

      This function creates a new stream linked to a specified VCS service.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The VCS handle is not coherent;
      - The stream type is not allowed;
      - The essence type is not allowed;
      - The stream handle pointer is null;
      - The stream creation fails.

      @see VMIP_DestroyStream, VMIP_STREAMTYPE and VMIP_ESSENCETYPE
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_CreateStream(const HANDLE VcsContext /*!< [i,] Handle of the VCS service to open a stream handle on*/
      , VMIP_STREAMTYPE StreamType                                             /*!< [in] Type of stream to open, must be a value of the VMIP_STREAMTYPE enumeration*/
      , VMIP_ESSENCETYPE EssenceType                                           /*!< [in] Type of the essence used in the stream, must be a value of the VMIP_ESSENCETYPE enumeration*/
      , HANDLE* pStreamHandle                                                  /*!< [out] Pointer to a caller\-allocated variable receiving the stream handle*/
   );

   /*!
      @brief VideoMaster stream handle deletion

      This destroys the handle of a specified stream

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent.

      @see VMIP_CreateStream
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_DestroyStream(const HANDLE StreamHandle /*!< [in] Handle of the stream to destroy */);

   /*!
      @brief VideoMasterIP stream common configuration

      This function configures the generic properties of a stream.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The value of a property is not allowed.

      @see VMIP_STREAM_ESSENCE_CONFIG and VMIP_GetStreamCommonConfig
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SetStreamCommonConfig(const HANDLE StreamHandle /*!< [in] Handle of the stream to set property on */
      , VMIP_STREAM_COMMON_CONFIG CommonConfig                                            /*!< [in] User-filled structure VMIP_STREAM_ESSENCE_CONFIG */
   );

   /*!
      @brief VideoMasterIP stream network configuration

      This function configures the network properties of a stream.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The value of a property is not allowed.

      @see VMIP_STREAM_NETWORK_CONFIG and VMIP_GetStreamNetworkConfig
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SetStreamNetworkConfig(const HANDLE StreamHandle /*!< [in] Handle of the stream to set property on */
      , VMIP_STREAM_NETWORK_CONFIG NetworkConfig                                           /*!< [in] User-filled structure VMIP_STREAM_NETWORK_CONFIG */
   );

   /*!
      @brief VideoMasterIP stream essence configuration

      This function configures the essence-related properties of a stream.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The essence type is not coherent with the stream configuration;
      - The value of a property is not allowed.

      @see VMIP_STREAM_ESSENCE_CONFIG and VMIP_GetStreamEssenceConfig
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SetStreamEssenceConfig(const HANDLE StreamHandle /*!< [in] Handle of the stream to set property on */
      , VMIP_STREAM_ESSENCE_CONFIG EssenceConfig                                           /*!< [in] User-filled structure VMIP_STREAM_ESSENCE_CONFIG containing the type of essence being configured and its configuration. */
   );

   /*!
      @brief VideoMasterIP stream common configuration query

      This function retrieves the common configuration of the specified stream

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The pointer is null.

      @see VMIP_STREAM_COMMON_CONFIG and VMIP_SetStreamCommonConfig
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetStreamCommonConfig(const HANDLE StreamHandle /*!< [in]  Handle of the stream to get property from */
      , VMIP_STREAM_COMMON_CONFIG* pCommonConfig                                          /*!< [out] Pointer to caller\-allocated variable to return the common config */
   );

   /*!
      @brief VideoMasterIP stream common status query

      This function retrieves the common status of the specified stream

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The pointer is null.

      @see VMIP_STREAM_COMMON_STATUS
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetStreamCommonStatus(const HANDLE StreamHandle /*!< [in]  Handle of the stream to get property from */
      , VMIP_STREAM_COMMON_STATUS* pCommonStatus                                          /*!< [out] Pointer to caller\-allocated variable to return the common status */
   );

   /*!
      @brief VideoMasterIP stream network configuration query

      This function retrieves the network configuration of the specified stream

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The pointer is null.

      @see VMIP_STREAM_NETWORK_CONFIG and VMIP_SetStreamNetworkConfig
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetStreamNetworkConfig(const HANDLE StreamHandle /*!< [in]  Handle of the stream to get property from */
      , VMIP_STREAM_NETWORK_CONFIG* pNetworkConfig                                         /*!< [out] Pointer to caller\-allocated variable to return the network config */
   );

   /*!
      @brief VideoMasterIP stream network status query

      This function retrieves the network status of the specified stream

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The pointer is null.

      @see VMIP_STREAM_NETWORK_STATUS
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetStreamNetworkStatus(const HANDLE StreamHandle /*!< [in]  Handle of the stream to get property from */
      , VMIP_STREAM_NETWORK_STATUS* pNetworkStatus                                         /*!< [out] Pointer to caller\-allocated variable to return the network status */
   );

   /*!
      @brief VideoMasterIP stream essence configuration query

      This function retrieves the essence configuration of the specified stream

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The pointer is null.

      @see VMIP_STREAM_ESSENCE_CONFIG and VMIP_SetStreamEssenceConfig
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetStreamEssenceConfig(const HANDLE StreamHandle /*!< [in]  Handle of the stream to get property from */
      , VMIP_STREAM_ESSENCE_CONFIG* pEssenceConfig                                         /*!< [out] Pointer to caller\-allocated variable to return the essence config */
   );

   /*!
      @brief FUTURE USE

      This function should not be used in the current VMIP version.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_OpenStream(const HANDLE StreamHandle /*!< [in] Handle of the stream to be opened */);

   /*!
      @brief FUTURE USE

      This function should not be used in the current VMIP version.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_CloseStream(const HANDLE StreamHandle /*!< [in] Handle of the stream to be closed */);

   /*!
      @brief VideoMasterIP stream start

      This function starts the specified stream

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The DataPipe opening fails.

      @see VMIP_StopStream and VMIP_OpenStream
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StartStream(const HANDLE StreamHandle /*!< [in] Handle of the stream to start */);

   /*!
      @brief VideoMaster stream stop

      This function stops the specified stream if it is running

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - The data-pipe closure fails.

      @see VMIP_StartStream and VMIP_CloseStream
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StopStream(const HANDLE StreamHandle /*!< [in] Handle of the stream to stop */);

   /*!
      @brief VideoMasterIP slot query

      This function locks a slot on the stream, and provides a
      handle to the locked slot.

      If no slot is ready, the caller is put asleep until such a
      slot is available, or until the time-out value specified in
      the VMIP_STREAM_COMMON_CONFIG is reached.

      Every locked slot must be unlocked using VMIP_UnlockSlot

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - Locking the data-pipe fails.

      @see VMIP_UnlockSlot
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_LockSlot(const HANDLE StreamHandle /*!< [in] Handle of the stream to operate on */
      , HANDLE* pSlotHandle_H                                                /*!< [out] Pointer to a caller\-allocated variable receiving the slot handle */
   );

   /*!
      @brief VideoMasterIP slot release

      This function unlocks a locked slot. Every locked slot must be unlocked using this function.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The stream handle is not coherent;
      - Unlocking the data-pipe fails.

      @see VMIP_LockSlot
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_UnlockSlot(const HANDLE StreamHandle /*!< [in] Handle of the stream to operate on */
      , const HANDLE SlotHandle                                                /*!< [in] Handle of the slot to operate on */
   );

   /*!
      @brief VideoMasterIP slot buffer query

      This function provides pointer to the specified buffer from the given locked slot handle

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The slot handle is not coherent;
      - One of the pointer is null;
      - An error occures when getting the data buffer.

      @see VMIP_LockSlot
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetSlotBuffer(const HANDLE StreamHandle /*!< [in] Handle of the stream to operate on */
      , const HANDLE SlotHandle                                                   /*!< [in] Handle of the slot to operate on */
      , const uint32_t BufferType                                                 /*!< [in] Type of the buffer that will be retrieved */
      , uint8_t** ppDataBuffer                                                    /*!< [out] Pointer to a caller\-allocated variable receiving the buffer pointer */
      , uint32_t* pDataBufferSize                                                 /*!< [out] Pointer to a caller\-allocated variable receiving the buffer size */
   );

   /*!
      @brief VideoMasterIP slot timestamps configuration

      __TX Only__
      This function sets the timestamps associated to the given slot.

      _Note_ : This does not change the time at which packets will be emitted.
      This only changes the content of the RTP timestamps in the network packet.

      More information is available in the \ref slot-timestamp "documentation".

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The slot handle is not coherent;

      @see VMIP_TIMESTAMP
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SetSlotTimestamps(const HANDLE StreamHandle /*!< [in] Handle of the stream to operate on  */
      , const HANDLE SlotHandle                                                       /*!< [in] Handle of the slot to operate on */
      , const uint32_t BufferType                                                     /*!< [in] Type of the buffer that will be retrieved */
      , const VMIP_TIMESTAMP pTimestamp[]                                             /*!< [in] Array containing the RTP timestamp information */
      , const uint32_t TimestampNb                                                    /*!< [in] number of timestamp in the array */
   );

   /*!
      @brief VideoMasterIP slot timestamps query

      This function retrieves the RTP timestamps associated to the given slot.

      More information is available in the \ref slot-timestamp "documentation".

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The slot handle is not coherent;
      - One of the pointer is null.

      @see VMIP_TIMESTAMP
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetSlotTimestamps(const HANDLE StreamHandle /*!< [in] Handle of the stream to operate on  */
      , const HANDLE SlotHandle                                                       /*!< [in] Handle of the slot to operate on */
      , const uint32_t BufferType                                                     /*!< [in] Type of the buffer that will be retrieved */
      , VMIP_TIMESTAMP pTimestamp[]                                                   /*!< [out] Pointer to a caller\-allocated array receiving the RTP timestamp information */
      , uint32_t* pTimestampNb                                                        /*!< [in,out] Pointer to a caller\-allocated variable.  
                                                                                                    As input, this gives the size of the ppTimestamp_X array.  
                                                                                                    As output, it receives the number of timestamp retrieved */
   );

   /*!
      @brief VideoMasterIP slot lost data information query

      This function retrieves the lost data information associated to the given slot.

      More information is available in the \ref slot-lost-data "documentation".

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The slot handle is not coherent;
      - One of the pointer is null.

      @see VMIP_LOST_DATA
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetSlotLostDataInfo(const HANDLE StreamHandle /*!< [in] Handle of the stream to operate on  */
      , const HANDLE SlotHandle                                                         /*!< [in] Handle of the slot to operate on */
      , const uint32_t BufferType                                                       /*!< [in] Type of the buffer that will be retrieved */
      , VMIP_LOST_DATA pLostData[]                                                      /*!< [out] Pointer to a caller\-allocated array receiving the lost data information */
      , uint32_t* pLostDataNb                                                           /*!< [in,out] Pointer to a caller\-allocated variable.  
                                                                                                      As input, this gives the size of the pLostData_X array.  
                                                                                                      As output, it receives the number of lost data segments retrieved */
   );

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS

#endif //_VIDEOMASTERIP_ST2110_STREAM_