/*!
   @file
   @brief This file gathers ST2110-20 essence related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/

#ifndef _VIDEOMASTERIP_ST2110_20_H_
#define _VIDEOMASTERIP_ST2110_20_H_

#include "videomasterip_video.h"

#define ESSENCE_ST2110_20_ENUMBASE 0x01000000  /*!< Enumeration base value for ST2110-20 related enumerations */

/*!
   @brief VideoMaster supported ST2110-20 packing mode.

   The VMIP_ST2110_20_PACKING_MODE enumeration lists all the supported ST2110-20 packing.

   @see VMIP_STREAM_ESSENCE_CONFIG_ST2110_20
*/
typedef enum _VMIP_ST2110_20_PACKING_MODE
{
   VMIP_ST2110_20_PM_GENERAL,     /*!< General packing mode */
   NB_VMIP_ST2110_20_PACKING_MODE
} VMIP_ST2110_20_PACKING_MODE;

/*!
   @brief VideoMaster supported ST2110-20 buffer type.

   The VMIP_ST2110_20_BUFFERTYPE enumeration lists all the supported ST2110-20 buffer type.

   @see VMIP_GetSlotBuffer
*/
typedef enum _VMIP_ST2110_20_BUFFERTYPE
{
   VMIP_ST2110_20_BT_VIDEO = ESSENCE_ST2110_20_ENUMBASE, /*!< Video buffer type.*/
   NB_VMIP_ST2110_20_BUFFERTYPE
} VMIP_ST2110_20_BUFFERTYPE;

/*!
   @brief VideoMasterIP streams properties linked to a ST2110-20 essence.

   The VMIP_STREAM_ESSENCE_CONFIG_ST2110_20 structure contains all the available parameters for ST2110-20 stream configuration.

   @see VMIP_GetStreamEssenceConfig and VMIP_SetStreamEssenceConfig
*/
typedef struct _VMIP_STREAM_ESSENCE_CONFIG_ST2110_20
{
   VMIP_VIDEO_STANDARD VideoStandard;       /*!< Video standard. Default is VMIP_VIDEO_STANDARD_1920X1080P30. */
   VMIP_VIDEO_BIT_DEPTH NetworkBitDepth;          /*!< Bit depth on the network side. Default is VMIP_VIDEO_DEPTH_8BIT. */
   VMIP_VIDEO_BIT_SAMPLING NetworkBitSampling;    /*!< Bit sampling on the network side. Default is VMIP_VIDEO_SAMPLING_YUV_422. */
   VMIP_ST2110_20_PACKING_MODE PackingMode;           /*!< Packing mode. Only meaningful in TX. Default is VMIP_ST2110_20_PM_GENERAL. */
   VMIP_VIDEO_COLORIMETRY Colorimetry;            /*!< Colorimetry. Default is VMIP_VIDEO_COLORIMETRY_BT709. */
   VMIP_VIDEO_BIT_DEPTH UserBitDepth;             /*!< Bit depth on the applicative side. Default is VMIP_VIDEO_DEPTH_8BIT. */
   VMIP_VIDEO_BIT_SAMPLING UserBitSampling;       /*!< Bit sampling on the applicative side. Default is VMIP_VIDEO_SAMPLING_YUV_422. */
   VMIP_VIDEO_BIT_PADDING UserBitPadding;         /*!< Bit padding on the applicative side. Default is VMIP_VIDEO_NO_PADDING. */
   VMIP_VIDEO_ENDIANNESS UserEndianness;          /*!< Endianness on the applicative side. Default is VMIP_VIDEO_BIG_ENDIAN. */
   bool8_t UseDefaultTrOffset;                    /*!< Indicates if the default TR offset is used. Default value is defined by ST2110-21. Default is true.
                                                       If not, value must be indicated in the TrOffset field. */
   uint32_t TrOffset;                             /*!< TR Offset value (in s). It is only used if UseDefaultTrOffset is false. */
   VMIP_VIDEO_TCS Tcs;                            /*!< TCS. This field is only use when reading/writing SDP (see VMIP_ReadSDP and VMIP_WriteSDP). */
   bool8_t UseDefaultVrxCapacity;                 /*!< Indicates if the default capacity for the VRX bucket is used. Default value is defined by ST2110-21. Default is true.
                                                       If not, value must be indicated in the VrxCapacity field. */
   uint32_t VrxCapacity;                           /*!< VRX bucket capacity (in number of packet). It is only used if UseDefaultVrxCapacity is false. */
}VMIP_STREAM_ESSENCE_CONFIG_ST2110_20;

#endif //_VIDEOMASTERIP_ST2110_20_H_