/*!
   @file
   @brief This file gathers SDP related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/


#ifndef _VIDEOMASTERIP_SDP_H_
#define _VIDEOMASTERIP_SDP_H_

#include "videomasterip_core.h"
#include "videomasterip_stream.h"
#include "videomasterip_st2110_21.h"

#define MAX_CHAR_AUDIO_CHANNEL_ORDER_SIZE 256    /*!< Maximum number of char in VMIP_SDP_ADDITIONAL_INFORMATION.pAudioChannelOrder */

/*!
   @brief VideoMaster supported essence type

   The VMIP_REFERENCE_CLOCK_TYPE enumeration lists all the reference clock type.

   @see VMIP_WriteSDP VMIP_ReadSDP VMIP_REFERENCE_CLOCK
*/
typedef enum _VMIP_REFERENCE_CLOCK_TYPE
{
   VMIP_REF_CLOCK_PTP_GRANDMASTER_ID, /*!< Reference clock type for a PTP grandmaster. */
   VMIP_REF_CLOCK_PTP_IS_TRACEABLE,   /*!< Reference clock type for a PTP which is traceable. */
   VMIP_REF_CLOCK_LOCAL_MAC,          /*!< Reference clock type for a local NIC, identified by its MAC address. */
   NB_VMIP_REFERENCE_CLOCK_TYPE
}VMIP_REFERENCE_CLOCK_TYPE;

/*!
   @brief VideoMaster supported media clock type

   The VMIP_MEDIA_CLOCK_TYPE enumeration lists all the media clock type as defined in ST2110-10:2021.

   @see VMIP_WriteSDP VMIP_ReadSDP
*/
typedef enum _VMIP_MEDIA_CLOCK_TYPE
{
   VMIP_MEDIA_CLOCK_DIRECT,           /*!< Direct media clock. */
   VMIP_MEDIA_CLOCK_SENDER,           /*!< Sender media clock. */
   NB_VMIP_MEDIA_CLOCK_TYPE
}VMIP_MEDIA_CLOCK_TYPE;


/*!
   @brief VideoMaster supported RTP timestamp mode

   The VMIP_TIMESTAMP_MODE enumeration lists all the RTP timestamping mode as defined in ST2110-10:2021.

   @see VMIP_WriteSDP VMIP_ReadSDP
*/
typedef enum _VMIP_TIMESTAMP_MODE
{
   VMIP_TS_MODE_SAMPLING,       /*!< RTP timestamp indicates the effective sampling instant of the media. */
   VMIP_TS_MODE_NEW,            /*!< RTP timestamp has been created anew. */
   VMIP_TS_MODE_PRESERVED,      /*!< RTP timestamp has been preserved from an input signal. */
   NB_VMIP_TIMESTAMP_MODE       /*!< Number of value in the enumeration. */
}VMIP_TIMESTAMP_MODE;


/*!
   @brief PTP grandmaster structure

   The PTP_GRANDMASTER structure contains parameters regarding the configuration of a PTP grandmaster.

   @see VMIP_REFERENCE_CLOCK_TYPE
*/
typedef struct _PTP_GRANDMASTER
{
   uint8_t pId[8];  /*!< Identity of the grandmaster clock. */
   uint8_t Domain;  /*!< Domain used by the grandmaster clock. */
}PTP_GRANDMASTER;



/*!
   @brief Reference clock description

   The VMIP_REFERENCE_CLOCK structure information regarding the reference clock.

   @see VMIP_WriteSDP VMIP_ReadSDP VMIP_SDP_ADDITIONAL_INFORMATION
*/
typedef struct _VMIP_REFERENCE_CLOCK
{
   VMIP_REFERENCE_CLOCK_TYPE ReferenceClockType; /*!< Type of reference clock. */
   union
   {
      PTP_GRANDMASTER Grandmaster;               /*!< Structure containing grandmaster information. Only used if ReferenceClockType is VMIP_REF_CLOCK_PTP_GRANDMASTER_ID. */
      uint8_t pLocalMac[6];                      /*!< Mac of the local NIC used. Only used if ReferenceClockType is VMIP_REF_CLOCK_LOCAL_MAC*/
   };
}VMIP_REFERENCE_CLOCK;

/*!
   @brief Reference clock description

   The VMIP_REFERENCE_CLOCK structure information regarding the reference clock.

   @see VMIP_WriteSDP VMIP_ReadSDP VMIP_SDP_ADDITIONAL_INFORMATION
*/
typedef struct _VMIP_SDP_ADDITIONAL_INFORMATION
{
   char pSessionName[MAX_CHAR_ARRAY_SIZE];                          /*!< Name of the SDP session. */
   char pSessionDescription[MAX_CHAR_ARRAY_SIZE];                   /*!< Description of the SDP session. */
   VMIP_MEDIA_CLOCK_TYPE MediaClockType;                            /*!< Type of media clock. */
   VMIP_TIMESTAMP_MODE TimestampMode;                               /*!< Timestamping mode. */
   bool8_t HasTimestampDelay;                                       /*!< Indicates if the timestamp delay is present in the SDP. */
   uint32_t TimestampDelay;                                         /*!< Timestamp delay, as specified in ST2110-10:2021. */
   char pAudioChannelOrder[MAX_CHAR_AUDIO_CHANNEL_ORDER_SIZE];      /*!< For ST2110-30 essences, audio channel order. */
   VMIP_REFERENCE_CLOCK ReferenceClock;                             /*!< Information regarding the reference clock. */
   VMIP_ST2110_21_TRAFFIC_SHAPING_PROFILE TrafficShapingProfile;    /*!< Traffic shaping profile. */
}VMIP_SDP_ADDITIONAL_INFORMATION;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

   /*!
   @brief VideoMasterIP SDP reading

   This function takes a SDP, parses it and translates it in VMIP symbols.

   @returns
   The function returns the status of its execution as a VMIP error code.
   Errors can happen if :
   - One of the pointer is null;
   - The SDP is incorrectly formated.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_ReadSDP(const char* pSDPBuffer /*!< [in] Buffer string containing the SDP buffer to read */
      , const uint32_t InputBufferSize                                   /*!< [in] Size of the input buffer string */
      , VMIP_STREAM_NETWORK_CONFIG* pStreamNetworkConfig                 /*!< [out] Pointer to a caller\-allocated variable receiving the stream network configuration */
      , VMIP_STREAM_ESSENCE_CONFIG* pStreamEssenceConfig                 /*!< [out] Pointer to a caller\-allocated variable receiving the stream essence configuration */
      , VMIP_SDP_ADDITIONAL_INFORMATION* pSdpAdditionalInformation       /*!< [out] Pointer to a caller\-allocated variable receiving addition SDP information */
   );

   /*!
   @brief VideoMasterIP SDP writing
   
   This function takes VMIP symbols and writes a SDP based on them.
   
   @returns
   The function returns the status of its execution as a VMIP error code.
   Errors can happen if :
   - One of the pointer is null;
   - The size of the SDP buffer is too small.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_WriteSDP(const VMIP_STREAM_NETWORK_CONFIG StreamNetworkConfig /*!< [in] Stream network configuration */
      , const VMIP_STREAM_ESSENCE_CONFIG StreamEssenceConfig                                            /*!< [in] Stream essence configuration */
      , const VMIP_SDP_ADDITIONAL_INFORMATION SdpAdditionalInformation                                  /*!< [in] Additional SDP information */
      , char* pSDPBuffer                                                                                /*!< [out] Pointer to a caller\-allocated variable receiving the SDP buffer */
      , uint32_t* pInputBufferSize                                                                      /*!< [in, out] Input : Size of the SDP buffer string 
                                                                                                                     Output : Size of the SDP buffer string actually used after writing in it*/
   );

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS

#endif //_VIDEOMASTERIP_SDP_H_