/*!
   @file
   @brief This file gathers PTP related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/

#ifndef _VIDEOMASTERIP_PTP_H_
#define _VIDEOMASTERIP_PTP_H_

#include "videomasterip_core.h"


/*!
   @brief VideoMasterIP PTP state enumeration

   The VMIP_PTP_STATE enumeration contains the different available PTP port states.
   These values are returned as indexes by VMIP_GetPTPStatus functions calls through the structure VMIP_PTP_STATUS.

   @see VMIP_PTP_STATUS and VMIP_GetPTPStatus
*/
typedef enum _VMIP_PTP_STATE
{
   VMIP_PTP_STATE_INITIZALIZING,       /*!< The device is initializing its PTP configuration */
   VMIP_PTP_STATE_FAULTY,              /*!< Faulty state */
   VMIP_PTP_STATE_DISABLED,            /*!< The PTP device capability is disabled */
   VMIP_PTP_STATE_LISTENING,           /*!< The device is listening for PTP synchronization messages */
   VMIP_PTP_STATE_PREMASTER,           /*!< The device is in a pre-master state */
   VMIP_PTP_STATE_MASTER,              /*!< The device is a PTP master */
   VMIP_PTP_STATE_PASSIVE,             /*!< Passive state */
   VMIP_PTP_STATE_UNCALIBRATED,        /*!< The device receives PTP synchronization messages and tries to lock on the master clock */
   VMIP_PTP_STATE_SLAVE,               /*!< The device is a slave of the PTP master clock */
   NB_VMIP_PTP_STATE
}VMIP_PTP_STATE;

/*!
   @brief VideoMasterIP PTP delay mechanism

   The VMIP_PTP_DELAY_MECHANISM enumeration lists all the allowed
   PTP delay mechanism.
   
   These values are used as indexes by VMIP_GetPTPConfig / VMIP_SetPTPConfig functions calls through the structure VMIP_PTP_CONFIG.

   @see VMIP_PTP_STATUS, VMIP_GetPTPConfig and VMIP_SetPTPConfig
*/
typedef enum _VMIP_PTP_DELAY_MECHANISM
{
   VMIP_PTP_DM_E2E,          /*!< End To End delay mechanism */
   VMIP_PTP_DM_P2P,          /*!< Peer To Peer delay mechanism */
   NB_VMIP_PTP_DELAY_MECHANISM
}VMIP_PTP_DELAY_MECHANISM;

/*!
   @brief VideoMasterIP PTP delay mechanism

   The VMIP_PTP_DELAY_MECHANISM enumeration lists all the allowed
   PTP delay mechanism.
   
   These values are used as indexes by VMIP_GetPTPConfig / VMIP_SetPTPConfig functions calls through the structure VMIP_PTP_CONFIG.

   @see VMIP_PTP_STATUS, VMIP_GetPTPConfig and VMIP_SetPTPConfig
*/
typedef enum _VMIP_PTP_TIME_STAMPING_METHOD
{
   VMIP_PTP_TM_SW,          /*!< Software time stamping */
   VMIP_PTP_TM_HW,          /*!< Hardware time stamping */
   NB_VMIP_PTP_TIME_STAMPING_METHOD
}VMIP_PTP_TIME_STAMPING_METHOD;

/*!
   @brief VideoMasterIP PTP profile

   The VMIP_PTP_DELAY_MECHANISM enumeration lists all the allowed PTP profiles.
   
   These values are used as indexes by VMIP_CreatePTPDSFromProfile.

   @see VMIP_CreatePTPDSFromProfile
*/
typedef enum _VMIP_PTP_PROFILE
{
   VMIP_PTP_PROFILE_ST2059_2,   /*!< ST2059-2 PTP profile */
   VMIP_PTP_PROFILE_AES67,      /*!< AES67 PTP profile */
   NB_VMIP_PTP_PROFILE
}VMIP_PTP_PROFILE;

/*!
   @brief VideoMasterIP PTP default data set

   This structure holds the necessary PTP default dataset variables used to affect the behavior of a PTP device.

   @see VMIP_CreatePTPDSFromProfile, VMIP_PTP_CONFIG
*/
typedef struct _VMIP_PTP_DEFAULTDS
{
   uint8_t ClockAccuracy;                  /*!< Clock accuracy */
   uint16_t OffsetScaledLogVariance;       /*!< Offset scaled log variance */
   uint8_t DomainNumber;                   /*!< Domain number */
   uint8_t Priority1;                      /*!< Priority 1 */
   uint8_t Priority2;                      /*!< Priority 2 */
}VMIP_PTP_DEFAULTDS;

/*!
   @brief VideoMasterIP PTP port dataset

   This structure holds the necessary PTP port dataset variables used to affect the behavior of a PTP device.

   @see VMIP_CreatePTPDSFromProfile, VMIP_PTP_CONFIG
*/
typedef struct _VMIP_PTP_PORTDS
{
   int8_t LogMinDelayReqInterval;              /*!< Base-2 log of the minimum delay request interval */
   uint8_t AnnounceReceiptTimeout;             /*!< Timeout on announce response */
   VMIP_PTP_DELAY_MECHANISM DelayMechanism;    /*!< Delay mechanism */
   int8_t LogMinPDelayReqInterval;             /*!< Base-2 log of the minimum p-delay request interval */
}VMIP_PTP_PORTDS;

/*!
   @brief VideoMasterIP PTP configuration

   The VMIP_PTP_CONFIG structure contains information regarding the ptp configuration.

   @see VMIP_SetPTPConfig VMIP_GetPTPConfig
*/
typedef struct _VMIP_PTP_CONFIG {
   uint32_t Size;                                    /*!< Size of the structure used */
   uint64_t NetworkInterfaceId;                      /*!< Id of the interface on which the Ptp packets are received */
   bool8_t HybridMode;                               /*!< Hybrid delay mechanism (if true slave send delay_req with unicast)  */
   VMIP_PTP_DEFAULTDS DefaultDS;                     /*!< Default dataset */
   VMIP_PTP_PORTDS PortDS;                           /*!< Port dataset */
   VMIP_PTP_TIME_STAMPING_METHOD TimeStampingMethod; /*!< Timestamping method */
}VMIP_PTP_CONFIG;

/*!
   @brief VideoMasterIP PTP status

   The VMIP_PTP_STATUS structure contains information regarding the ptp status.

   @see VMIP_GetPTPStatus
*/
typedef struct _VMIP_PTP_STATUS {
   uint32_t Size;                        /*!< Size of the current structure. Should be set to sizeof(VMIP_PTP_STATUS). */
   uint64_t CurrentSystemTime;           /*!< Current time on the system hosting VCS. Value is in microseconds since epoch time. */
   struct
   {
      uint16_t OffsetScaledLogVariance;  /*!< Estimation of the variation of the local clock. */
      uint8_t pClockIdentity[8];         /*!< Clock identity. */
   }DefaultDS;                           /*!< Default data set. */
   struct                               
   {                                    
      double PeerMeanPathDelay;           /*!< If delay mechanism is P2P, this value is an image of the one-way propagation delay (s). */
      int8_t LogAnnounceInterval;         /*!< Base-2 log of the announce interval. */
      uint8_t pPortIdentity[10];          /*!< Local port identity. */
      VMIP_PTP_STATE PortState;           /*!< Local port state. */
   }PortDS;                               /*!< Port data set. */
   struct                               
   {                                    
      uint16_t StepsRemoved;              /*!< Number of communication path between the local clock and grandmaster clock. */
      double OffsetFromMaster;            /*!< Offset from master (in sec.). */
      double MeanPathDelay;               /*!< Image of the network delay between the local clock and the grandmaster clock. */
   }CurrentDS;                            /*!< Current data set. */
   struct
   {
      uint8_t pParentPortIdentity[10];                  /*!< Port identity of the parent clock. */
      bool8_t ParentStats;                              /*!< True if the ObservedParentOffsetScaledLogVariance and ObservedParentClockPhaseChangeRate fields are valid. */
      uint16_t ObservedParentOffsetScaledLogVariance;   /*!< Parent clock variance as measured by the local clock. */
      uint32_t ObservedParentClockPhaseChangeRate;      /*!< Parent clock phase change rate as measured by the local clock. */
      uint8_t pGrandMasterIdentity[8];                  /*!< Identity of the grandmaster clock. */
      uint8_t GrandMasterClockClass;                    /*!< Grandmaster clock class. */
      uint8_t GrandMasterClockAccuracy;                 /*!< Grandmaster clock accuracy. */
      uint16_t GrandMasterOffsetScaledLogVariance;      /*!< Grandmaster clock variance. */
      uint8_t GrandMasterPriority1;                     /*!< Grandmaster priority 1. */
      uint8_t GrandMasterPriority2;                     /*!< Grandmaster priority 2. */
   }ParentDS;                                           /*!< Parent data set. */
}VMIP_PTP_STATUS;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

/*!
   @brief VideoMasterIP PTP service configuration

   This function configures the properties of the PTP service.

   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The PTP service handle is not coherent;
   - The value of a configuration is not allowed.

   @see VMIP_PTP_CONFIG VMIP_GetPTPConfig
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SetPTPConfig(HANDLE VcsContext /*!< [in] Handle of the PTP session to set property on */
   , VMIP_PTP_CONFIG PtpConfig                                        /*!< [in] User-filled structure VMIP_PTP_CONFIG the PTP configuration. */
);

/*!
   @brief VideoMasterIP PTP service configuration query

   This function retrieves the properties of the PTP service.

   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The PTP service handle is not coherent;
   - The pointer is null;

   @see VMIP_PTP_CONFIG VMIP_SetPTPConfig
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetPTPConfig(HANDLE VcsContext /*!< [in] Handle of the PTP service to get property from */
   , VMIP_PTP_CONFIG* pPtpConfig                                      /*!< [out] Pointer to caller\-allocated variable to return the PTP config */
);

/*!
   @brief VideoMasterIP PTP service status query

   This function retrieves the status of the PTP service.

   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The PTP service handle is not coherent;
   - The pointer is null.

   @see VMIP_PTP_STATUS
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetPTPStatus(HANDLE VcsContext /*!< [in] Handle of the PTP service to get property from */
   , VMIP_PTP_STATUS* pPtpStatus                                      /*!< [out] Pointer to caller\-allocated variable to return the PTP status */
);

/*!
   @brief VideoMasterIP PTP datasets creation

   This function creates the default and port datasets based on a given PTP profile.

   @returns
   The function returns the status of its execution as VideoMasterIP error code.
   Errors can happen if :
   - The value for PTPProfile is unknown;
   - One of the pointers is null.

   @see VMIP_PTP_PROFILE
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_CreatePTPDSFromProfile(VMIP_PTP_PROFILE PTPProfile /*!< [in] PTP profile from which the datasets are defined */
   , VMIP_PTP_DEFAULTDS* pDefaultDS                                                       /*!< [out] Pointer to caller\-allocated variable to return default dataset*/
   , VMIP_PTP_PORTDS* pPortDS                                                             /*!< [out] Pointer to caller\-allocated variable to return port dataset*/
);

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS

#endif //_VIDEOMASTERIP_PTP_H_