/*!
   @file
   @brief This file gathers network interface related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/

#include "videomasterip_core.h"
#include "videomasterip_st2110_21.h"

#ifndef _VIDEOMASTERIP_NETWORKINTERFACE_H_
#define _VIDEOMASTERIP_NETWORKINTERFACE_H_

/*!
   @brief VideoMasterIP duplex mode enumeration

   @see VMIP_NETWORK_ITF_STATUS
*/
typedef enum _VMIP_NETWORK_ITF_DUPLEX_MODE
{
   VMIP_NETWORK_ITF_DUPLEX_MODE_HALF,    /*!< Half duplex mode. */
   VMIP_NETWORK_ITF_DUPLEX_MODE_FULL,    /*!< Full duplex mode. */
   VMIP_NETWORK_ITF_DUPLEX_MODE_UNKNOWN, /*!< Unknown duplex mode. */
   NB_NETWORK_ITF_DUPLEX_MODE
}VMIP_NETWORK_ITF_DUPLEX_MODE;

/*!
   @brief VideoMasterIP network interface configuration structure

   The VMIP_NETWORK_ITF_CONFIG structure contains information regarding a network interface.

   @see VMIP_GetNetworkInterfaceConfig
*/
typedef struct _VMIP_NETWORK_ITF_CONFIG
{
   uint32_t InterfaceIpAddress;  /*!< Ip address. */
   uint32_t SubnetMask;          /*!< Subnet mask. */
}VMIP_NETWORK_ITF_CONFIG;

/*!
   @brief VideoMasterIP network interface status structure

   The VMIP_NETWORK_ITF_STATUS structure contains status regarding a network interface.

   @see VMIP_GetNetworkInterfaceStatus
*/
typedef struct _VMIP_NETWORK_ITF_STATUS
{
   uint32_t NumaNode;                                               /*!< Numa node on which the network interface is connected. */
   uint64_t MacAddress;                                             /*!< Mac addres. */
   uint32_t OsId;                                                   /*!< Id as given by the Os. */
   char pName[MAX_CHAR_ARRAY_SIZE];                                 /*!< Name as given by the Os.*/
   bool8_t LinkStatus;                                              /*!< Link status.*/
   uint32_t LinkSpeed;                                              /*!< Link speed (in Mbits/sec).*/
   VMIP_ST2110_21_TRAFFIC_SHAPING_PROFILE TrafficShapingProfile;    /*!< Traffic Shaping Profile */
   VMIP_NETWORK_ITF_DUPLEX_MODE DuplexMode;                         /*!< Duplex mode (Half or Full) */
   uint32_t MaxTransferUnit;                                        /*!< Maximum transmission unit */
   char pDriverVersion[MAX_CHAR_ARRAY_SIZE];                        /*!< Driver version */
   uint64_t UploadRate;                                             /*!< Current upload (egress) throughput on the interface (in byte/sec). */
   uint64_t DownloadRate;                                           /*!< Current download (ingress) throughput on the interface (in byte/sec). */
}VMIP_NETWORK_ITF_STATUS;

/*!
   @brief VideoMasterIP queue configuration structure for one essence

   The VMIP_NETWORK_ITF_DPDK_NB_QUEUE_PER_ESSENCE structure contains the number of DPDK RX/TX queue for a given essence.

   @see VMIP_NETWORK_ITF_DPDK_NB_QUEUE
*/
typedef struct _VMIP_NETWORK_ITF_DPDK_NB_QUEUE_PER_ESSENCE
{
   uint32_t NbRxQueue;  /*!< Maximum nimber of RX queue. */
   uint32_t NbTxQueue;  /*!< Maximum nimber of TX queue. */
}VMIP_NETWORK_ITF_DPDK_NB_QUEUE_PER_ESSENCE;

/*!
   @brief VideoMasterIP queue configuration structure for a network interface

   The VMIP_NETWORK_ITF_STATUS structure contains the number of DPDK RX/TX queue for each essence.
   The configuration related to an essence is only applied if the corresponding "Modify" flag is set to true.

   @see VMIP_DPDK_SetNetworkItfNbQueue
*/
typedef struct _VMIP_NETWORK_ITF_DPDK_NB_QUEUE
{
   bool8_t Modify_St2110_20;                                     /*!< Flag specifying if the ST2110-20 configuration will be modified */
   VMIP_NETWORK_ITF_DPDK_NB_QUEUE_PER_ESSENCE St2110_20;      /*!< ST2110-20 number of queue configuration. Only used if Modify_St2110_20 is 'true' */                           
   bool8_t Modify_St2110_22;                                     /*!< Flag specifying if the ST2110-22 configuration will be modified */
   VMIP_NETWORK_ITF_DPDK_NB_QUEUE_PER_ESSENCE St2110_22;      /*!< ST2110-22 number of queue configuration. Only used if Modify_St2110_22 is 'true' */
   bool8_t Modify_St2110_30;                                     /*!< Flag specifying if the ST2110-30 configuration will be modified */
   VMIP_NETWORK_ITF_DPDK_NB_QUEUE_PER_ESSENCE St2110_30;      /*!< ST2110-30 number of queue configuration. Only used if Modify_St2110_30 is 'true' */
   bool8_t Modify_St2110_40;                                     /*!< Flag specifying if the ST2110-40 configuration will be modified */
   VMIP_NETWORK_ITF_DPDK_NB_QUEUE_PER_ESSENCE St2110_40;      /*!< ST2110-40 number of queue configuration. Only used if Modify_St2110_40 is 'true' */
}VMIP_NETWORK_ITF_DPDK_NB_QUEUE;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

   /*!
      @brief VideoMasterIP network interface handle query

      This function retrieves the handle of the network interfaces available to a VCS session

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The VCS handle is not coherent;
      - The network interface array pointer is null;
      - The network interface number pointer is null.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetNetworkInterfaceHandlesList(HANDLE VCSHandle /*!< [in] Handle of the VCS service to retrieve the interfaces from */
      , uint64_t pNetworkInterfaceHandlesList[MAX_HANDLE_ARRAY_SIZE]                      /*!< [out] Pointer to a caller\-allocated array receiving the network interfaces handles */
      , uint32_t* pNetworkInterfaceHandlesNb                                              /*!< [out] Pointer to a caller\-allocated variable receiving the number of network interfaces available */
   );

   /*!
      @brief VideoMasterIP network interface configuration query

      This function retrieves the properties of a network interface.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The network interface handle is not coherent;
      - The pointer is null.

      @see VMIP_NETWORK_ITF_CONFIG
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetNetworkInterfaceConfig(HANDLE VcsContext /*!< [in] Handle of VCS context related to the network interface to get the configuration from */
      , uint64_t NetworkItfId                                                         /*!< [in] id of the network interface to get the configuration from */
      , VMIP_NETWORK_ITF_CONFIG* pNetworkInterfaceConfig                              /*!< [out] Pointer to caller\-allocated variable to return the network interface configuration */
   );

   /*!
      @brief VideoMasterIP network interface status query

      This function retrieves the status of a network interface.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The network interface handle is not coherent;
      - The pointer is null.

      @see VMIP_NETWORK_ITF_STATUS
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetNetworkInterfaceStatus(HANDLE VcsContext /*!< [in] Handle of VCS context related to the network interface to get the status from */
      , uint64_t NetworkItfId                                                         /*!< [in] id of the network interface to get the status from */
      , VMIP_NETWORK_ITF_STATUS* pNetworkInterfaceStatus                              /*!< [out] Pointer to caller\-allocated variable to return the PTP status */
   );

   /*!
      @brief VideoMasterIP UDP port drop (DPDK only)

      This function drops a UDP port on a given interface.
      This means that packets toward this UDP port will be dropped by the DPDK module.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The network interface handle is not coherent;
      - The port could not be dropped.

      @see VMIP_DPDK_AllowUdpPort

   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_DPDK_DropUdpPort(
      HANDLE VcsContext       /*!< [in] Handle of VCS context related to the network interface */
      , uint64_t NetworkItfId /*!< [in] id of the network interface we want to drop the port on */
      , uint16_t UdpPort      /*!< [in] Value of the udp port we want to drop */
   );

   /*!
      @brief VideoMasterIP UDP port allowance (DPDK only)

      This function allows a UDP port on a given interface.
      This means that packets toward this UDP port will not be dropped by the DPDK module.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The network interface handle is not coherent;
      - The port could not be allowed.

      @see VMIP_DPDK_DropUdpPort
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_DPDK_AllowUdpPort(
      HANDLE VcsContext       /*!< [in] Handle of VCS context related to the network interface */
      , uint64_t NetworkItfId /*!< [in] id of the network interface we want to drop the port on */
      , uint16_t UdpPort      /*!< [in] Value of the udp port we want to allow */
   );


   /*!
      @brief VideoMasterIP configuration of the number of queue available on a network interface (DPDK only)

      When calling this function, the user can modify the number of queue/streams that can be started on a given network interface.  
      This function will also restart the device so no stream can be running on it when the function is called.
      After being called, it may takes a few seconds for the device to acquire an IP address if DHCP is used.

      The user can also choose to write the new configuration in the VCS configuration file so that the new configuration is kept after restarting VCS.
      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - VMIPERR_INCORRECTSTATE : At least one stream is started on the network interface;
      - VMIPERR_OPERATIONFAILED : DPDK did not allow the new configuration.

      @see VMIP_DPDK_DropUdpPort
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_DPDK_SetNetworkItfNbQueue(
      HANDLE VcsContext                              /*!< [in] Handle of VCS context related to the network interface */
      , uint64_t NetworkItfId                        /*!< [in] id of the network interface we want to configure */
      , VMIP_NETWORK_ITF_DPDK_NB_QUEUE NbQueues      /*!< [in] Configuration to be applied */
      , bool SaveToConfigFile                        /*!< [in] Flag specifying if the configuration should be saved in the configuration file */
   );

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS

#endif //_VIDEOMASTERIP_NETWORKINTERFACE_H_