/*!
   @file
   @brief This file gathers CPU related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/

#include "videomasterip_core.h"

#ifndef _VIDEOMASTERIP_CPU_H_
#define _VIDEOMASTERIP_CPU_H_

/*!
   @brief VideoMasterIP CPU core conductor availability

   This structures indicates if a CPU core can host a conductor.

   @see VMIP_CPUCORE_STATUS
*/
typedef enum _VMIP_CONDUCTOR_AVAILABILITY
{
   VMIP_CONDUCTOR_AVAILABLE,        /*!< A conductor can be started on this CPU core */
   VMIP_CONDUCTOR_NOT_AVAILABLE,    /*!< A conductor can not be started on this CPU core */
   VMIP_CONDUCTOR_ALREADY_USED,     /*!< A conductor is already running on this CPU core */
   NB_VMIP_CONDUCTOR_AVAILABILITY
} VMIP_CONDUCTOR_AVAILABILITY;

/*!
   @brief VideoMasterIP CPU core status structure

   The VMIP_CPUCORE_STATUS structure contains information regarding a CPU core.

   @see VMIP_GetCpuCoreStatus
*/
typedef struct _VMIP_CPUCORE_STATUS
{
   uint32_t CoreId;                                   /*!< Index of the Cpu Core as given by the Os */
   uint32_t NumaNode;                                 /*!< Numa node of the Cpu Core*/
   uint32_t MaxSpeed;                                 /*!< Maximum speed of the Cpu Core (in MHz)*/
   uint8_t  CoreUsage;                                /*!< Current usage of the cpu core (in %)*/
   uint32_t GroupId;                                  /*!< Group of the CpuCore (Windows Only)*/
   VMIP_CONDUCTOR_AVAILABILITY ConductorAvailability; /*!< Indicates if core can be used to host a conductor.. */ 
}VMIP_CPUCORE_STATUS;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

   /*!
      @brief Available CPU handles query

      This function retrieves the handles of CPU Core and their number.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The handle is invalid;
      - One of the pointer is null.

   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetCPUCoresHandlesList(HANDLE VCSContext /*!< [in]  Handle of the VCS session to get the CPU core list from */
      , uint64_t pCPUCoresIdList[MAX_HANDLE_ARRAY_SIZE]                            /*!< [out] Pointer to caller\-allocated array to return a list of CPU core Id's */
      , uint32_t* pCPUCoresIdNb                                                    /*!< [out] Pointer to caller\-allocated variable to return the number of available CPU cores */
   );

   /*!
      @brief CPU handles query related to the OS Id

      This function retrieves the handles of the CPU Core corresponding to an OS id.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The handle is invalid;
      - There is no Cpu with this OS Id;
      - One of the pointers is null.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetCPUCoreFromOSId(HANDLE VCSContext /*!< [in] Handle of the VCS session to get the CPU core from */
      , uint64_t OSId_u64                                                      /*!< [in] OS Cpu Id */
      , uint64_t* pCPUCoreId_u64                                               /*!< [out] Pointer to caller\-allocated variable to return CPU Core Id */
   );

   /*!
      @brief VideoMasterIP CPU Core status query

      This function retrieves the status of a CPU core.

      @returns
      The function returns the status of its execution as a Vmip error code .
      Errors can happen if :
      - The CPU Core handle is not coherent;
      - The pointer is null;

      @see VMIP_CPU_STATUS
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetCpuCoreStatus(HANDLE VcsContext_H /*!< [in] Handle of the VCS session to get the CPU core status from*/
      , uint64_t CpuCoreId_u64                                                 /*!< [in] The CPU Core Id to get the status from*/
      , VMIP_CPUCORE_STATUS* pCpuStatus_X                                      /*!< [out] Pointer to caller\-allocated variable to return CPU Core status */
   );

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS

#endif //_VIDEOMASTERIP_CPU_H_