/*!
   @file
   @brief This file gathers general public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/


#ifndef _VIDEOMASTERIP_CORE_H_
#define _VIDEOMASTERIP_CORE_H_


#ifdef __GNUC__
#include <stdint-gcc.h>
#else
#include <stdint.h>
#endif

#define MAX_HANDLE_ARRAY_SIZE 128      /*!< Maximum number of handle in an arrray */
#define MAX_CHAR_ARRAY_SIZE 50         /*!< Maximum number of char in an array */
#define MAX_CHAR_VERSION_SIZE 256      /*!< Maximum number of char in an array */


#if defined _WIN64 || defined __CYGWIN__
#ifdef _BUILDING_API
#ifdef __GNUC__
#define VIDEOMASTER_IP_API __attribute__ ((dllexport))
#else
#define VIDEOMASTER_IP_API __declspec(dllexport)
#endif
#else
#ifdef __GNUC__
#define VIDEOMASTER_IP_API __attribute__ ((dllimport))
#else
#define VIDEOMASTER_IP_API __declspec(dllimport)
#endif
#endif
#else
#if __GNUC__ >= 4
#define VIDEOMASTER_IP_API __attribute__ ((visibility ("default")))
#else
#define VIDEOMASTER_IP_API
#endif
#endif

typedef void* HANDLE;        /*!< HANDLE type definition  */
typedef uint8_t  bool8_t;    /*!< Boolean value stored in 8 bits  */

/*!
   @brief VideoMasterIP error code enumeration

   The VMIP_ERRORCODE enumeration lists all the available error codes.

   These values are used as return values for all the functions.
*/
typedef enum _VMIP_ERRORCODE
{                                 
   VMIPERR_NOERROR,               /*!< No error */
   VMIPERR_FATALERROR,            /*!< Fatal error occurred (should re-install) */
   VMIPERR_OPERATIONFAILED,       /*!< Operation failed (undefined error) */
   VMIPERR_INVALIDHANDLE,         /*!< Invalid handle */
   VMIPERR_INVALIDPOINTER,        /*!< Invalid pointer */
   VMIPERR_BADARG,                /*!< Bad argument value */
   VMIPERR_NOTIMPLEMENTED,        /*!< Not implemented yet */
   VMIPERR_NOTFOUND,              /*!< Required element was not found */
   VMIPERR_INCORRECTSTATE,        /*!< The state of the component is not compatible with the requested operation.*/
   VMIPERR_BUFFERTOOSMALL,        /*!< Buffer too small */
   VMIPERR_TIMEOUT,               /*!< Timeout */
   VMIPERR_COMMUNICATION_ERROR,   /*!< Error when communicating with VCS. */
   VMIPERR_INVALID_LICENSE,       /*!< The license is invalid for the required stream. */
   VMIPERR_BAD_CONFIGURATION,     /*!< The configuration of the component is invalid. */
   NB_VMIP_ERRORCODES
} VMIP_ERRORCODE;

/*!
   @brief VideoMasterIP core status structure

   The VMIP_VCS_STATUS structure contains information regarding the Virtual Card Service.

   @see VMIP_GetVCSStatus
*/
typedef struct _VMIP_VCS_STATUS {
   uint64_t VcsVersion;                       /*!< 64bits Vcs version defined as 0xMMMMmmmmbbbbbbbc, where :
                                                                          MMMM represents the MAJOR version,
                                                                          mmmm represents the MINOR version and
                                                                          bbbbbbb represents the BUILD version.
                                                                          c represents the CUSTOM version flag*/
   char VersionString[MAX_CHAR_VERSION_SIZE]; /*!< VCS version string */

}VMIP_VCS_STATUS;

/*!
   @brief Libraries status structure

   The VMIP_LIBRARIES_STATUS structure contains information regarding libraries used by VMIP.

   @see VMIP_GetLibrariesStatus
*/
typedef struct _VMIP_LIBRARIES_STATUS
{
   uint64_t VmipVersion;      /*!< 64bits VMIP library version defined as 0xMMMMmmmmbbbbbbbb, where :
                                                           MMMM represents the MAJOR version,
                                                           mmmm represents the MINOR version and
                                                           bbbbbbbb represents the BUILD version.*/
   uint64_t DataPipeVersion;  /*!< 64bits Data Pipe library version defined as 0xMMMMmmmmbbbbbbbb, where :
                                                           MMMM represents the MAJOR version,
                                                           mmmm represents the MINOR version and
                                                           bbbbbbbb represents the BUILD version.*/

}VMIP_LIBRARIES_STATUS;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

/*!
   @brief    VCS context creation

   This function creates a handle to a VCS context by connecting to the VCS service.

   To do so, the user should provide the URL to the VCS service. There are 3 options:
   - Given url is an empty string. In this case, the TCP port used by the service will be detected and VMIP will attempt to reach http://localhost:[VCS_PORT]/.
   - Given url is a valid URL (http://127.0.0.1/ or http://localhost/) but the TCP port is not given. In this case, the TCP port used by the service will be detected and VMIP will attempt to reach http://[GIVEN_URL]:[VCS_PORT]/
   - Given url is a valid URL and the port is present : VMIP will attempt to reach the given URL.

   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The server is not responding;
   - URL is not valid (not well formatted);
   - URL is not 127.0.0.1 or localhost;
   - The VCS context could not be opened.

   @see VMIP_DestroyVCSContext
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_CreateVCSContext(const char* pUrl /*!< [in] URL used to connect to the VCS service. See above for the different possibilities. */
   , HANDLE* pVCSContext                                                 /*!< [out] Pointer to a caller\-allocated variable receiving the VCS context handle*/
);

/*!
   @brief VCS context destruction

   This function destroys a VCS context previously created with VMIP_CreateVCSContext

   @returns 
   The function returns the status of its execution as Vmip error code.
   Errors can happen if :
   - The server is not responding;
   - The VCS context was not opened.

   @see VMIP_CreateVCSContext
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_DestroyVCSContext(HANDLE VCSContext /*!< [in] Handle of the VCS session to close */);

/*!
   @brief VCS session property query

   This function retrieves value of the VCS properties

   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The handle is invalid;
   - The pointer is null.

   @see VMIP_VCS_STATUS
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetVCSStatus(HANDLE VCSContext /*!< [in] Handle of the VCS session to get the status from */
   , VMIP_VCS_STATUS* pVCSStatus                                      /*!< [out] Pointer to caller\-allocated variable to return status structure*/
);

   /*!
      @brief Libraries status query

      This function retrieves the properties of the libraries used by VMIP.

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The pointer is null.

      @see VMIP_LIBRARIES_STATUS
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetLibrariesStatus(VMIP_LIBRARIES_STATUS* pLibrariesStatus /*!< [out] Pointer to caller\-allocated variable to return status structure */);

   /*!
      @brief Stringify error code

      This function stringifies an error code

      @returns
      The function returns the status of its execution as a Vmip error code.
      Errors can happen if :
      - The error code is not known.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_ErrorToString(VMIP_ERRORCODE ErrorCode             /*!< [in] Error code to stringify */
                                                      , char pErrorCode[MAX_CHAR_ARRAY_SIZE] /*!< [out] Character array containing the string associated to the error code */
   );

   /*!
   @brief    Licensing update
   
   This function refresh the licensing information inf VCS

   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The server is not responding.
*/
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_RefreshLicensing(const char* pUrl /*!< [in] VCS server URL */);

   /*!
   @brief    Last error message retrieval
   
   This function get the last error message for the current thread
   */
   VIDEOMASTER_IP_API void VMIP_GetLastErrorMessage(char* pMessage /*!< [in, out] Pointer to caller\-allocated string of char where the last API error is returned */, uint32_t Size /*!< [in] Size of the caller\-allocated string of char */);

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS

#endif //_VIDEOMASTERIP_CORE_H_