/*! 
   @file
   @brief This file gathers Conductor related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/

#ifndef _VIDEOMASTERIP_CONDUCTOR_H_
#define _VIDEOMASTERIP_CONDUCTOR_H_

#include "videomasterip_core.h"


/*!
   @brief VideoMasterIP Conductor configuration structure

   The VMIP_CONDUCTOR_CONFIG structures contains information regarding a conductor.

   @see VMIP_SetConductorConfig and VMIP_GetConductorConfig
*/
typedef struct _VMIP_CONDUCTOR_CONFIG{
   uint64_t CpuCoreId; /*!< Id of the Cpu Core associated with the conductor */
   bool8_t Shared;     /*!< Tells if the conductor can be used by multiple contexts.  
                            For more information, see the \ref conductor-sharing section in the documentation.*/
}VMIP_CONDUCTOR_CONFIG;


/*!
   @brief VideoMasterIP conductor status structure

   The VMIP_CONDUCTOR_STATUS structures contains information regarding a conductor.

   @see VMIP_GetConductorStatus
*/
typedef struct _VMIP_CONDUCTOR_STATUS {
   uint64_t Rate;   /*!< The rate that has been transferred (sent and received) by the conductor during the last second (in byte/sec). */
   double Reserved; /*!< Reserved field. */
}VMIP_CONDUCTOR_STATUS;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

/*!
   @brief VideoMasterIP shared conductors handles query
   
   This function retrieves the handles of the available shared conductors.

   @returns The status of its execution as a Vmip error code.
   Errors can happen if :
   - The VCS handle is not coherent;
   - The conductor array pointer is null;
   - The conductor number pointer is null.
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetSharedConductorList(HANDLE VCSContext /*!< [in]  Handle of the VCS session to get the shared conductor list from. */
   , uint64_t pConductorHandlesList[MAX_HANDLE_ARRAY_SIZE]                      /*!< [out] Pointer to caller\-allocated array to return a list of shared conductor Id's. */
   , uint32_t* pConductorNb                                                     /*!< [out] Pointer to caller\-allocated variable to return the number of available shared conductors. */
);

/*!
   @brief VideoMasterIP shared conductor addition
   
   This function adds a conductor to the local context

   @returns The status of its execution as a Vmip error code.
   Errors can happen if :
   - The VCS handle is not coherent;
   - The conductor is not shared.
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_AddSharedConductorToContext(HANDLE VCSContext /*!< [in] Handle of the VCS context in which the conductor must be added. */
   , uint64_t ConductorId                                                            /*!< [in] Id of the conductor to be added to the context. */
);

/*!
   @brief VideoMaster conductor handle creation
   
   This function creates a new conductor linked to a specified VCS service

   @returns The status of its execution as a Vmip error code.
   Errors can happen if :
   - The VCS handle is not coherent;
   - The conductor id pointer is null;
   - The conductor creation fails.
   
   @see VMIP_DestroyConductor
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_CreateConductor(HANDLE VCSContext /*!< [in] Handle of the VCS context in which the conductor must be created. */
   , bool8_t Shared                                                      /*!< [in] Parameter telling if the conductor can be shared between contexts. */
   , uint64_t* pConductorId                                              /*!< [out] Id of the created conductor. */
);

/*!
   @brief VideoMaster conductor handle destruction
   
   This function destroys the handle of a specified conductor
   
   @returns The status of its execution as a Vmip error code.
   Errors can happen if :
   - The VCS handle is not coherent;
   - The conductor deletion fails.
   
   @see VMIP_CreateConductor
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_DestroyConductor(HANDLE VCSContext_H /*!< [in] Handle of the VCS context in which the conductor must be detroyed. */
   , uint64_t ConductorId_u64                                               /*!< [in] Id of the conductor to destroy. */
);

/*!
   @brief VideoMasterIP conductor configuration
   
   This function configures the properties of a conductor.

   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The conductor handle is not coherent;
   - A value in the configuration is not allowed.
   
   @see  VMIP_CONDUCTOR_CONFIG and VMIP_GetConductorConfig
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SetConductorConfig(HANDLE VCSContext /*!< [in] Handle of the VCS context*/
   , uint64_t ConductorId                                                   /*!< [in] Id of the conductor to set the configuration on. */
   , VMIP_CONDUCTOR_CONFIG ConductorConfig                                  /*!< [in] User-filled structure VMIP_CONDUCTOR_CONFIG containing the conductor configuration. */
);

/*!
   @brief VideoMasterIP conductor configuration query
   
   This function retrieves the properties of a conductor.

   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The conductor handle is not coherent;
   - The pointer is null;
   - No configuration was set beforehand
   
   @see VMIP_CONDUCTOR_CONFIG and VMIP_SetConductorConfig
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetConductorConfig(HANDLE VCSContext /*!< [in] Handle of the VCS context. */
   , uint64_t ConductorId                                                   /*!< [in] Id of the conductor to get the configuration from. */
   , VMIP_CONDUCTOR_CONFIG* pConductorConfig                                /*!< [out] Pointer to caller\-allocated variable to return the conductor configuration. */
);

/*!
   @brief VideoMasterIP conductor start
   
   This function starts the specified conductor
   
   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The conductor handle is not coherent;
   - The conductor configuration has not been set;
   - An error occurs when starting the conductor.

   @see VMIP_StopConductor
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StartConductor(HANDLE VCSContext /*!< [in] Handle of the VCS context. */
   , uint64_t ConductorId                                               /*!< [in] Id of the conductor to start. */
);

/*! 
   @brief VideoMaster conductor stop

   This function stops the specified conductor if it is running
   
   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The conductor handle is not coherent;
   - An error occurs when stopping the conductor
   
   @see VMIP_StartConductor
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_StopConductor(HANDLE VCSContext /*!< [in] Handle of the VCS context. */
   , uint64_t ConductorId                                              /*!< [in] Id of the conductor to stop. */
);

/*!
   @brief VideoMasterIP conductor status query
   
   This function retrieves the status of a conductor.

   @returns
   The function returns the status of its execution as a Vmip error code
   Errors can happen if :
   - The conductor handle is not coherent;
   - The pointer is null;
   
   @see VMIP_NETWORK_ITF_STATUS
*/
VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetConductorStatus(HANDLE VCSContext /*!< [in] Handle of the VCS context. */
   , uint64_t ConductorId                                                   /*!< [in] Id of the conductor to get the status from. */
   , VMIP_CONDUCTOR_STATUS* pConductortatus                                 /*!< [out] Pointer to caller\-allocated variable to return the conductor status. */
);

#ifdef __cplusplus
}
#endif

#endif

#endif