/*!
   @file
   @brief This file gathers ANC Processing related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/

#ifndef _VIDEOMASTERIP_ANC_PROCESSING_H_
#define _VIDEOMASTERIP_ANC_PROCESSING_H_

#include "videomasterip_core.h"

/*!
   @brief Closed caption types

   The VMIP_CLOSED_CAPTION_TYPE enumeration lists all the supported Closed caption types. These values are used by the VMIP_CLOSED_CAPTION_INFO.

   @see VMIP_CLOSED_CAPTION_INFO
 */
typedef enum _VMIP_CLOSED_CAPTION_TYPE
{
   VMIP_CLOSED_CAPTION_EIA_608B_VANC,  /*!< Close Caption EIA 608B on vertical ancillary data */
   VMIP_CLOSED_CAPTION_EIA_708,        /*!< Close Caption EIA 708 */
   NB_VMIP_CLOSED_CAPTION_TYPE
} VMIP_CLOSED_CAPTION_TYPE;


/*!
   @brief VideomasterIP AFD and AR codes

   The VMIP_AFD_AR_CODE enumeration lists all the existing AFD code values. These values are used in VMIP_SlotEmbedAFD and VMIP_SlotExtractAFD functions calls.

   @see VMIP_AFD_AR_SLOT
 */

typedef enum _VMIP_AFD_AR_CODE
{
   VMIP_AFD_AR_U4_3,                   /*!< Undefined 4:3 */
   VMIP_AFD_AR_U16_9,                  /*!< Undefined 16:9 */
   VMIP_AFD_AR_TLB16_9,                /*!< Letterbox 16:9 image, at top of the coded frame */
   VMIP_AFD_AR_FF16_9,                 /*!< Full frame 16:9 image, the same as the coded frame */
   VMIP_AFD_AR_TLB14_9,                /*!< Letterbox 14:9 image, at top of the coded frame */
   VMIP_AFD_AR_HCPB14_9,               /*!< Pillarbox 14:9 image, horizontally centered in the coded frame */
   VMIP_AFD_AR_LBARG4_3,               /*!< Letterbox image with an aspect ratio greater than 16:9, vertically centered in the coded frame */
   VMIP_AFD_AR_LBARG16_9,              /*!< Letterbox image with an aspect ratio greater than 16:9, vertically centered in the coded frame */
   VMIP_AFD_AR_FF4_3,                  /*!< Full frame 4:3 image, the same as the coded frame */
   VMIP_AFD_AR_FF16_9P,                /*!< Full frame 16:9 image, the same as the coded frame */
   VMIP_AFD_AR_FF4_3P,                 /*!< Full frame 4:3 image, the same as the coded frame */
   VMIP_AFD_AR_HCPB4_3,                /*!< Pillarbox 4:3 image, horizontally centered in the coded frame */
   VMIP_AFD_AR_VCLB16_9,               /*!< Letterbox 16:9 image, vertically centered in the coded frame with all image areas protected */
   VMIP_AFD_AR_AIAPFF16_9,             /*!< Full frame 16:9 image, with all image areas protected */
   VMIP_AFD_AR_VCLB14_9,               /*!< Letterbox 14:9 image, vertically centered in the coded frame */
   VMIP_AFD_AR_HCPB14_9P,              /*!< Pillarbox 14:9 image, horizontally centered in the coded frame */
   VMIP_AFD_AR_FF4_3A14_9C,            /*!< Full frame 4:3 image, with alternative 14:9 center */
   VMIP_AFD_AR_PB4_3A14_9C,            /*!< Pillarbox 4:3 image, with alternative 14:9 center */
   VMIP_AFD_AR_LB16_9A14_9C,           /*!< Letterbox 16:9 image, with alternative 14:9 center */
   VMIP_AFD_AR_FF16_9A14_9C,           /*!< Full frame 16:9 image, with alternative 14:9 center */
   VMIP_AFD_AR_LB16_9A4_3C,            /*!< Letterbox 16:9 image, with alternative 4:3 center */
   VMIP_AFD_AR_FF16_9A4_3C,            /*!< Full frame 16:9 image, with alternative 4:3 center */
   NB_VMIP_AFD_AR_CODE
} VMIP_AFD_AR_CODE;

/*!
   @brief VideoMasterIP embedded timecode types

   The VMIP_TIMECODE_TYPE enumeration lists all the available timecode embedding modes.

   @see VMIP_SlotExtractTimecode and VMIP_SlotEmbedTimecode
 */
typedef enum _VMIP_TIMECODE_TYPE
{
   VMIP_TIMECODE_RP188_LTC,              /*!< LTC timecode embedded in ancillary data according to SMPTERP188 */
   VMIP_TIMECODE_RP188_VITC1,            /*!< First VITC timecode embedded in ancillary data according to SMPTE RP188 */
   VMIP_TIMECODE_RP188_VITC2,            /*!< Second VITC timecode embedded in ancillary data according to SMPTE RP188 */
   NB_VMIP_TIMECODE_TYPE
} VMIP_TIMECODE_TYPE;

/*!
   @brief HDR metadata types

   The VMIP_HDR_METADATA_TYPE enumeration lists all the supported HDR metadata types.

   @see VMIP_HDR_METADATA_INFO
 */
typedef enum _VMIP_HDR_METADATA_TYPE
{
   VMIP_HDR_METADATA_ST2108,             /*!< HDR/WCG Metadata Packing and Signaling in the VANC Data Space (SMPTE ST 2108-1) */
   NB_VMIP_HDR_METADATA_TYPE
} VMIP_HDR_METADATA_TYPE;

/*!
   @brief  Closed caption slot

   The VMIP_CLOSED_CAPTION_SLOT structure contains closed caption slot.

   @see VMIP_SlotExtractClosedCaptions and VMIP_SlotEmbedClosedCaptions
 */
typedef struct _VMIP_CLOSED_CAPTION_SLOT
{
   uint32_t DataSize;                   /*!< CLOSED_CAPTION data size */
   uint8_t pData[90];                   /*!< CLOSED_CAPTION data */
} VMIP_CLOSED_CAPTION_SLOT;

/*!
   @brief OP47 VBI packet

   The VMIP_VBI_PACKET structure contains a teletext packet.

   @see VMIP_OP47_SLOT
 */
typedef struct _VMIP_OP47_VBI_PACKET
{
   uint32_t LineNumber;                 /*!< Line Number */
   uint32_t FieldNumber;                /*!< Field Number (0 = even field, 1 = odd field) */
   uint32_t MagazineNumber;             /*!< Magazine number */
   uint32_t PacketNumber;               /*!< Packet number */
   uint8_t pData[40];                   /*!< Data */
} VMIP_OP47_VBI_PACKET;

/*!
   @brief OP47 data

   The VMIP_OP47_SLOT structure contains the OP47 data.

   @see VMIP_SlotExtractOP47 and VMIP_SlotEmbedOP47
 */
typedef struct _VMIP_OP47_SLOT
{
   VMIP_OP47_VBI_PACKET pVbiPacket[5];  /*!< Vbi Packet */
   uint32_t NbPackets;                  /*!< Number of Vbi Packets to insert */
} VMIP_OP47_SLOT;

/*!
   @brief Timecode data

   The VMIP_TIMECODE_SLOT structure contains the Timecode data.

   @see VMIP_SlotExtractTimecode and VMIP_SlotEmbedTimecode
 */
typedef struct _VMIP_TIMECODE_SLOT
{
   uint8_t  Hour;                /*!< Timecode hour component */
   uint8_t  Minute;              /*!< Timecode minute component */
   uint8_t  Second;              /*!< Timecode second component */
   uint8_t  Frame;               /*!< Timecode frame component */
   uint32_t BinaryGroups;        /*!< Timecode binary groups component. This field comprises 8 groups of 4-bit, stored in a 32-bit variable with LSB being the LSB of BG1 until MSB being MSB of BG8 */
   uint8_t  Flags;               /*!< Timecode 6 flag bits, as specified by SMPTE 12M
                                   * Bit 0: Drop frame flag
                                   * Bit 1: Color frame flag
                                   * Bit 2: Field identification flag
                                   * Bit 3: Binary group flag BGF0
                                   * Bit 4: Binary group flag BGF1
                                   * Bit 5: Binary group flag BGF2
                                  */
   uint8_t  Dbb[2];              /*!< Timecode distributed binary bit groups component (ATC only) */
} VMIP_TIMECODE_SLOT;


/*!
   @brief HDR metadata packet info

   The VMIP_HDR_METADATA_INFO structure contains information about the HDR metadata packet.

   @see VMIP_HDR_METADATA_PACKET
 */
typedef struct _VMIP_HDR_METADATA_INFO
{
   uint32_t LineNumber;                     /*!< HDR metadata packet Line number */
   uint32_t SampleType;                     /*!< HDR metadata packet ANC sample type */
   VMIP_HDR_METADATA_TYPE HdrMetadataType;  /*!< HDR metadata packet type */
} VMIP_HDR_METADATA_INFO;

/*!
   @brief HDR metadata packet data

   The VMIP_HDR_METADATA_DATA structure contains the HDR metadata packet data.

   @see VMIP_HDR_METADATA_PACKET
 */
typedef struct _VMIP_HDR_METADATA_DATA
{
   uint32_t DataSize;                      /*!< HDR metadata packet data size */
   uint8_t pData[255];                     /*!< HDR metadata packet data zone */
} VMIP_HDR_METADATA_DATA;

/*!
   @brief HDR metadata packet

   The VMIP_HDR_METADATA_PACKET structure contains the HDR metadata packet data.

   @see VMIP_HDR_METADATA_SLOT
 */
typedef struct _VMIP_HDR_METADATA_PACKET
{
   VMIP_HDR_METADATA_INFO Info;   /*!< HDR metadata packet info */
   VMIP_HDR_METADATA_DATA Data;   /*!< HDR metadata packet data */
} VMIP_HDR_METADATA_PACKET;

/*!
   @brief HDR metadata packets

   The VMIP_HDR_METADATA_SLOT structure contains the HDR metadata packets.

   @see VMIP_SlotExtractHdrMetadata and VMIP_SlotEmbedHdrMetadata
 */
typedef struct _VMIP_HDR_METADATA_SLOT
{
   uint32_t PacketCount;                     /*!< HDR metadata packets count in the slot */
   VMIP_HDR_METADATA_PACKET pPackets[45];    /*!< HDR metadata packets zone */
} VMIP_HDR_METADATA_SLOT;

/*!
   @brief ST2108 HDR metadata frames

   The VMIP_HDR_METADATA_ST2108_FRAME structure contains the ST2108 HDR metadata frames.

   @see VMIP_HDR_METADATA_DATA
 */
typedef struct _VMIP_HDR_METADATA_ST2108_FRAME
{
   uint8_t Type;        /*!< ST2108 HDR metadata frame type */
   uint8_t Length;      /*!< ST2108 HDR metadata frame length */
   uint8_t pData[253];  /*!< ST2108 HDR metadata data zone */
} VMIP_HDR_METADATA_ST2108_FRAME;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

   /*!
      @brief Active format description and aspect ratio extraction from slot

      This function extract the active format description and aspect ratio code from the provided slot.
      If LineNumber parameter is set to 0, all lines will be searched.

      @return The function returns the status of its execution as VideoMasterIP error code
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotExtractAFD(const HANDLE StreamHandle     /*!< [in] Handle of the stream to work on */
                                                             , const HANDLE SlotHandle     /*!< [in] Handle of the slot to work on */
                                                             , VMIP_AFD_AR_CODE* AfdArCode /*!< [out] AFD_AR code */
                                                             , uint32_t LineNumber         /*!< [in] line to search on */
   );

   /*!
      @brief Active format description and aspect ratio embed in slot

      This function embed the active format description and aspect ratio code in the provided slot.

      @return The function returns the status of its execution as VideoMasterIP error code
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotEmbedAFD(const HANDLE StreamHandle    /*!< [in] Handle of the stream to work on */
                                                         , const HANDLE SlotHandle    /*!< [in] Handle of the slot to work on */
                                                         , VMIP_AFD_AR_CODE AfdArCode /*!< [in] AFD_AR code */
                                                         , uint32_t LineNumber        /*!< [in] line to insert on */
   );

   /*!
      @brief OP47 data slot extraction

      This function extracts the OP47 data slot of the given slot.

      @return The function returns the status of its execution as VideoMasterIP error code
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotExtractOP47(const HANDLE StreamHandle /*!< [in] Handle of the stream to work on */
                                                              , const HANDLE SlotHandle /*!< [in] Handle of the slot to work on */
                                                              , VMIP_OP47_SLOT* pOP47   /*!< [out]  Pointer to a caller\-allocated structure receiving the OP47 slot */
                                                              , uint32_t LineNumber     /*!< [in] line to search on */
   );

   /*!
      @brief OP47 data embedding.

      This function embeds the provided OP47 data slot in the provided slot, according to the specified standard.

      @return The function returns the status of its execution as VideoMasterIP error code

      @note If parity bits are needed for the specified standard, they will be automatically generated.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotEmbedOP47(const HANDLE StreamHandle /*!< [in] Handle of the stream to work on */
                                                          , const HANDLE SlotHandle /*!< [in] Handle of the slot to work on */
                                                          , VMIP_OP47_SLOT* pOP47   /*!< [in] Pointer to a caller\-allocated structure containing the OP47 slot */
                                                          , uint32_t LineNumber     /*!< [in] line to insert on */
   );

   /*!
      @brief Closed captions slot extraction

      This function extracts the CC Slot of the given slot.

      @return The function returns the status of its execution as VideoMasterIP error code
      
      @note If LineNumber1 or LineNumber2 is set to 0, the standard default will be used.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotExtractClosedCaptions(const HANDLE StreamHandle         /*!< [in] Handle of the stream to work on */
                                                                       , const HANDLE SlotHandle         /*!< [in] Handle of the slot to work on */
                                                                       , VMIP_CLOSED_CAPTION_SLOT* pCC   /*!< [out] Pointer to a caller\-allocated structure receiving the CC slot */
                                                                       , VMIP_CLOSED_CAPTION_TYPE CCType /*!< [in] CC standard used */
                                                                       , uint32_t LineNumber1            /*!< [in] first line to search on */
                                                                       , uint32_t LineNumber2            /*!< [in] second line to search on */
   );

   /*!
      @brief Closed caption embedding.

      This function embeds the provided CC in the provided slot, according to the specified standard.

      @return The function returns the status of its execution as VideoMasterIP error code

      @note If parity bits are needed for the specified standard, they will be automatically generated.
      @note If LineNumber1 or LineNumber2 is set to 0, the standard default will be used.
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotEmbedClosedCaptions(const HANDLE StreamHandle         /*!< [in] Handle of the stream to work on */
                                                                   , const HANDLE SlotHandle         /*!< [in] Handle of the slot to work on */
                                                                   , VMIP_CLOSED_CAPTION_SLOT* pCC   /*!< [in] Pointer to a caller\-allocated structure containing the CC slot */
                                                                   , VMIP_CLOSED_CAPTION_TYPE CCType /*!< [in] CC standard used */
                                                                   , uint32_t LineNumber1            /*!< [in] first line to insert on */
                                                                   , uint32_t LineNumber2            /*!< [in] second line to insert on */
   );

   /*!
      @brief SMPTE RP 188 Timecode extraction
      
      This function extracts the SMPTE RP 188 timecode slot of the given slot.
      
      @return The function returns the status of its execution as VideoMasterIP error code
    */

   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotExtractTimecode(const HANDLE StreamHandle         /*!< [in] Handle of the stream to work on */
                                                                , const HANDLE SlotHandle         /*!< [in] Handle of the slot to work on */
                                                                , VMIP_TIMECODE_SLOT* pTimecode   /*!< [out] Reference to timecode slot */
                                                                , uint32_t LineNumber1            /*!< [in] first line to search on */
                                                                , uint32_t LineNumber2            /*!< [in] second line to search on */
                                                                , bool SampleYAndC                /*!< [in] Setting this to true result in looking for timecode in both Y and C streams. BufferType param is not used when this is set to true */
                                                                , uint32_t BufferType             /*!< [in] Buffer type of the stream to work on */
                                                                , VMIP_TIMECODE_TYPE TimecodeType /*!< [in] Timecode embedding standard */
   );

   /*!
      @brief SMPTE 266M Timecode embedding
      
      This function embeds the provided timecode slot in the provided slot, according to the RP 188 standard.
      
      @return The function returns the status of its execution as VideoMasterIP error code
    */

   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotEmbedTimecode(const HANDLE StreamHandle         /*!< [in] Handle of the stream to work on */
                                                              , const HANDLE SlotHandle         /*!< [in] Handle of the slot to work on */
                                                              , VMIP_TIMECODE_SLOT* pTimecode   /*!< [in] Reference to timecode slot */
                                                              , uint32_t LineNumber             /*!< [in] line to embed on */
                                                              , bool SampleYAndC                /*!< [in] Setting this to true result in looking for timecode in both Y and C streams. BufferType param is not used when this is set to true */
                                                              , uint32_t BufferType             /*!< [in] Buffer type of the stream to work on */
                                                              , VMIP_TIMECODE_TYPE TimecodeType /*!< [in] Timecode embedding standard */
   );

   /*!
      @brief HDR metadata slot extraction
      
      This function extracts the HDR metadata slot of the given slot.
      
      @return The function returns the status of its execution as VideoMasterIP error code
    */

   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotExtractHdrMetadata(const HANDLE StreamHandle              /*!< [in] Handle of the stream to work on */
                                                             , const HANDLE SlotHandle              /*!< [in] Handle of the slot to work on */
                                                             , VMIP_HDR_METADATA_SLOT* pHdrMetadata /*!< [out] Reference to HDR metadata slot */
   );
   /*!
       @brief HDR metadata slot embedding.
       
       This function embeds the provided HDR metadata slot in the provided slot.
       
       @return The function returns the status of its execution as VideoMasterIP error code
    */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotEmbedHdrMetadata(const HANDLE StreamHandle              /*!< [in] Handle of the stream to work on */
                                                           , const HANDLE SlotHandle              /*!< [in] Handle of the slot to work on */
                                                           , VMIP_HDR_METADATA_SLOT* pHdrMetadata /*!< [in] Reference to HDR metadata slot */
   );

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS

#endif //_VIDEOMASTERIP_ANC_PROCESSING_H_