/*!
   @file
   @brief This file gathers ANC related public definitions as enumerations, structures and functions

   Copyright (c) DELTACAST. All rights reserved.

   THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
   PURPOSE.
*/



#ifndef _VIDEOMASTERIP_ANC_H_
#define _VIDEOMASTERIP_ANC_H_

#include "videomasterip_core.h"

   /*!
      @brief VideoMasterIP supported ANC SDI video standard

      The VMIP_ANC_SDI_VIDEOSTANDARD enumeration lists all the supported SDI video standard.

      @see VMIP_STREAM_ESSENCE_CONFIG_ST2110_40
   */
typedef enum _VMIP_ANC_SDI_VIDEOSTANDARD
{
   VMIP_ANC_SDI_VIDEOSTD_720X480I59,         /*!< 720x480i @ 59Hz */
   VMIP_ANC_SDI_VIDEOSTD_720X487I59,         /*!< 720x487i @ 59Hz */
   VMIP_ANC_SDI_VIDEOSTD_720X576I50,         /*!< 720x576i @ 50Hz */
   VMIP_ANC_SDI_VIDEOSTD_1280X720P50,        /*!< 1280x720p @ 50Hz */
   VMIP_ANC_SDI_VIDEOSTD_1280X720P59,        /*!< 1280x720p @ 59Hz */
   VMIP_ANC_SDI_VIDEOSTD_1280X720P60,        /*!< 1280x720p @ 60Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080I50,       /*!< 1920x1080i @ 50Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080I59,       /*!< 1920x1080i @ 59Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080I60,       /*!< 1920x1080i @ 60Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080P23,       /*!< 1920x1080p @ 23Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080P24,       /*!< 1920x1080p @ 24Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080P25,       /*!< 1920x1080p @ 25Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080P29,       /*!< 1920x1080p @ 29Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080P30,       /*!< 1920x1080p @ 30Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080P50,       /*!< 1920x1080p @ 50Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080P59,       /*!< 1920x1080p @ 59Hz */
   VMIP_ANC_SDI_VIDEOSTD_1920X1080P60,       /*!< 1920x1080p @ 60Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P23,       /*!< 2048x1080p @ 23Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P24,       /*!< 2048x1080p @ 24Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P25,       /*!< 2048x1080p @ 25Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P29,       /*!< 2048x1080p @ 29Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P30,       /*!< 2048x1080p @ 30Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P47,       /*!< 2048x1080p @ 47Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P48,       /*!< 2048x1080p @ 48Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P50,       /*!< 2048x1080p @ 50Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P59,       /*!< 2048x1080p @ 59Hz */
   VMIP_ANC_SDI_VIDEOSTD_2048X1080P60,       /*!< 2048x1080p @ 60Hz */
   VMIP_ANC_SDI_VIDEOSTD_3840X2160P23,       /*!< 3840x2160p @ 23Hz */
   VMIP_ANC_SDI_VIDEOSTD_3840X2160P24,       /*!< 3840x2160p @ 24Hz */
   VMIP_ANC_SDI_VIDEOSTD_3840X2160P25,       /*!< 3840x2160p @ 25Hz */
   VMIP_ANC_SDI_VIDEOSTD_3840X2160P29,       /*!< 3840x2160p @ 29Hz */
   VMIP_ANC_SDI_VIDEOSTD_3840X2160P30,       /*!< 3840x2160p @ 30Hz */
   VMIP_ANC_SDI_VIDEOSTD_3840X2160P50,       /*!< 3840x2160p @ 50Hz */
   VMIP_ANC_SDI_VIDEOSTD_3840X2160P59,       /*!< 3840x2160p @ 59Hz */
   VMIP_ANC_SDI_VIDEOSTD_3840X2160P60,       /*!< 3840x2160p @ 60Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P23,       /*!< 4096x2160p @ 23Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P24,       /*!< 4096x2160p @ 24Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P25,       /*!< 4096x2160p @ 25Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P29,       /*!< 4096x2160p @ 29Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P30,       /*!< 4096x2160p @ 30Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P47,       /*!< 4096x2160p @ 47Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P48,       /*!< 4096x2160p @ 48Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P50,       /*!< 4096x2160p @ 50Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P59,       /*!< 4096x2160p @ 59Hz */
   VMIP_ANC_SDI_VIDEOSTD_4096X2160P60,       /*!< 4096x2160p @ 60Hz */
   VMIP_ANC_SDI_VIDEOSTD_1280X720P23,        /*!< 1280x720p @ 23Hz */
   VMIP_ANC_SDI_VIDEOSTD_1280X720P24,        /*!< 1280x720p @ 24Hz */
   VMIP_ANC_SDI_VIDEOSTD_1280X720P25,        /*!< 1280x720p @ 25Hz */
   VMIP_ANC_SDI_VIDEOSTD_1280X720P29,        /*!< 1280x720p @ 29Hz */
   VMIP_ANC_SDI_VIDEOSTD_1280X720P30,        /*!< 1280x720p @ 30Hz */
   NB_VMIP_ANC_SDI_VIDEOSTANDARDS
} VMIP_ANC_SDI_VIDEOSTANDARD;

/*!
   @brief Ancillary data packet content.

   The VMIP_ANC_PACKET_CONTENT structure abstracts an ancillary data packet.

   @see VMIP_SlotAncGetPacket and VMIP_SlotAncInsertPacket
*/

typedef struct _VMIP_ANC_PACKET_CONTENT
{
   uint8_t  DataID;              /*!< 8 LSB of ANC packet Data ID field */
   union
   {
      uint8_t  DataBlockNumber;  /*!< 8 LSB of ANC packet Data Block Number field (Type 1 packets) */
      uint8_t  SecondaryDataID;  /*!< 8 LSB of ANC packet Secondary Data ID field (Type 2 packets) */
   };									
   uint8_t  DataCount;           /*!< Number of User Data words in this packet */
   uint16_t* pUserDataWords;     /*!< Pointer to ANC packet User Data Words */
   bool8_t  InHANC;             /*!< TRUE if this packet concerns horizontal blanking space, FALSE if located in horizontal active space. Please note that this only concerns packets of VANC space since during active lines, ANC are only allowed during horizontal blanking */
} VMIP_ANC_PACKET_CONTENT;

#ifndef EXCLUDE_API_FUNCTIONS

#ifdef __cplusplus
extern "C" {
#endif

   /*!
      @brief Number of ancillary data packets on a line query

      This function gives the number of ancillary data packets found on the specified line number of the provided slot.

      @return The function returns the status of its execution as error code.

      @note When working in SD, the BufferType argument value must be VMIP_ST2110_40_BT_ANC_C_1, since Y and C samples are multiplexed together to carry on ancillary data packets
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotAncGetNbPackets(const HANDLE StreamHandle   /*!< [in] Handle of the stream to work on*/
                                                              , const HANDLE SlotHandle   /*!< [in] Handle of the slot to work on*/
                                                              , const uint32_t LineNumber /*!< [in] Video line number to get count from (1..N, with N depending on the standard)*/
                                                              , const uint32_t BufferType /*!< [in] Buffer type of the stream to work on*/
                                                              , uint32_t* pNbAncPackets   /*!< [out] Pointer to caller\-allocated variable receiving the ancillary data packets count*/
   );

   /*!
      @brief Ancillary data packet retrieval

      This function provides the specified ancillary data packet regarding the provided slot.
   
      @return The function returns the status of its execution as error code.

      @note When working in SD, the BufferType argument value must be VMIP_ST2110_40_BT_ANC_C_1, since Y and C samples are multiplexed together to carry on ancillary data packets

      @see VMIP_LockSlot VMIP_SlotAncGetNbPackets
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotAncGetPacket(const HANDLE StreamHandle               /*!< [in] Handle of the stream to work on*/
                                                           , const HANDLE SlotHandle               /*!< [in] Handle of the slot to work on*/
                                                           , const uint32_t LineNumber             /*!< [in] Video line number to get count from (1..N, with N depending on the standard) */
                                                           , const uint32_t BufferType             /*!< [in] Buffer type of the stream to work on*/
                                                           , const int32_t PacketIndex             /*!< [in] Index (zero\-based) of the packet to retrieve*/
                                                           , VMIP_ANC_PACKET_CONTENT** ppAncPacket /*!< [out] Pointer receiving a pointer to allocated ANC packet structure */
   );

   /*!
      @brief Ancillary data packet allocation

      This function allocates memory for a new ancillary data packet. The allocated memory is attached to a slot.
   
      @return The function returns the status of its execution as error code.

      @note
            - Once a new packet has been allocated and properly filled in, it may be inserted on a specific video line using the VMIP_SlotAncInsertPacket function.  
            - Allocated memory is automatically released by VideoMasterIP once the ancillary data slot is unlocked.  
            - User data words of allocated packets are all reset to zero by this function.
            - Although the allocated packet will contain DataCount user data words, the caller may freely set the actual DataCount
              field of the packet structure to a value less or equal to it in before subsequent call to VMIP_SlotAncInsertPacket.

      @see VMIP_LockSlot VMIP_SlotAncInsertPacket
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotAncAllocatePacket(const HANDLE StreamHandle  /*!< [in] Handle of the stream to work on*/
      , const HANDLE SlotHandle                                                            /*!< [in] Handle of the slot to work on*/
      , const uint8_t DataCount                                                            /*!< [in] Number of User Data Words to allocate*/
      , VMIP_ANC_PACKET_CONTENT** ppAncPacket                                              /*!< [out] Pointer receiving a pointer to allocated ANC packet structure*/
   );

   /*!
      @brief Ancillary data packet insertion

      This function inserts the given ancillary data packet at the specified line within the provided slot.

      @return The function returns the status of its execution as error code.

      @note
            - When working in SD, the AncStream argument is ignored, since Y and C samples are multiplexed together to carry on ancillary data packets.
            - Packet insertion automatically release allocated packet, so that the pAncPacket pointer cannot be re-used without any call to VMIP_SlotAncAllocatePacket.
            - When inserting a packet on a line already containing ancillary data, the PacketIndex argument let you decide where to place the new packet.  
              It is related to other packets already present in the same zone as the packet to insert, that is in HANC space or in active part of the line.  
              For example, if a line already contains packets A,B and C in HANC space, inserting a new HANC packet N at location 1 will give the following result : A,N,B,C.

      @see VMIP_LockSlot VMIP_SlotAncAllocatePacket
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotAncInsertPacket(const HANDLE StreamHandle                   /*!< [in] Handle of the stream to work on*/
                                                              , const HANDLE SlotHandle                   /*!< [in] Handle of the slot to work on*/
                                                              , const uint32_t LineNumber                 /*!< [in] Video line number to get count from (1..N, with N depending on the standard) */
                                                              , const uint32_t BufferType                 /*!< [in] Buffer type of the stream to work on*/
                                                              , const int32_t PacketIndex                 /*!< [in] Position (zero\-based) to put this packet at (\-1 to insert at last location)*/
                                                              , const VMIP_ANC_PACKET_CONTENT* pAncPacket /*!< [in] Pointer to ancillary data packet to insert*/
   );

   /*!
      @brief Ancillary data packet removal

      This function remove the specified ancillary data packet from the specified line on the provided slot.

      @return The function returns the status of its execution as error code.

      @note When working in SD, the AncStream argument is ignored, since Y and C samples are multiplexed together to carry on ancillary data packets.

      @see VMIP_LockSlot VMIP_SlotAncGetNbPackets
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_SlotAncRemovePacket(const HANDLE StreamHandle   /*!< [in] Handle of the stream to work on*/
                                                              , const HANDLE SlotHandle   /*!< [in] Handle of the slot to work on*/
                                                              , const uint32_t LineNumber /*!< [in] Video line number to get count from (1..N, with N depending on the standard) */
                                                              , const uint32_t BufferType /*!< [in] Buffer type of the stream to work on*/
                                                              , const int32_t PacketIndex /*!< [in] Position (zero\-based) of the packet to remove (\-1 to remove all packets from line)*/
                                                              , const bool8_t InHANC      /*!< [in] Specifies if PacketIndex is related to HANC zone or to active line zone.
                                                                                                    If PacketIndex is -1, then this argument is ignored and the whole ANC line is erased*/
   );

   /*!
   @brief Gets information regarding a ANC video standard

   Retrieves the information associated to the given video standard
   
   @note The properties related to 3840/4096p formats are the same as their 1920/2048 equivalent. This is due to the fact that no RAW frame exists for those format, instead, 4 smaller RAW frames are used, as described in the norm ST2082-10.

   @returns
   The function returns the status of its execution as a Vmip error code.
   Errors can happen if :
   - The video standard is not supported;
   - One of the pointer is null.

   @see VMIP_ANC_SDI_VIDEOSTANDARD
   */
   VIDEOMASTER_IP_API VMIP_ERRORCODE VMIP_GetAncSdiVideoStandardInfo(VMIP_ANC_SDI_VIDEOSTANDARD VidStd /*!< [in] Video standard to get information from*/
      , uint32_t* pRawFrameWidth                                                                       /*!< [out] Pointer to a caller\-allocated variable receiving the RAW frame Width*/
      , uint32_t* pActiveFrameWidth                                                                    /*!< [out] Pointer to a caller\-allocated variable receiving the active frame Width*/
      , uint32_t* pRawFrameHeight                                                                      /*!< [out] Pointer to a caller\-allocated variable receiving the active RAW height*/
      , uint32_t* pActiveFrameHeight                                                                   /*!< [out] Pointer to a caller\-allocated variable receiving the active frame height*/
      , uint32_t* pFramerate                                                                           /*!< [out] Pointer to a caller\-allocated variable receiving the frame rate*/
      , bool8_t* pInterlaced                                                                           /*!< [out] Pointer to a caller\-allocated variable receiving the interlaced information*/
      , bool8_t* pIsUs                                                                                 /*!< [out] Pointer to a caller\-allocated variable telling if the video standard is US or not*/
   );

#ifdef __cplusplus
}
#endif

#endif //EXCLUDE_API_FUNCTIONS


#endif //_VIDEOMASTERIP_ANC_H_