#!/usr/bin/env python3
import platform
import subprocess
from sys import exit
import os
import shutil
import tempfile
from pathlib import Path
import ctypes
import tarfile
from zipfile import ZipFile
import argparse
import re
import glob

'''
This script is intended to install the IPVC seamlessly on Linux and Windows.
'''

is_dev_package = False
is_protected_package = False
package_root_dir = os.getcwd()
vcs_linux_dir = '/opt/deltacast/vcs/'
vcs_windows_dir = "C:/Program Files/DELTACAST/VCS/"
vmip_windows_dir = "C:/Program Files/DELTACAST/VideoMasterIP/"
licensing_windows_dir = "C:/Program Files/DELTACAST/Licensing/"
configure_script_name = 'ipvc_configure'
linux_vmip_configuration_file = "/etc/videomasterip.conf"
windows_key_path = r"Software\Deltacast\VideoMasterIP"

if os.path.isdir('/usr/local/lib64'):
   lib_dir = '/usr/local/lib64'
else:
   lib_dir = '/usr/local/lib'

# HELPER FUNC
def package_type():
   if is_dev_package:
      return "dev"
   else:
      return "redist"
   
def is_windows():
   if platform.system() == "Windows":
      return True
   else:
      return False
   
def run_os_specific_func(func):
   def wrapper(*args, **kwargs):
      try:
         os_func = globals()[func.__name__+"_"+platform.system()]
         return os_func(*args, **kwargs)
      except KeyError:
         print("Os not supported")
   return wrapper

def exec_cmd_Linux(command: str, path=Path(".")):
   return subprocess.run([command], stdout=subprocess.PIPE, stderr=subprocess.STDOUT, shell=True,cwd=path)

def exec_cmd_Windows(command: str, path=Path(".")):
   return subprocess.run(["powershell",command], stdout=subprocess.PIPE, stderr=subprocess.STDOUT, shell=True, cwd=path)

@run_os_specific_func
def exec_cmd(command: str, path=Path(".")):
   pass
   
def display_vcs_version_Linux():
   try:
      result = subprocess.run(['/opt/deltacast/videomasterip/vcs/vcs','--version'], stdout=subprocess.PIPE, check=True)
      print(f"VCS installed version : {result.stdout.decode().strip()}")
   except Exception as e:
      print(f"ERROR: Displaying VCS version - {e}")

def display_vcs_version_Windows():
   pass

@run_os_specific_func
def display_vcs_version():
   pass

# SCRIPT FUNC

def check_access_rights_Linux():
   if os.getuid() != 0:
      print("Please run as root")
      exit(1)

def check_access_rights_Windows():
   if ctypes.windll.shell32.IsUserAnAdmin() == 0:
      print("Please run as administrator")
      exit(1)

@run_os_specific_func
def check_access_rights():
   pass

def check_package_common(required_folders):
   required_folders += ["vcs","videomasterip"]
   optional_dev_folders = ["documentation","examples"]
   global is_dev_package
   is_dev_package = True

   for folder in optional_dev_folders:
      if not os.path.exists(folder):
         is_dev_package = False
   folder_missing = False

   global is_protected_package
   if os.path.exists("licensing"):
         is_protected_package = True

   for folder in required_folders:
    if not os.path.exists(folder):
        print(f"Error: Required folder {folder} not found!")
        folder_missing = True
   if folder_missing:
      print(f"Error: the shell script should be executed from IP Virtual Card {platform.system()} {package_type()} folder or the package is corrupted")
      exit(1)

def check_package_Linux():
   required_folders = ["kbdpdk","linuxptp"]
   check_package_common(required_folders)

def check_package_Windows():
   required_folders = ["redist","deltaptp"]
   check_package_common(required_folders)


@run_os_specific_func
def check_package():
   pass

def check_common_tools_Linux(dpdk_mode=False):
   tools_to_check = ["g++","make","git","patch"]
   if dpdk_mode:
      tools_to_check = tools_to_check + ["meson","ninja", "pkg-config"]
   results = {tool: exec_cmd(tool+" --version").returncode == 0 for tool in tools_to_check}
   for key in results:
      if results[key] is False:
         print(f"ERROR : {key} is missing")
   if not all(results[result] == True for result in results):
      print("ERROR : common tools check failed")
      exit(1)

def check_common_tools_Windows(dpdk_mode=False):
   pass

@run_os_specific_func
def check_common_tools(dpdk_mode=False):
   pass

def install_ptp_Linux():
   try:
      # Disabling NTP
      subprocess.run(["timedatectl", "set-ntp", "0"], check=True)

      # Build+patch linuxptp instead of installing pre-built package
      ptp_version = "2.0"
      tmpdir = tempfile.mkdtemp()
      os.chdir(tmpdir)

      # Clone linuxptp from git
      print("Cloning linuxptp repo...")
      subprocess.run(["git", "clone","http://git.code.sf.net/p/linuxptp/code", "linuxptp"],stdout=subprocess.DEVNULL,stderr=subprocess.DEVNULL, check=True)
      os.chdir("linuxptp")
      subprocess.run(["git", "checkout", "-b", ptp_version, f"v{ptp_version}"],stdout=subprocess.DEVNULL,stderr=subprocess.DEVNULL, check=True)

      # Apply patches
      print("Patching linuxptp...")
      patch_files = ["donotrespondst2059_2tlv.patch", "kni.patch"]
      for patch_file in patch_files:
         subprocess.run(["patch", "-p1","-i", f"{package_root_dir}/linuxptp/{patch_file}"],stdout=subprocess.DEVNULL, check=True)

      # Build and install linuxptp
      print("Building linuxptp...")
      subprocess.run(["make"],stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL, check=True)
      print("Installing linuxptp...")
      subprocess.run(["make", "install"],stdout=subprocess.DEVNULL, check=True)

      # Cleanup
      print("Cleaning linuxptp install files...")
      subprocess.run(["make", "distclean"],stdout=subprocess.DEVNULL, check=True)

      # Install ptp4l.conf
      print("Installing ptp4l.conf...")
      os.makedirs("/etc/linuxptp",exist_ok=True)
      shutil.copy(f"{package_root_dir}/linuxptp/ptp4l.conf", "/etc/linuxptp")
      os.chmod("/etc/linuxptp/ptp4l.conf", 0o644)
   except subprocess.CalledProcessError as e:
      print(f"ERROR: PTP installation - {e}")
      exit(1)

   if os.path.exists(tmpdir):
      shutil.rmtree(tmpdir)

   os.chdir(package_root_dir)

def install_ptp_Windows():
   print("Installing DeltaPTP...")
   try:
      subprocess.run([f"{package_root_dir}/deltaptp/Setup.exe","/VERYSILENT"],stdout=subprocess.DEVNULL,check=True)
   except Exception as e:
      print(f"ERROR: PTP installation - {e}")
      exit(1)

def uninstall_ptp_Windows():
   print("Uninstalling DeltaPTP...")
   try:
      uninstall_exe = subprocess.check_output(['powershell.exe', '-Command', "get-package '*Delta PTP*' -erroraction 'silentlycontinue'| % { $_.metadata['uninstallstring'] }"])
   except subprocess.CalledProcessError as e:
      print(f"ERROR: Delta PTP is not installed on the system")
      return
   try:
      subprocess.run(['powershell.exe', '-Command','Start-Process','-FilePath',uninstall_exe.decode().strip(),'-ArgumentList','/VERYSILENT', '-Wait'], check=True)
   except Exception as e:
      print(f"ERROR: PTP installation - {e}")
      exit(1)

def install_redist_Windows():
   print("Installing Visual Studio redist...")
   try:
      subprocess.run([f"{package_root_dir}/redist/vc_redist.x64.exe",'/install','/quiet', '/norestart'], check=True)
   except subprocess.CalledProcessError as e:
      if e.returncode != 1638:
         print(f"ERROR: Visual Studio redist installation - {e}")
   except Exception as e:
         print(f"ERROR: Visual Studio redist installation - {e}")
      
def install_tools_Linux():
   install_ptp_Linux()

def install_tools_Windows():
   install_ptp_Windows()
   install_redist_Windows()

@run_os_specific_func
def install_tools():
   pass

def uninstall_tools_Linux():
   pass

def uninstall_tools_Windows():
   uninstall_ptp_Windows()
   pass

@run_os_specific_func
def uninstall_tools():
   pass

def install_kbdpdk_Linux():
   print("Installing KBDPDK...")
   try:
      tmpdir = tempfile.mkdtemp()
      os.chdir(tmpdir)
      for file in os.listdir(f"{package_root_dir}/kbdpdk/"):
         if re.search("kbdpdk\..*\.tar\.gz",file):
            archive_name = file
            break
      archive_path = f"{package_root_dir}/kbdpdk/{archive_name}"
      with tarfile.open(archive_path, "r:gz") as tar:
        tar.extractall()
      os.chdir(os.path.join(tmpdir,"lib","DPDK"))
      args = ["sh","install_dpdk.sh","mlnx5"]
      subprocess.run(args, check=True)
      os.chdir(os.path.join(tmpdir,"lib"))
      subprocess.run(["make"],stdout=subprocess.DEVNULL, check=True)
      subprocess.run(["make","install_all"],stdout=subprocess.DEVNULL, check=True)
      os.chdir(package_root_dir)
      shutil.rmtree(tmpdir)
   except Exception as e:
      print(f"ERROR: KBDPDK installation - {e}")
      exit(1)


def install_kbdpdk_Windows():
   print("Installing KBDPDK...")
   try:
      tmpdir = tempfile.mkdtemp()
      os.chdir(tmpdir)
      for file in os.listdir(f"{package_root_dir}/kbdpdk/"):
         if re.search("kbdpdk\..*\.zip",file):
            archive_name = file
            break
      archive_path = f"{package_root_dir}/kbdpdk/{archive_name}"
      with ZipFile(archive_path) as zip_file:
        zip_file.extractall()
      DevconPath = os.path.join(tmpdir, "tools", "devcon.exe")
      os.chdir(os.path.join(tmpdir,"drivers"))
      with ZipFile("netuio.zip") as zip_file:
        zip_file.extractall(path="netuio/")
      with ZipFile("tap-windows6.zip") as zip_file:
        zip_file.extractall(path="tap-windows6/")
      with ZipFile("virt2phys.zip") as zip_file:
        zip_file.extractall(path="virt2phys/")

      args = [DevconPath,"dp_add",os.path.join(tmpdir,"drivers","tap-windows6", "bin", "x64", "delta-tap.inf")]
      subprocess.run(args, stdout=subprocess.DEVNULL, check=True)
      args = [DevconPath,"dp_add",os.path.join(tmpdir,"drivers","netuio", "netuio.inf")]
      subprocess.run(args, stdout=subprocess.DEVNULL, check=True)
      args = [DevconPath,"install",os.path.join(tmpdir,"drivers","virt2phys", "virt2phys.inf"), "root\\virt2phys"]
      subprocess.run(args, stdout=subprocess.DEVNULL, check=True)

      shutil.copytree(os.path.join(tmpdir,"tools"), vcs_windows_dir, dirs_exist_ok=True)
      shutil.copytree(os.path.join(tmpdir,"drivers", "netuio"), os.path.join(vcs_windows_dir, "netuio"), dirs_exist_ok=True)
      shutil.copytree(os.path.join(tmpdir,"drivers", "tap-windows6", "bin", "x64"), vcs_windows_dir, dirs_exist_ok=True)
      shutil.copy2(os.path.join(tmpdir,"lib","libKBAPI.dll"), vcs_windows_dir)
      shutil.copy2(os.path.join(tmpdir,"config","kbapi.conf"), vcs_windows_dir)

      os.chdir(package_root_dir)
      shutil.rmtree(tmpdir)

   except Exception as e:
      print(f"ERROR: KBDPDK installation - {e}")
      exit(1)

@run_os_specific_func
def install_kbdpdk():
   pass

def uninstall_kbdpdk_Linux():
   pass

def uninstall_kbdpdk_Windows():
   print("Uninstalling KBDPDK...")
   try:
      DevconPath = os.path.join(vcs_windows_dir, "devcon.exe")
      if os.path.exists(DevconPath):
         args = [DevconPath,"removeall", "delta-tap"]
         subprocess.run(args, stdout=subprocess.DEVNULL, check=True)
         args = [DevconPath,"removeall", "root\\virt2phys"]
         subprocess.run(args, stdout=subprocess.DEVNULL, check=True)
         os.remove(DevconPath)
      shutil.rmtree(os.path.join(vcs_windows_dir, "netuio"),ignore_errors=True)
      for f in glob.glob(os.path.join(vcs_windows_dir, "delta-tap*")):
         os.remove(f)
      for f in glob.glob(os.path.join(vcs_windows_dir, "kb*.conf")):
         os.remove(f)
      for f in glob.glob(os.path.join(vcs_windows_dir, "kb*.exe")):
         os.remove(f)
      if os.path.exists(os.path.join(vcs_windows_dir, "libKBAPI.dll")):
         os.remove(os.path.join(vcs_windows_dir, "libKBAPI.dll"))

   except Exception as e:
      print(f"ERROR: KBDPDK uninstallation - {e}")
      exit(1)

@run_os_specific_func
def uninstall_kbdpdk():
   pass

def install_libs_Linux():
   print("Installing VideomasterIP libs...")
   try:
      for file in os.listdir(f"{package_root_dir}/videomasterip"):
         if os.path.isfile(os.path.join(f"{package_root_dir}/videomasterip", file)):
            lib_path = os.path.join(f"{package_root_dir}/videomasterip", file)
            os.chmod(lib_path, 0o644)
            shutil.copy2(lib_path, lib_dir,follow_symlinks=False)

      if lib_dir not in open('/etc/ld.so.conf.d/libc.conf').read():
         with open('/etc/ld.so.conf.d/libc.conf', 'a') as f:
            f.write(lib_dir + '\n')

      subprocess.run(["/sbin/ldconfig", "-n",lib_dir],stdout=subprocess.DEVNULL, check=True)
      subprocess.run(["/sbin/ldconfig", "-X"],stdout=subprocess.DEVNULL, check=True)
   except Exception as e:
      print(f"ERROR: VideomasterIP libs installation - {e}")
      exit(1)

   if is_dev_package:
      print("Installing VideomasterIP headers...")
      try:
         headers_dir = "/usr/local/include/videomasterip"
         os.makedirs(headers_dir, exist_ok=True)

         src_dir = f"{package_root_dir}/videomasterip/headers"
         for header_file in os.listdir(src_dir):
            src_path = os.path.join(src_dir, header_file)
            dst_path = os.path.join(headers_dir, header_file)
            shutil.copy2(src_path, dst_path)
            os.chmod(dst_path, 0o644)
      except Exception as e:
         print(f"ERROR: VideomasterIP headers installation - {e}")
         exit(1)

def install_libs_Windows():
   if is_dev_package:
      print("Installing VideomasterIP libs...")
      try:
         lib_pathes = [("videomasterip","lib/x64"),("videomasterip/headers","videomasterip"),("videomasterip/lib","lib/x64")]

         for src,dst in lib_pathes:
            os.makedirs(os.path.join(vmip_windows_dir,dst),exist_ok=True)
            for filename in os.listdir(src):
               src_file = os.path.join(package_root_dir,src, filename)
               dst_file = os.path.join(vmip_windows_dir,dst, filename)
               if os.path.isfile(src_file):
                  shutil.copy2(src_file, dst_file)

         subprocess.run(['powershell', f'[System.Environment]::SetEnvironmentVariable("VIDEOMASTERIP", "{vmip_windows_dir}", "Machine")'], check=True)
         subprocess.run(['powershell', f'[System.Environment]::SetEnvironmentVariable("VIDEOMASTERIP_LIB", "{os.path.join(vmip_windows_dir,"lib")}", "Machine")'], check=True)

      except Exception as e:
         print(f"ERROR: VideomasterIP libs installation - {e}")
         exit(1)


@run_os_specific_func
def install_libs():
   pass

def uninstall_libs_Linux():
   print("Uninstalling VideomasterIP libs..")
   try:
      libnames = ["videomasterip","datapipe"]
      for libname in libnames:
         lib_files = [file for file in os.listdir(lib_dir) if libname in file]
         for file in lib_files:
            os.remove(os.path.join(lib_dir,file))
   except Exception as e:
      print(f"ERROR: VideomasterIP libs uninstallation - {e}")
      exit(1)
   
   shutil.rmtree("/usr/local/include/videomasterip",ignore_errors=True)

def uninstall_libs_Windows():
   print("Uninstalling VideomasterIP libs..")
   shutil.rmtree(vmip_windows_dir,ignore_errors=True)
   subprocess.run(['powershell', '[Environment]::SetEnvironmentVariable("VIDEOMASTERIP", $null, "Machine")'])
   subprocess.run(['powershell', '[Environment]::SetEnvironmentVariable("VIDEOMASTERIP_LIB", $null, "Machine")'])

@run_os_specific_func
def uninstall_libs():
   pass

def install_vcs_Linux():
   print("Installing VCS...")
   try:
      vcs_package_dir = f"{package_root_dir}/vcs/"

      os.makedirs(vcs_linux_dir, exist_ok=True)

      # install VCS files
      for file in os.listdir(vcs_package_dir):
         shutil.copy2(os.path.join(vcs_package_dir, file), vcs_linux_dir)
         os.chmod(os.path.join(vcs_linux_dir, file), 0o755)

   except Exception as e:
      print(f"ERROR: VCS installation - {e}")
      exit(1)

def install_vcs_Windows():
   print("Installing VCS...")
   try:
      vcs_package_dir = f"{package_root_dir}/vcs/"
      os.makedirs(vcs_windows_dir,exist_ok=True)

      for item in os.listdir(vcs_package_dir):
         source_path = os.path.join(vcs_package_dir, item)
         destination_path = os.path.join(vcs_windows_dir, item)
         shutil.copy2(source_path, destination_path)
   except Exception as e:
         print(f"ERROR: VCS installation - {e}")
         exit(1)

@run_os_specific_func
def install_vcs():
   pass

def uninstall_vcs_Linux():
   print("Uninstalling VCS...")
   try:
      shutil.rmtree(vcs_linux_dir)
   except Exception as e:
      print(f"ERROR: VCS uninstallation - {e}")
      exit(1)

def uninstall_vcs_Windows():
   print("Uninstalling VCS...")
   try:
      shutil.rmtree(vcs_windows_dir)
   except Exception as e:
      print(f"ERROR: VCS uninstallation - {e}")
      exit(1)

@run_os_specific_func
def uninstall_vcs():
   pass

def install_configure_script_Linux():
   print("Installing configure script...")
   try:
      shutil.copy(os.path.join(package_root_dir,configure_script_name),vcs_linux_dir)
      os.chmod(os.path.join(vcs_linux_dir, configure_script_name), 0o755)
      os.symlink(os.path.join(vcs_linux_dir, configure_script_name),os.path.join('/usr/local/sbin', configure_script_name))
   except Exception as e:
      print(f"ERROR: configure script installation - {e}")
      exit(1)

def install_configure_script_Windows():
   print("Installing configure script...")
   try:
      shutil.copy(os.path.join(package_root_dir,configure_script_name),vcs_windows_dir)
   except Exception as e:
      print(f"ERROR: configure script installation - {e}")
      exit(1)

@run_os_specific_func
def install_configure_script():
   pass

def uninstall_configure_script_Linux():
   print("Uninstalling configure script...")
   try:
      os.remove(os.path.join(vcs_linux_dir,configure_script_name))
      os.remove(os.path.join('/usr/local/sbin',configure_script_name))
   except FileNotFoundError:
      pass
   except Exception as e:
      print(f"ERROR: configure script uninstallation - {e}")

def uninstall_configure_script_Windows():
   print("Uninstalling configure script...")
   try:
      os.remove(os.path.join(vcs_windows_dir,configure_script_name))
   except FileNotFoundError:
      pass
   except Exception as e:
      print(f"ERROR: configure script uninstallation - {e}")

@run_os_specific_func
def uninstall_configure_script():
   pass

class service_operation():
   install = "--install"
   uninstall = "--uninstall"
   start = "--start"
   stop = "--stop"

def vcs_service_windows(operation):

   check_return_status = True
   if operation is service_operation.install:
      operation_str = "Installing"
   elif operation is service_operation.uninstall:
      operation_str = "Uninstalling"
   elif operation is service_operation.start:
      operation_str = "Starting"
   elif operation is service_operation.stop:
      operation_str = "Stopping"
      check_return_status = False
   print(f"{operation_str} VCS service...")

   try:
      subprocess.run([os.path.join(vcs_windows_dir,"vcs.exe"), operation],check=check_return_status)
   except FileNotFoundError as e:
      print(f"ERROR : VCS executable not found")
      return
   except Exception as e:
      print(f"ERROR: VCS service {operation_str} - {e}")
      exit(1)

def install_services_Linux():
   print("Installing services...")
   services_dir = "/lib/systemd/system/"
   services_to_start = ["vcs","ptp4l","network_sysctl"]
   services_src = [f"{package_root_dir}/vcs/vcs.service",
               f"{package_root_dir}/vcs/network_sysctl.service",
               f"{package_root_dir}/linuxptp/ptp4l.service",
               f"{package_root_dir}/linuxptp/phc2sys.service"]
   
   try:
      for service in services_src:
         service_name = Path(service).name
         shutil.copy2(service,services_dir)
         os.chmod(Path(services_dir).joinpath(service_name), 0o644)
         subprocess.run(["systemctl", "enable", service_name], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL, check=True)

      subprocess.run(['systemctl', 'daemon-reload'], stdout=subprocess.DEVNULL, check=True)

      for service in services_to_start:
         subprocess.run(['systemctl', 'start',service], stdout=subprocess.DEVNULL, check=True)
   except Exception as e:
      print(f"ERROR: Services installation - {e}")
      exit(1)

def install_services_Windows():
   vcs_service_windows(service_operation.install)
   vcs_service_windows(service_operation.start)

@run_os_specific_func
def install_services():
   pass

def stop_services_Linux():
   print("Stopping services...")
   try:
      services = ['vcs','network_sysctl','ptp4l','phc2sys']
      for service in services :
         subprocess.run(['systemctl', 'disable', f'{service}.service'], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
         subprocess.run(['systemctl', 'stop', f'{service}.service'], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
   except Exception as e:
      print(f"ERROR: Stopping services - {e}")
      exit(1)

def stop_services_Windows():
   vcs_service_windows(service_operation.stop)

@run_os_specific_func
def stop_services():
   pass

def uninstall_services_Linux():
   print("Uninstalling services...")
   try:
      services = ['vcs','network_sysctl','ptp4l','phc2sys']
      for service in services :
         if os.path.exists(f'/lib/systemd/system/{service}.service'):
            os.remove(f'/lib/systemd/system/{service}.service')

      # Reload systemctl daemon and reset failed services
      subprocess.run(['systemctl', 'daemon-reload'], stdout=subprocess.DEVNULL, check=True)
      subprocess.run(['systemctl', 'reset-failed'], stdout=subprocess.DEVNULL, check=True)
   except Exception as e:
      print(f"ERROR: Services uninstallation - {e}")
      exit(1)

def uninstall_services_Windows():
   vcs_service_windows(service_operation.uninstall)

@run_os_specific_func
def uninstall_services():
   pass

def add_firewall_rule_Linux():
   pass

def add_firewall_rule_Windows():
   print("Adding firewall rule...")
   try:
      command = f'New-NetFirewallRule -DisplayName "Deltacast VCS" -Direction Inbound -Program "{os.path.normpath(os.path.join(vcs_windows_dir,"vcs.exe"))}" -RemoteAddress Any -Action Allow'
      subprocess.run(['powershell.exe', '-Command',command], stdout=subprocess.DEVNULL, check=True)
   except Exception as e:
      print(f"ERROR: Adding firewall rule - {e}")
      exit(1)

@run_os_specific_func
def add_firewall_rule():
   pass

def remove_firewall_rule_Linux():
   pass

def remove_firewall_rule_Windows():
   print("Removing firewall rule...")
   try:
      command = f'Remove-NetFirewallRule -DisplayName "Deltacast VCS" -ErrorAction Ignore'
      subprocess.run(['powershell.exe', '-Command',command], stdout=subprocess.DEVNULL)
   except Exception as e:
      print(f"ERROR: Removing firewall rule - {e}")
      exit(1)

@run_os_specific_func
def remove_firewall_rule():
   pass

class licensing_operation():
   install = 0
   update = 1
   uninstall = 3

def licensing_Linux(operation=licensing_operation.install):
   if operation is licensing_operation.update:
      operation_str = "Updating"
   elif operation is licensing_operation.uninstall:
      operation_str = "Uninstalling"
   elif operation is licensing_operation.install:
      operation_str = "Installing"
   print(f"{operation_str} licensing...")

   licensing_archive = f"{package_root_dir}/licensing/dlmcli.tar.gz"

   try:
      tmpdir = tempfile.mkdtemp()
      os.chdir(tmpdir)

      with tarfile.open(licensing_archive,"r:gz") as tar:
         tar.extractall()

      os.chdir(os.path.join(tmpdir, "x64"))

      args = ["sh","install_dlm.sh"]
      if operation is licensing_operation.update:
         args.append("update")
      if operation is licensing_operation.uninstall:
         args.append("clean")

      subprocess.run(args, stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL, check=True)

      os.chdir(package_root_dir)

      shutil.rmtree(tmpdir)
   except Exception as e:
      print(f"ERROR: {operation_str} licensing - {e}")
      exit(1)

def licensing_Windows(operation=licensing_operation.install):
   if operation is licensing_operation.install:
      licensing_Windows(licensing_operation.uninstall)
      operation_str = "Installing"
   elif operation is licensing_operation.uninstall:
      operation_str = "Uninstalling"
   elif operation is licensing_operation.update:
      licensing_Windows(licensing_operation.uninstall)
      licensing_Windows(licensing_operation.install)
      return
   print(f"{operation_str} licensing...")

   licensing_package_dir = f"{package_root_dir}/licensing/"
   try:
      if operation is licensing_operation.install:
         shutil.copytree(licensing_package_dir, licensing_windows_dir)

      if operation is licensing_operation.uninstall:
         shutil.rmtree(licensing_windows_dir, ignore_errors=True)

   except Exception as e:
      print(f"ERROR: {operation_str} licensing - {e}")
      exit(1)

@run_os_specific_func
def licensing():
   pass

def clean_deltacast_folder_common(folder_path):
   print("Removing DELTACAST folder...")
   try:
      shutil.rmtree(folder_path)
   except Exception as e:
      print(f"ERROR: Removing DELTACAST folder - {e}")

def clean_deltacast_folder_Linux():
   clean_deltacast_folder_common("/opt/deltacast/")

def clean_deltacast_folder_Windows():
   clean_deltacast_folder_common("C:\Program Files\DELTACAST")

@run_os_specific_func
def clean_deltacast_folder():
   pass

def display_vcs_version_Linux():
   try:
      result = subprocess.run([os.path.join(vcs_linux_dir,'vcs'),'--version'], stdout=subprocess.PIPE, check=True)
      print(f"VCS installed version : {result.stdout.decode().strip()}")
   except Exception as e:
      print(f"ERROR: Displaying VCS version - {e}")

def display_vcs_version_Windows():
   try:
      result = subprocess.run([os.path.join(vcs_windows_dir,'vcs.exe'),'--version'], stdout=subprocess.PIPE, check=True)
      print(f"VCS installed version : {result.stdout.decode().strip()}")
   except Exception as e:
      print(f"ERROR: Displaying VCS version - {e}")

@run_os_specific_func
def display_vcs_version():
   pass

def activate_vmip_logs_Linux():
   try:
      f = open(linux_vmip_configuration_file, "w")
      f.write('''DebugFilter=Emergency,Alert,Error,Warning
EventOutput=SysLog
DebugOutput=SysLog
''')
      f.close()
   except Exception as e:
      print(f"ERROR: Creating {linux_vmip_configuration_file} - {e}")

def activate_vmip_logs_Windows():
   try:
      import winreg
      key = winreg.CreateKeyEx(winreg.HKEY_LOCAL_MACHINE, windows_key_path, 0, winreg.KEY_WRITE)
      winreg.SetValueEx(key, "DebugFilter", 0, winreg.REG_SZ, "Emergency,Alert,Error,Warning")
      winreg.SetValueEx(key, "EventOutput", 0, winreg.REG_SZ, "DebugApiLog")
      winreg.SetValueEx(key, "DebugOutput", 0, winreg.REG_SZ, "DebugApiLog")
      winreg.CloseKey(key)
   except Exception as e:
      print(f"ERROR: Creating registry key to activate VMIP logs - {e}")

@run_os_specific_func
def activate_vmip_logs():
   pass

def deactivate_vmip_logs_Linux():
   try:
      os.remove(linux_vmip_configuration_file)
   except Exception as e:
      if(e.errno != 2): #Error 2 means that the file does not exist
         print(f"ERROR: Deleting {linux_vmip_configuration_file} - {e}")

def deactivate_vmip_logs_Windows():
   try:
      import winreg
      key = winreg.DeleteKey(winreg.HKEY_LOCAL_MACHINE, windows_key_path)
   except Exception as e:
      if(e.winerror != 2): #Error 2 means that the key does not exist
         print(f"ERROR: deleting registry key to deactivate VMIP logs - {e}")

@run_os_specific_func
def deactivate_vmip_logs():
   pass

# SCRIPT START

parser = argparse.ArgumentParser(description='IPVC installation script')
mode_group = parser.add_argument_group()
mode_group.description = 'Choose the operation you want to execute, default operation is --install'
mode_exclusive_group = mode_group.add_mutually_exclusive_group()
mode_exclusive_group.add_argument('-i', '--install', action='store_const', dest='mode', const='install', default='install', help='Install IPVC on your system')
if not is_windows():
   mode_exclusive_group.add_argument('-u', '--update', action='store_const', dest='mode', const='update', help='Update existing IPVC installation')
mode_exclusive_group.add_argument('-c', '--clean', action='store_const', dest='mode', const='clean', help='Remove IPVC from your system')
if not is_windows():
   parser.add_argument('-k','--kbdpdk',action='store_true', help='Install KBDPDK')

args = parser.parse_args()

#DPDK is always installed on Windows as it only implies some file copy
if not is_windows():
   kbdpdk = args.kbdpdk
else:
   kbdpdk = True

if is_windows():
   configure_script_name += ".exe"
else:
   configure_script_name += ".py"
   
check_access_rights()
check_package()
stop_services()
uninstall_libs()
uninstall_services()
uninstall_configure_script()
uninstall_kbdpdk()
remove_firewall_rule()
uninstall_tools()
deactivate_vmip_logs()
check_common_tools(kbdpdk)
if args.mode == 'install':
   install_tools()
   if is_protected_package:
      licensing(licensing_operation.install)
if args.mode == 'update':
   if is_protected_package:
      licensing(licensing_operation.update)
if args.mode == 'install' or args.mode == 'update':
   if kbdpdk:
      install_kbdpdk()
   install_libs()
   install_vcs()
   install_configure_script()
   add_firewall_rule()
   install_services()
   display_vcs_version()
   activate_vmip_logs()

if args.mode == 'clean':
   uninstall_vcs()
   if is_protected_package:
      licensing(licensing_operation.uninstall)
   clean_deltacast_folder()