#!/usr/bin/env python3

import sys
import os
import signal
import subprocess
import platform
import re
import argparse
import tempfile
import inspect
from collections import OrderedDict

import kbtools
from kbutils import *

__script_desc__='This script configures the system for usage of the KBDPDK library.'
__version_info__ = (1, 0, 7)
__version__ = '.'.join(map(str,__version_info__))
script_dir = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))

## Following lines are for compatibility between python 2.x and 3.x
try:
   input = raw_input
except NameError:
   pass

# Compute the next power of 2 of a 32-bit number 
# https://graphics.stanford.edu/~seander/bithacks.html#RoundUpPowerOf2
def next_pow_of_2(v):
   v -= 1
   for i in (1, 2, 4, 8, 16):
      v |= v >> i
   return v + 1

def human_readable_size(size, decimals=2):
   out_fmt = '{:.'+str(decimals)+'f}{}'
   for unit in ['B','KB','MB','GB','TB']:
      if abs(size) < 1024.0:
         return out_fmt.format(size, unit)
      size /= 1024.0
   return 'infinite'

def get_default_filename():
   i = 0
   while os.path.exists('kbconfig{}.cfg'.format(i)):
      i += 1
   return 'kbconfig{}.cfg'.format(i)

def save_config(config, fname):
   with open(fname, 'w') as f:
      f.write('# ===================================\n')
      f.write('# KBDPDK resources configuration file\n')
      f.write('# ===================================\n')
      f.write('\n')
      f.write('# PLEASE DO NOT EDIT THIS FILE DIRECTLY.\n')
      f.write('# This file was automatically generated by {} v{}.\n'.format(os.path.basename(__file__), __version__))
      f.write('\n')
      nic_cfg = config[0] # (nic1, nic2,...)
      nic_mlx_cfg = config[1] # (nicmlx1, nicmlx2,...)
      mem_cfg = config[2] # (hugepagesz, {'1G_pages':1G_hugepage_count, '2M_pages':2M_hugepage_count),{...})
      cpu_cfg = config[3] # (core1, core2, core3,...)
      # NIC section
      f.write('# Reserved NICS\n')
      f.write('reserved_nics='+','.join(str(nic) for nic in nic_cfg)+'\n')
      f.write('reserved_mlx_nics='+','.join(str(nic) for nic in nic_mlx_cfg)+'\n')
      f.write('\n')
      # Huge pages
      f.write('# Huge pages\n')
      f.write('hugepagesz={}\n'.format(mem_cfg[0]))
      for i in range(len(mem_cfg[1])):
         node_id = i
         nb_1G_hp = mem_cfg[1][i]['1G_pages']
         nb_2M_hp = mem_cfg[1][i]['2M_pages']
         f.write('huge_page_1G_numa{}={}\n'.format(node_id, nb_1G_hp))
         f.write('huge_page_2M_numa{}={}\n'.format(node_id, nb_2M_hp))
      f.write('\n')
      # CPU cores
      f.write('# CPU isolation\n')
      f.write('isol_cpu_list='+','.join('{}'.format(cpu) for cpu in cpu_cfg))
      f.write('\n')

def get_numa_entries():
   sysnode_dir='/sys/devices/system/node/'
   sysnode_entries = os.listdir(sysnode_dir)
   # entries shall match node* pattern
   numa_entries = [i for i in sysnode_entries if re.match(r'node\d*', i)]
   # entries shall be a directory
   numa_entries = [i for i in numa_entries if os.path.isdir(os.path.join(sysnode_dir, i))]
   return numa_entries

def get_free_mem(numa):
   fname='/sys/devices/system/node/node'+str(numa)+'/meminfo'
   free_mem = 0
   with open(fname) as file:
      for line in file:
         matchGroup = re.match(r'^(Node\ \d\ MemFree:)(\s*)(\d*)(\s*)(\w*)', line)
         if matchGroup and matchGroup.group(5) == "kB":
            free_mem = float(matchGroup.group(3)) * 1000
   return free_mem

def reserve_nic():
   print('Network interface reservation')
   print('=============================')
   nic_list = kbtools.get_nic_list()

   print('Detected network devices:')
   print('')
   kbtools.disp_nic_list(nic_list)
   print('')

   # Ask user to select
   selected = []
   while True:
      resp = input('Please enter NIC ID(s) (from {} to {}) of device you want to use with DPDK\n'
                   'You can select multiple device in a comma-separated list: '.format(0, len(nic_list)-1))
      selected = [int(i) for i in resp.split(',') if i.isdigit()]
      if (len(selected) == 0):
         print(red_str('Invalid output "{}": you must at least choose one valid NIC Id\n'.format(resp)))
         continue
      if not all(i>=0 and i<len(nic_list) for i in selected):
         print(red_str('Invalid output "{}": select NIC Id from {} to {}\n'.format(selected, 0, len(nic_list)-1)))
         continue

      # Remove duplicates
      selected = list(OrderedDict.fromkeys(selected))
      print('You have selected {} NIC{}: {}'.format(len(selected), 's' if len(selected)>1 else '', selected))
      
      if query_yes_no('Do you confirm?'):
         break
      print('')

   print('')
   return tuple(nic_list[i] for i in selected)

def reserve_mem(nic_selection):
   """! @todo Add room for objects NUMA allocation (rte_malloc_socket allocates in the huge pages)
   """
   mem_cfg = []
   hugepage_type = 0
   print('Huge pages configuration')
   print('========================')

   # Ask users information about queues
   numa_chan_cfg = {}
   default_queue_count = 15
   default_buffer_size = 8191
   for nic in nic_selection:
      numa = int(nic['NUMANode'])
      print(" * Collect information for NIC {} ('{}')".format(blue_str(nic['Slot']), nic['Device_str']))
      question = '   - How many transmission (TX) channels would you like to use at most with this interface?'
      tx_count = query_number(question, default_queue_count)
      print('     Max TX queues: {}'.format(tx_count))
      question = '   - How many reception (RX) channels would you like to use at most with this interface?'
      rx_count = query_number(question, default_queue_count)
      print('     Max RX queues: {}'.format(rx_count))
      question = '   - How many network packets would you like the channels buffer be able to hold?'
      buffer_size = query_number(question, default_buffer_size)
      print('     Channel buffer size : {} '.format(buffer_size))
      if numa not in numa_chan_cfg:
         numa_chan_cfg[numa] = []
      numa_chan_cfg[numa].append((tx_count, rx_count, buffer_size))

   # Compute request memory per NUMA
   numa_mem = []
   print('')
   title   = '     MEMORY INFORMATION     '
   print('')
   title   = '        | Requested | Free  '
   sep =     '----------------------------'
   row_fmt = ' NUMA {} | {:^9} | {}'
   print(title)
   print(sep)
   numa_count = len(get_numa_entries())
   for numa in range(numa_count):
      chan_cfg = numa_chan_cfg[numa] if numa in numa_chan_cfg.keys() else []
      needed_mem = 0
      for cfg in chan_cfg:
         tx_count = cfg[0] + 1 # add KNI channel
         rx_count = cfg[1] + 1 # add KNI channel
         buf_size = next_pow_of_2(cfg[2]) # mempool size must be 2^n-1 for performance (round it to 2^n)
         needed_mem += (tx_count+rx_count) * (buf_size * 4096) # 4KB per packet

      free_mem = get_free_mem(numa)
      numa_mem.append({'needed':needed_mem,'free':free_mem})
      if free_mem <= needed_mem:
         free_mem_str = red_str(human_readable_size(free_mem))
      elif free_mem <= 2*needed_mem:
         free_mem_str = yellow_str(human_readable_size(free_mem))
      else:
         free_mem_str = green_str(human_readable_size(free_mem))
      print(row_fmt.format(numa, human_readable_size(needed_mem), free_mem_str))

   print('')
   
   # Query users hugepages configuration
   while True:
      mem_cfg = []
      total_pages = 0
      page_size = (1024*1024*1024, 2*1024*1024)
      hugepage_type = 0
      while True:
         question =  'What huge page size would you like to use?\n'
         question += '\t[0] 1GB huge pages (default)\n'
         question += '\t[1] 2MB huge pages\n'
         hugepage_type = query_number(question, 0)
         if hugepage_type not in [0,1]:
            print(red_str('Invalid input {}, please select 0 or 1\n'.format(hugepage_type)))
            continue
         if hugepage_type == 0:
            print('You have selected huge pages of 1GB\n')
         else:
            print('You have selected huge pages of 2MB\n')
         break

      while True:
         mem_cfg = []
         total_pages = 0
         # TODO: add a default value based on needed_mem and free_mem
         hp_1G = 0
         hp_2M = 0
         for numa in range(numa_count):
            needed_mem = numa_mem[numa]['needed']
            needed_pages = (needed_mem//page_size[hugepage_type]) + (1 if needed_mem%page_size[hugepage_type]>0 else 0)
            if hugepage_type == 0:
               question = 'Please enter the amount of 1GB huge pages for NUMA {}'.format(numa)
               hp_1G = query_number(question, needed_pages)
            else:
               question = 'Please enter the amount of 2MB huge pages for NUMA {}'.format(numa)
               hp_2M = query_number(question, needed_pages)
            mem_cfg.append({'1G_pages':hp_1G, '2M_pages':hp_2M})
            total_pages += hp_1G + hp_2M
         print('')

         if total_pages == 0:
            print(red_str('ERROR: you must select at least one huge page'))
            continue
         break

      print('You have selected:')
      for i in range(len(mem_cfg)):
         cfg = mem_cfg[i]
         if hugepage_type == 0:
            print(' * {} huge pages of 1GB on NUMA {}'.format(cfg['1G_pages'], i))
         else:
            print(' * {} huge pages of 2MB on NUMA {}'.format(cfg['2M_pages'], i))

      # Basic checks
      looks_good=True
      for numa in range(numa_count):
         reserved_mem = mem_cfg[numa]['1G_pages']*1024*1024*1024 + mem_cfg[numa]['2M_pages']*2*1024*1024
         free_mem = numa_mem[numa]['free']
         needed_mem = numa_mem[numa]['needed']
         if reserved_mem >= free_mem:
            print(red_str('WARNING: You reserved more memory ({}) than available ({}) on NUMA {}'.format(
                           human_readable_size(reserved_mem), human_readable_size(free_mem), numa)))
            looks_good=False
         if reserved_mem < needed_mem:
            print(yellow_str('WARNING: You reserved less memory ({}) than needed ({}) on NUMA {}'.format(
                           human_readable_size(reserved_mem), human_readable_size(needed_mem), numa)))
            looks_good=False

      if query_yes_no('Do you confirm?', 'yes' if looks_good else 'no'):
         break
      print('')
    
   hugepagesz=''
   if hugepage_type == 0:
     hugepagesz = '1G'
   else:
     hugepagesz = '2M'
   return (hugepagesz, mem_cfg)

def reserve_cpu(nic_selection):
   '''! @todo Checks: add a warning if no core selected on NUMA of the selected NIC '''
   cpu_cfg = ()
   print('CPU cores reservation')
   print('=====================')
   sockets, cores, core_map = kbtools.get_cpu_topology()
   kbtools.disp_cpu_topology(sockets, cores, core_map)
   print('')

   # Ask user to select
   selected = []
   thread_per_core = len(list(core_map.values())[0])
   lcores_count = len(core_map)*thread_per_core
   while True:
      resp = input('Please enter logical cores ID(s) (from {} to {}) of cores you want to isolate ' \
                   'from kernel scheduling\n'
                   'You can select multiple logical cores in a comma-separated list of ids or ranges: '.format(
                   0, lcores_count-1))
      #selected = [int(i) for i in resp.split(',') if i.isdigit()]
      selected, selected_str = parse_int_range(resp)
      if (len(selected) == 0 and len(selected < 3)):
         print(red_str('Invalid output "{}": you must at least choose 3 core Id for isolation\n'.format(resp)))
         continue
      if not all(i>=0 and i<lcores_count for i in selected):
         print(red_str('Invalid output "{}": select logical core IDs from {} to {}\n'.format(
                       selected, 0, lcores_count-1)))
         continue

      # Remove duplicates
      selected = list(OrderedDict.fromkeys(selected))

      # Basic checks
      selected.sort()
      print('You have selected {} logical core{}: {}'.format(len(selected), 's' if len(selected)>1 else '', selected))
      if query_yes_no('Do you confirm?'):
         break
      print('')

   print('')
   return tuple([i for i in selected_str.split(',')])

def sig_handler(sig, frame):
   print('\nSIGINT')
   sys.exit(0)

def main(args=None):
   ret = 0
   parser = argparse.ArgumentParser(description=__script_desc__)
   parser.add_argument('-v', '--version', help='show program version', 
                       action='version',
                       version='{} version {}\n'.format(os.path.basename(__file__), __version__))
   args = parser.parse_args(args)

   # Regiter signal handler to properly exit on ctrl-c
   signal.signal(signal.SIGINT, sig_handler)

   print('=================================')
   print('KBDPDK system configuration setup')
   print('=================================')
   print('')

   # check if lspci is installed
   with open(os.devnull, 'w') as devnull:
      if subprocess.call(['which', 'lspci'], stdout=devnull, stderr=devnull) != 0:
         print("'lspci' not found - please install 'pciutils'")
         return(1)

   # NIC selection
   nic_selection = reserve_nic()
   nic_cfg = tuple(nic['Slot'] for nic in nic_selection if "mellanox" not in nic['Vendor_str'].lower())
   nic_mlx_cfg = tuple(nic['Slot'] for nic in nic_selection if "mellanox" in nic['Vendor_str'].lower())

   # Huge pages configuration
   mem_cfg = reserve_mem(nic_selection)

   # CPU Cores reservation
   cpu_cfg = reserve_cpu(nic_selection)

   # Propose to save the configuration file
   config_saved=False
   saved_fname=''
   if query_yes_no('Do you want to save the configuration in a file?'):
      saved_fname=get_default_filename()
      saved_fname = input('Enter filename to save configuration [{}]: '.format(saved_fname)) or saved_fname
      save_config((nic_cfg,nic_mlx_cfg, mem_cfg, cpu_cfg), saved_fname)
      print('Configuration saved in {} '.format(saved_fname))
      config_saved=True

   # Copy the config file into /etc/kbdpdk/kbconfig.conf
   while True:
      if query_yes_no('Do you want to apply the new configuration (reboot needed)?'):
         tmp_file = tempfile.mktemp()
         save_config((nic_cfg,nic_mlx_cfg, mem_cfg, cpu_cfg), tmp_file)
         if subprocess.call(['sudo', '{}/kbconfig_apply.py'.format(script_dir), '-s', tmp_file]):
            print(red_str('Failed to apply the configuration'))
            return 1
         else:
            print('Done. You need to reboot the system to apply the new configuration')
            break
      else:
         if config_saved:
            print('To apply the configuration later, please execute "kbconfig_apply.py {}"'.format(saved_fname))
            break
         else:
            if query_yes_no(yellow_str('The new configuration will be lost, are you sure to continue?'), 'no'):
               break

   return 0

if __name__ == '__main__':
   sys.exit(main())

