#!/usr/bin/env python3

import sys
import os
import subprocess
import platform
import re
import argparse

__script_desc__='This script is used to check the compatibility of the system with the KBDPDK library.'
__version_info__ = (1, 0, 3)
__version__ = '.'.join(map(str,__version_info__))


# Display results in color
class bcolors:
    RED   = '\033[31m'
    GREEN = '\033[32m'
    BLUE  = '\033[34m'
    ENDC  = '\033[0m'

def print_ok(str1, str2):
    print('\t* {:.<50} {}'.format(str1+' ', bcolors.GREEN+str2+bcolors.ENDC))

def print_nok(str1, str2):
    print('\t* {:.<50} {}'.format(str1+' ', bcolors.RED+str2+bcolors.ENDC))

def check_output(args, stderr=None):
    '''Run a command and capture its output'''
    return subprocess.Popen(args, stdout=subprocess.PIPE, stderr=stderr).communicate()[0]

def is_package_installed(package_name):
   with open(os.devnull, 'w') as devnull:
      if subprocess.call(['dpkg', '-s', package_name], stdout=devnull, stderr=devnull):
         return False
      else:
         return True

def check_package_status(package_name):
   if is_package_installed(package_name):
      print_ok(package_name, 'installed')
      return 0
   else:
      print_nok(package_name, 'not installed')
      return 1

def is_module_installed(module_name):
   with open(os.devnull, 'w') as devnull:
      if subprocess.call(['modinfo', module_name], stdout=devnull, stderr=devnull):
         return False
      else:
         return True

def check_module_status(module_name):
   if is_module_installed(module_name):
      print_ok(module_name, 'installed')
      return 0
   else:
      print_nok(module_name, 'not installed')
      return 1

def check_kernel_version(minimum_version):
   kernel_version = platform.release()
   major = (int)(kernel_version.split('.')[0])
   minor = (int)(kernel_version.split('.')[1])
   minimum_version_str=(str)(minimum_version[0])+'.'+(str)(minimum_version[1])
   if (major,minor) >= minimum_version:
      print_ok('Kernel version: {}'.format(kernel_version), 'OK (>= {})'.format(minimum_version_str))
      return 0
   else:
      print_nok('Kernel version: {}'.format(kernel_version), 'NOT OK (< {})'.format(minimum_version_str))
      return 1

def check_kernel_option(option):
   with open(os.devnull, 'w') as devnull:
      if subprocess.call(['grep', option+'=', '/boot/config-'+platform.release()], stdout=devnull, stderr=devnull):
         return False
      else:
         return True
            
def check_kernel_opt_status(opt):
   if check_kernel_option(opt):
      print_ok(opt, 'OK')
      return 0
   else:
      print_nok(opt, 'NOK')
      return 1

def check_glibc_version(minimum_version):
   ldd_out = check_output(['ldd', '--version']).decode()
   glibc_ver = (ldd_out.splitlines()[0]).split(' ')[-1]
   major = (int)(glibc_ver.split('.')[0])
   minor = (int)(glibc_ver.split('.')[1])
   minimum_version_str=(str)(minimum_version[0])+'.'+(str)(minimum_version[1])
   if (major,minor) >= minimum_version:
      print_ok('glibc version: {}'.format(glibc_ver), 'OK (>= {})'.format(minimum_version_str))
      return 0
   else:
      print_nok('glibc version: {}'.format(glibc_ver), 'NOT OK (< {})'.format(minimum_version_str))
      return 1

def check_cpu_cores(minimum_core_count):
   ''' @todo Check support of hugepages ("pse") and 1G hugepages ("pdpe1gb") '''
   core_count = open('/proc/cpuinfo').read().count('processor\t:')
   if core_count >= minimum_core_count:
      print_ok('Number of cores: {}'.format(core_count), 'OK (>= {})'.format(minimum_core_count))
      return 0
   else:
      print_nok('Number of cores: {}'.format(core_count), 'NOT OK (< {})'.format(minimum_core_count))
      return 1

def get_numa_entries():
   sysnode_dir='/sys/devices/system/node/'
   sysnode_entries = os.listdir(sysnode_dir)
   # entries shall match node* pattern
   numa_entries = [i for i in sysnode_entries if re.match(r'node\d*', i)]
   # entries shall be a directory
   numa_entries = [i for i in numa_entries if os.path.isdir(os.path.join(sysnode_dir, i))]
   return numa_entries

def check_free_memory(minimum_free_mem):
   sysnode_dir='/sys/devices/system/node/'
   numa_entries = get_numa_entries()
   print('Number of NUMA nodes: {}'.format(len(numa_entries)))
   for node in numa_entries:
      fname = sysnode_dir+node+'/meminfo'
      with open(fname) as file:
         for line in file:
            matchGroup = re.match(r'^(Node\ \d\ MemFree:)(\s*)(\d*)(\s*)(\w*)', line)
            if matchGroup and matchGroup.group(5) == "kB":
               free_mem_gb = ((float(matchGroup.group(3)) * 1000) / (1024* 1024*1024))
               minimum_free_mem_gb = float(minimum_free_mem)/(1024*1024*1024)
               if free_mem_gb >= minimum_free_mem_gb:
                  print_ok('Available memory for {}: {:.2f}GB'.format(node,free_mem_gb),
                           'OK (>= {:.2f}GB)'.format(minimum_free_mem_gb))
               else:
                  print_nok('Available memory for {}: {:.2f}GB'.format(node,free_mem_gb),
                            'NOT OK (>= {:.2f}GB)'.format(minimum_free_mem_gb))


def main(args=None):
   err_cnt = 0
   minimum_kernel_version=(4, 4)
   minimum_glibc_version=(2, 7)
   minimum_core_count=4
   minimum_free_memory=(1*1024*1024*1024) # 1GB

   print('=================================')
   print('KBDPDK system compatibility check')
   print('=================================')
   print('')

   parser = argparse.ArgumentParser(description=__script_desc__)
   parser.add_argument('-v', '--version', help='show program version', action='store_true')
   args = parser.parse_args(args)

   if args.version:
      print('{} version {}\n'.format(os.path.basename(__file__), __version__))
      return 0

   print('Checking required tools and packages')

   # Check presence of build-essential
   err_cnt += check_package_status('build-essential')
   err_cnt += check_package_status('libnuma-dev')
   err_cnt += check_package_status('linux-headers-'+platform.release())
   print('');

   # Check presence of UIO driver
   print('Checking required modules')
   err_cnt += check_module_status('uio')
   print('');

   # Check Kernel version
   print('Checking Linux kernel version')
   err_cnt += check_kernel_version(minimum_kernel_version)
   print('');

   # Check Kernel options
   print ('Checking Kernel options')
   err_cnt += check_kernel_opt_status('HUGETLBFS')
   err_cnt += check_kernel_opt_status('PROC_PAGE_MONITOR')
   print('');

   # Check glibc version
   print('Checking glibc version')
   err_cnt += check_glibc_version(minimum_glibc_version)
   print('');

   # Check CPU cores
   print('Checking CPU cores')
   err_cnt += check_cpu_cores(minimum_core_count)
   print('');
  
   # Check memory available
   check_free_memory(minimum_free_memory)

   return 1 if err_cnt > 0 else 0

if __name__ == "__main__":
   sys.exit(main())


