#!/bin/sh
###############################################################################
#             IP Virtual Card Linux installation shell script                 #
###############################################################################

check_package_integrity()
{
	VAR_INTEGRITY=1

	if [ ! -d ./kbdpdk ]; then VAR_INTEGRITY=0; fi
	if [ ! -d ./linuxptp ]; then VAR_INTEGRITY=0; fi
	if [ ! -d ./vcs/ ]; then VAR_INTEGRITY=0; fi
	if [ ! -d ./videomasterip/ ]; then VAR_INTEGRITY=0; fi

	if [ $VAR_INTEGRITY = "0" ]; then
		echo "Package integrity KO"
		echo "Error : the shell script should be executed from VideoMasterIP Linux redist folder or the package is corrupted"
		exit 1
	fi
}

set_member_variables()
{
	VAR_SDKDIR=`pwd` 

	if [ -d /usr/local/lib64 ]; then
		VAR_LIBDIR=/usr/local/lib64
	else
		VAR_LIBDIR=/usr/local/lib
	fi


	if [ -f /etc/os-release ]; then
		. /etc/os-release
		OS=$ID
	fi


	if [ $OS = "redhat"  ]; then
		PACKAGE_MANAGER=yum
	elif [ $OS = "centos"  ]; then
		PACKAGE_MANAGER=yum
	else
		PACKAGE_MANAGER=apt-get
	fi
}

install_common_tools()
{
	# Common tools installation
	$PACKAGE_MANAGER -y update
	$PACKAGE_MANAGER -y install g++ make libcpprest git

	if [ ! $? -eq 0 ]; then
		echo "ERROR : common tools installation failed"
	fi                                     
}

install_ptp()
{
	# Disabling NTP
	timedatectl set-ntp 0

	if [ ! $? -eq 0 ]; then
		echo "ERROR: NTP disabling failed"
	fi


	# Build+patch linuxptp instead of installing pre-built package
	PTP_VERSION=2.0
	DIR=$(mktemp -d)
	cd $DIR/
	git clone http://git.code.sf.net/p/linuxptp/code linuxptp

	if [ ! $? -eq 0 ]; then
		echo "ERROR: git linuxptp clone failed"
	fi

	cd linuxptp
	git checkout -b $PTP_VERSION v$PTP_VERSION

	if [ ! $? -eq 0 ]; then
		echo "ERROR: git linuxptp checkout failed"
	fi

	patch -p1 < $VAR_SDKDIR/linuxptp/donotrespondst2059_2tlv.patch
	patch -p1 < $VAR_SDKDIR/linuxptp/kni.patch
	make
	make install

	if [ ! $? -eq 0 ]; then
		echo "ERROR: PTP installation failed"
	fi

	make distclean
	rm -rf $DIR

	install -d /etc/linuxptp/ 
	install -m 644 $VAR_SDKDIR/linuxptp/ptp4l.conf     /etc/linuxptp/ptp4l.conf

	if [ ! $? -eq 0 ]; then
		echo "ERROR: PTP .conf file installation failed"
	fi
 
	cd $VAR_SDKDIR                                     
 }   

install_libs()
{  
	for VAR_LIBPATH in "$VAR_SDKDIR/videomasterip"/*
	do
	    [ -f "$VAR_LIBPATH" ] || continue

		chmod 644 $VAR_LIBPATH;
		cp -P $VAR_LIBPATH $VAR_LIBDIR;
   
		if [ ! $? -eq 0 ]; then 
			echo "Error : $VAR_LIBPATH library installation error"
			exit 1
		fi
	done
	
	if ! grep -qx "$VAR_LIBDIR" /etc/ld.so.conf.d/libc.conf; then
		echo "$VAR_LIBDIR" >>/etc/ld.so.conf.d/libc.conf
	fi
 
	/sbin/ldconfig -n $VAR_LIBDIR
	/sbin/ldconfig -X
}

install_vcs()
{ 
	install -d /opt/deltacast/videomasterip/vcs/
    #back-up the configuration files
	install -b -m 755 ./vcs/vcsconfig*  /opt/deltacast/videomasterip/vcs/
	install -m 755 ./vcs/*  /opt/deltacast/videomasterip/vcs/
  
	if [ ! $? -eq 0 ]; then 
		echo "Error : VCS installation error"
	fi
} 

stop_services()
{ 
	# Stop services before install               
	systemctl disable vcs.service
	systemctl stop vcs.service
	
	systemctl disable ptp4l.service
	systemctl stop ptp4l.service

	systemctl disable phc2sys.service
	systemctl stop phc2sys.service
	
	systemctl disable network_sysctl.service
	systemctl stop network_sysctl.service
}     

install_licensing()
{
	if [ -d ./licensing ]; then

		VAR_LICENSINGDIR=$VAR_SDKDIR/licensing
		
		DIR=$(mktemp -d)
		cd $DIR

		tar -xvf $VAR_LICENSINGDIR/dlmcli.tar.gz 

		cd $DIR/x64

		sh install_dlm.sh update

		cd $VAR_SDKDIR

		rm -rf $DIR

	fi
}

install_services()
{ 
	# VCS service installation               
	install -m 644 ./vcs/vcs.service  /lib/systemd/system/
	cd /lib/systemd/system/   
	
	systemctl enable vcs.service

 
	# PTP service installation				 
	install -m 644 $VAR_SDKDIR/linuxptp/ptp4l.service  /lib/systemd/system/
 	install -m 644 $VAR_SDKDIR/linuxptp/phc2sys.service  /lib/systemd/system/

	systemctl enable ptp4l.service
    systemctl enable phc2sys.service
 
	# Network sysctl service installation				 
	install -m 644 $VAR_SDKDIR/vcs/network_sysctl.service  /lib/systemd/system/
	
	systemctl enable network_sysctl.service
 		
		
	systemctl daemon-reload
	
 
	# VCS service start 	
	systemctl start vcs.service
 
	if [ ! $? -eq 0 ]; then 
		echo "Error : VCS service installation error"
	fi
  	
   
	# PTP service start
	systemctl start ptp4l.service
 
	if [ ! $? -eq 0 ]; then 
		echo "Error : ptp4l service installation error"
	fi


	# Network sysctl service start	
	systemctl start network_sysctl.service
 
	if [ ! $? -eq 0 ]; then 
		echo "Error : Network sysctl service installation error"
	fi
 
	 
 	cd $VAR_SDKDIR  
}

uninstall_libs()
{
	for VAR_LIBPATH in "$VAR_SDKDIR/videomasterip"/*
	do
	    [ -s "$VAR_LIBPATH" ] || continue
		VAR_LIBNAME="${VAR_LIBPATH##*/}"
		 
		rm -rf $VAR_LIBDIR/"${VAR_LIBNAME%.*.*.*.*}"*;         
	done
}

uninstall_licensing()
{
	if [ -d ./licensing ]; then
		VAR_LICENSINGDIR=$VAR_SDKDIR/licensing
		
		DIR=$(mktemp -d)
		cd $DIR

		tar -xvf $VAR_LICENSINGDIR/dlmcli.tar.gz 

		cd $DIR/x64

		sh install_dlm.sh clean

		cd $VAR_SDKDIR

		rm -rf $DIR
	fi
}

update_licensing()
{
	if [ -d ./licensing ]; then
		VAR_LICENSINGDIR=$VAR_SDKDIR/licensing
		
		DIR=$(mktemp -d)
		cd $DIR

		tar -xvf $VAR_LICENSINGDIR/dlmcli.tar.gz 

		cd $DIR/x64

		sh install_dlm.sh update

		cd $VAR_SDKDIR

		rm -rf $DIR
	fi
}

uninstall_vcs()
{
	rm -rf /opt/deltacast
}

uninstall_services()
{
	systemctl disable vcs.service
	systemctl stop vcs.service	
	rm -rf /lib/systemd/system/vcs.service
 
	systemctl disable network_sysctl.service
	systemctl stop network_sysctl.service	
	rm -rf /lib/systemd/system/network_sysctl.service
  
	systemctl daemon-reload	
	systemctl reset-failed
} 

check_access_right()
{
	if [ $(id -u) -ne 0 ]; then 
		echo "Please run as root"
		exit 1 
	fi
}

case "$1" in 
    clean) 
      check_access_right
		check_package_integrity
		set_member_variables
		uninstall_services
		uninstall_licensing
		uninstall_libs
		uninstall_vcs
      ;; 
    update) 
		check_access_right
		check_package_integrity
		set_member_variables
		stop_services
		uninstall_libs
      install_libs
		update_licensing
		install_vcs
		install_services
		;; 
    *) 
		check_access_right
		check_package_integrity
		set_member_variables
		install_common_tools
		stop_services
		uninstall_libs
		install_ptp
		install_libs
		install_licensing
		install_vcs
		install_services 
      ;; 
esac    	   
