## @package kbtools
#

import os
import subprocess
from kbutils import *

## Following lines are for compatibility between python 2.x and 3.x
try:
   xrange
except NameError:
   xrange = range

def check_output(args, stderr=None):
    '''Run a command and capture its output'''
    return subprocess.Popen(args, stdout=subprocess.PIPE, stderr=stderr).communicate()[0]

def dev_is_nic(dev):
   ''' Check PCI device class, 02 for Network Controllers '''
   if dev['Class'][0:2] == '02':
      return True
   return False

## Retrieve the list of detected NICs
#  @return list of dict containing NIC info
def get_nic_list():
   nic_list = []
   dev = {}
   lspci_output = check_output(["lspci", "-Dvmmnnk"]).splitlines()
   for dev_line in lspci_output:
      if len(dev_line) == 0: # end of device info -> keep it in nic_list if it is a nic...
         if dev_is_nic(dev):
            dev_syspath='/sys/bus/pci/devices/{}'.format(dev['Slot'])
            net_syspath=os.path.join(dev_syspath, 'net')
            dev['Interface'] = ''
            dev['OperState'] = ''
            if os.path.exists(net_syspath):
               dev['Interface'] = ','.join(os.listdir(net_syspath))
               if_syspath=os.path.join(net_syspath, dev['Interface'])
               dev['OperState'] = open(os.path.join(if_syspath,'operstate')).read().rstrip('\n')
                           
            nic_list.append(dev)
         # Clear previous device's data
         dev = {}
      else:
         name, value = dev_line.decode().split("\t", 1)
         value_list = value.rsplit(' ', 1)
         if len(value_list) > 1:
            # String stored in <name>_str
            dev[name.rstrip(":") + '_str'] = value_list[0]
         # Numeric IDs
         dev[name.rstrip(":")] = value_list[len(value_list) - 1].rstrip("]").lstrip("[")
   return nic_list

## Display NIC list in a formatted table
#
def disp_nic_list(nic_list):
   itf_max_len = len(max([nic['Interface'] for nic in nic_list], key=len))
   slot_max_len = len(max([nic['Slot'] for nic in nic_list], key=len))
   vend_max_len = len(max([nic['Vendor_str'] for nic in nic_list], key=len))
   desc_max_len = len(max([nic['Device_str'] for nic in nic_list], key=len))

   title_fmt   = " {:^6} | {:^"+str(slot_max_len)+"} | {:^4} |" \
                 " {:"+str(itf_max_len)+"} | {:^6} |" \
                 " {:^"+str(vend_max_len)+"} | {:^"+str(desc_max_len)+"}"

   row_fmt_net = " {:^6} | {:^"+str(slot_max_len)+"} | {:^4} |" \
                 " {:"+str(itf_max_len)+"} | {:^15} |" \
                 " {:^"+str(vend_max_len)+"} | '{}'"
   row_fmt_igb = " {:^6} | {:^"+str(slot_max_len)+"} | {:^4} |" \
                 " {:^"+str(itf_max_len+18)+"}{} |" \
                 " {:^"+str(vend_max_len)+"} | '{}'"

   title=title_fmt.format('NIC Id', 'Slot', 'NUMA', 'Itf', 'State', 'Vendor', 'Description')
   separator = '-'*len(title)
   print(title)
   print(separator)
   for i in range(len(nic_list)):
      if 'Driver' in nic_list[i] and nic_list[i]['Driver'] == 'igb_uio':
         row_fmt = row_fmt_igb
         nic_list[i]['Interface'] = blue_str('BOUND to UIO')
      else:
         row_fmt = row_fmt_net

      if 'NUMANode' not in nic_list[i]:
         nic_list[i]['NUMANode'] = 0

      if nic_list[i]['OperState'] == 'up':
         oper_state_str = green_str(nic_list[i]['OperState'].upper())
      else:
         oper_state_str = red_str(nic_list[i]['OperState'].upper())

      print(row_fmt.format('['+str(i)+']', nic_list[i]['Slot'], 
                                                     nic_list[i]['NUMANode'],
                                                     nic_list[i]['Interface'],
                                                     oper_state_str,
                                                     nic_list[i]['Vendor_str'],
                                                     nic_list[i]['Device_str']))


## Get CPU topology
# Returns 
#   - 2 lists of sockets and cores
#   - a dict containing information located in /sys/devices/system/cpu
#
def get_cpu_topology():
   sockets = []
   cores = []
   core_map = {}
   base_path = "/sys/devices/system/cpu"
   # Read maximum CPU index allowed in the kernel
   fd = open("{}/kernel_max".format(base_path)) 
   max_cpus = int(fd.read())
   fd.close()
   # Retrieve sockets, core and logical cores
   for cpu in xrange(max_cpus + 1):
      try:
         fd = open("{}/cpu{}/topology/core_id".format(base_path, cpu))
      except IOError:
         continue
      except:
         break
      core = int(fd.read())
      fd.close()
      fd = open("{}/cpu{}/topology/physical_package_id".format(base_path, cpu))
      socket = int(fd.read())
      fd.close()
      if core not in cores:
         cores.append(core)
      if socket not in sockets:
         sockets.append(socket)
      key = (socket, core)
      if key not in core_map:
         core_map[key] = []
      core_map[key].append(cpu)

   return sockets, cores, core_map
  
## Display the core topology in a formatted table
#
def disp_cpu_topology(sockets, cores, core_map):
   # List of isolated CPUs
   isol_range = subprocess.check_output(('cat', '/sys/devices/system/cpu/isolated')).decode()
   isol_list, _ = parse_int_range(isol_range)
   thread_per_core = len(list(core_map.values())[0]) 
   print('{:.<40} {}'.format('NUMA sockets ', len(sockets)))
   print('{:.<40} {}'.format('Cores per sockets ', len(cores)))
   print('{:.<40} {}'.format('Threads per core ', thread_per_core))
   print('{:.<40} {}'.format('Logical cores ', len(core_map)*thread_per_core))
   print('')

   col1_head = 'Cores'
   col2_head = 'Logical cores'
   tables = [None] * len(sockets)
   for i in range(len(sockets)):
      s = sockets[i]
      tables[i] = {}
      col1_width = max(len(col1_head), len(str(max(cores))))
      col2_width = max(len(col2_head), max([len(str(v)) for v in core_map.values()]))
      table_width = col1_width + col2_width + 7
      
      title_str     = yellow_str('{:^'+str(table_width)+'}')
      top_line_str  = '+'+'-'*(col1_width+col2_width+5)+'+'
      bot_line_str  = top_line_str
      separator_str = '|'+'-'*(col1_width+col2_width+5)+'|'
      row_str       = '| {:^'+str(col1_width)+'} | {:^'+str(col2_width)+'} |'

      tables[i]['width'] = table_width
      tables[i]['lines'] = []
      tables[i]['lines'].append(title_str.format('Socket '+str(s)))
      tables[i]['lines'].append(top_line_str)
      tables[i]['lines'].append(row_str.format(col1_head, col2_head))
      tables[i]['lines'].append(separator_str)
      for c in cores:
         if (s,c) in core_map:
            col2_w = col2_width
            lcore_str = '['
            for lcore in core_map[(s,c)]:
               if lcore in isol_list:
                  lcore_str += '*{}*'.format(lcore)+', '
               else:
                  lcore_str += '{}, '.format(lcore)
            lcore_str = lcore_str[:-2]+']'
            row_str = '| {:^'+str(col1_width)+'} | {:^'+str(col2_w)+'} |'
            tables[i]['lines'].append(row_str.format(c, lcore_str))
      tables[i]['lines'].append(bot_line_str)

   # diplay tables 2 by 2 
   for group in range((len(tables)+1)//2):
      idx = 2*group
      if len(tables) > idx+1:
         line_fmt = '{:<'+str(tables[idx]['width'])+'}    {}'
      else:
         line_fmt = '{}'

      for line in range(len(tables[idx]['lines'])):
         l1 = tables[idx]['lines'][line]
         l2 = tables[idx+1]['lines'][line] if len(tables) > idx+1 else None
         print(line_fmt.format(l1, l2))
   if len(isol_list) > 0:
      print('Logical cores currently isolated are shown between asterix : {}.'.format('*core*'))
