#!/usr/bin/env python

import sys
import os
import subprocess
import platform
import re
import argparse
import tempfile

from kbutils import *

__script_desc__='This script apply a configuration to the system for usage of the KBDPDK library.'
__version_info__ = (1, 0, 4)
__version__ = '.'.join(map(str,__version_info__))

grub_cfg_backup = '/etc/default/.grub_kbdpdk_saved'

## Following lines are for compatibility with python 2.x
try:
   input = raw_input
except NameError:
   pass

def config_file_is_valid(fname):
   """! @todo Add more check (formatting...) """
   with open(fname) as f:
      file_str = f.read()

   # At least this must be present in file:
   mandatory_str=['reserved_nics=', 'huge_page_1G_numa0=', 'huge_page_2M_numa0=', 'isol_cpu_list=']
   for key in mandatory_str:
      if (not key in file_str):
         print(red_str("'{}' key not found in file".format(key)))
         return False
   
   return True

def remove_entry(line, entry):
   param = line.split('"')[0]
   value = line.split('"')[1]
   ending = line.split('"')[2]
   words = value.split()
   result_words = [word for word in words if entry not in word]
   return param+'"'+' '.join(result_words)+'"'+ending

def apply_config(cfg_file):
   # Copy config file to /etc
   config_dir = '/etc/kbdpdk/'
   target_file = config_dir + 'kbconfig.conf'

   if not os.path.exists(config_dir):
      os.mkdir(config_dir)

   if not (subprocess.call(['cp', cfg_file, target_file]) == 0):
      print(red_str('Failed to copy the configuration file'))
      return 1

   ## Grub configuration

   # Extract cpu isolation list and hugepage size
   isol_cpu=''
   hugepagesz=''
   with open(cfg_file) as f:
      for line in f:
         match_isol_cpu = re.match(r'^isol_cpu_list=([\d|,|-]*)(\s*#.*)?$', line)
         match_hugepage_sz = re.match(r'^hugepagesz=([\dmMgG]*)(\s*#.*)?$', line)
         if match_isol_cpu:
            isol_cpu = match_isol_cpu.group(1)
         if match_hugepage_sz:
            hugepagesz = match_hugepage_sz.group(1)

   kbdpdk_cmdline = ''
   if len(hugepagesz):
      kbdpdk_cmdline = 'default_hugepagesz={} hugepagesz={} '.format(hugepagesz, hugepagesz)
   if len(isol_cpu):
      kbdpdk_cmdline += 'isolcpus={} nohz_full={} rcu_nocbs={}'.format(isol_cpu, isol_cpu, isol_cpu)


   # Manipulation of /etc/default/grub
   
   if not os.path.isfile(grub_cfg_backup):
      if subprocess.call(['cp', '/etc/default/grub', grub_cfg_backup]):
         print(red_str('Failed to backup the grub configuration file'))
         return 1

   tmp_file = tempfile.mktemp()
   with open('/etc/default/grub', 'r') as orig, open(tmp_file, 'w') as new:
      for line in orig:
         if not 'KBDPDK' in line:
            if 'GRUB_CMDLINE_LINUX_DEFAULT=' in line:
               # Remove potential conflicting settings
               line = remove_entry(line, 'isolcpus=')
               line = remove_entry(line, 'nohz_full=')
               line = remove_entry(line, 'rcu_nocbs=')
               line = remove_entry(line, 'hugepagesz=')
               line = remove_entry(line, 'iommu=on')
            new.write(line)


      # only write following lines if kbdpdk_cmdline is not empty
      if len(kbdpdk_cmdline):
         if not (line == '\n' or 'KBDPDK' in line):
            new.write('\n')
         new.write('# CPU isolation for KBDPDK usage\n')
         new.write('GRUB_CMDLINE_KBDPDK="{}"\n'.format(kbdpdk_cmdline))
         new.write('GRUB_CMDLINE_LINUX_DEFAULT="$GRUB_CMDLINE_LINUX_DEFAULT $GRUB_CMDLINE_KBDPDK"\n')

   # Overwrite grub config
   if subprocess.call(['cp', tmp_file, '/etc/default/grub']):
      print(red_str('Failed to save the grub configuration file'))
      return 1
   os.remove(tmp_file)

   # Apply grub new config
   with open(os.devnull, 'w') as devnull:
      if subprocess.call(['update-grub'], stdout=devnull, stderr=devnull):
         print(red_str('Failed to apply the grub configuration file'))
         subprocess.call(['mv', '/etc/default/grub', '/etc/default/.grub_kbdpdk_failed'])
         # Switch back to the backuped configuration
         subprocess.call(['cp', grub_cfg_backup, '/etc/default/grub'])
         os.remove(grub_cfg_backup)
         return 1

   return 0

def main(args=None):
   parser = argparse.ArgumentParser(description=__script_desc__)
   parser.add_argument('-v', '--version', help='show program version', 
                       action='version', 
                       version='{} version {}\n'.format(os.path.basename(__file__), __version__))
   parser.add_argument('-s', '--silent', help='run silently', action='store_true')
   parser.add_argument('config_file')
   args = parser.parse_args(args)

   if os.getuid() != 0:
      print('Operation not permitted, please run this script as root user')
      return 1

   fname = args.config_file
   if not os.path.isfile(fname):
      print('Error: file not found {}'.format(fname))
      return 1

   if not args.silent:
      print('Applying configuration from file {}...'.format(fname))

   if not config_file_is_valid(fname):
      print('Error: {} is not a valid config file'.format(fname))
      print('Please select a valid configuration file or use the kbconfig_setup.py to generate a new configuration')
      return 1

   question = 'Do you want to apply the configuration from file {}?'.format(fname)
   if args.silent or query_yes_no(question):
      ret = apply_config(fname)

   if not args.silent:
      if ret == 0:
         print('Done. You need to reboot the system to apply the new configuration.')
      else:
         print(red_str('Error while applying configuration'))

   return ret

if __name__ == '__main__':
   sys.exit(main())

