#!/usr/bin/env python
import sys
import os
import subprocess
import argparse
import glob

sys.path.insert(0, '../KBCONFIG')
import kbtools
from kbutils import *

__script_desc__='This script is used to check that the configuration of the system' \
                ' is valid to use the KBDPDK library.'
__version_info__ = (1, 0, 0)
__version__ = '.'.join(map(str,__version_info__))

gSilentMode = False
CONFIG_DIR = '/etc/kbdpdk'
KBDPDK_PATH = '/usr/local/share/kbdpdk'

def kbprint(string):
   if not gSilentMode:
      print(string)

def print_ok(str1, str2):
   if not gSilentMode:
      print('\t* {:.<50} {}'.format(str1+' ', green_str(str2)))

def print_nok(str1, str2):
   if not gSilentMode:
      print('\t* {:.<50} {}'.format(str1+' ', red_str(str2)))

def file_exists(filename):
   return os.path.isfile(filename)

def dir_exists(dirname):
   return os.path.isdir(dirname)

def lib_found(libname):
   liblist = subprocess.check_output(('ldconfig', '-p')).decode()
   if (liblist.find(libname) != -1):
      return True
   return False

def check_lib_install():
   err = 0
   if lib_found('libkbapi.so'):
      print_ok('libkbapi.so', 'found')
   else:
      print_nok('libkbapi.so', 'not found')
      err = 1

   if dir_exists(KBDPDK_PATH):
      print_ok(KBDPDK_PATH, "found")
   else:
      print_nok(KBDPDK_PATH, "not found")
      err = 1

   return err

def check_config_files():
   err = 0
   if dir_exists(CONFIG_DIR):
      print_ok('Config dir ({})'.format(CONFIG_DIR), 'found')
      if file_exists(CONFIG_DIR+'/kbconfig.conf'):
         print_ok('kbconfig.conf', 'found')
      else:
         print_nok('kbconfig.conf', 'not found')
         err = 1
      
      if file_exists(CONFIG_DIR+'/kbapi.conf'):
         print_ok('kbapi.conf', 'found')
      else:
         print_nok('kbapi.conf', 'not found')
         err = 1
   else:
      print_nok('Config dir', 'not found')
      err = 1
   return err

def get_runlevel():
   runlevel = subprocess.check_output(('runlevel')).decode().split(' ')[1]
   return int(runlevel)

def check_boot_script():
   err = 0
   if file_exists('/etc/init.d/kbdpdk'):
      print_ok('boot script', 'found')
      cmd = 'systemctl list-unit-files | grep enabled | grep kbdpdk'
      sysctlproc = subprocess.Popen(['systemctl', 'list-unit-files'], stdout=subprocess.PIPE)
      grep1proc = subprocess.Popen(['grep', 'enabled'], stdin=sysctlproc.stdout, stdout=subprocess.PIPE)
      with open(os.devnull, 'w') as devnull:
         if subprocess.call(('grep', 'kbdpdk'), stdin=grep1proc.stdout, stdout=devnull, stderr=devnull) == 0:
            print_ok('kbdpdk service', 'enabled')
         else:
            print_nok('kbdpdk service', 'not enabled')
            err = 1
   else:
      print_nok('boot script', 'not found')
      err = 1
   return err

def check_isol_cpu():
   isol_range = subprocess.check_output(('cat', '/sys/devices/system/cpu/isolated')).decode()
   num_isol_cpu = len(parse_int_range(isol_range)[0])
   if num_isol_cpu >= 3:
      print_ok('{} cpu isolated'.format(num_isol_cpu), 'OK')
      return 0
   else:
      print_nok('{} cpu isolated'.format(num_isol_cpu), 'NOK')
      return 1

def check_hugepage_config():
   err = 0
   nr_hp_2M = 0
   nr_hp_1G = 0
   if os.path.isfile('/sys/kernel/mm/hugepages/hugepages-2048kB/nr_hugepages'):
      nr_hp_2M = int(subprocess.check_output(('cat', '/sys/kernel/mm/hugepages/hugepages-2048kB/nr_hugepages')))
   if os.path.isfile('/sys/kernel/mm/hugepages/hugepages-1048576kB/nr_hugepages'):
      nr_hp_1G = int(subprocess.check_output(('cat', '/sys/kernel/mm/hugepages/hugepages-1048576kB/nr_hugepages')))
   if nr_hp_2M + nr_hp_1G > 0:
      print_ok('{} pages of 2M / {} pages of 1G'.format(nr_hp_2M, nr_hp_1G), 'OK')
      hp_2M_mounted = False 
      hp_1G_mounted = False 
      with open(os.devnull, 'w') as devnull:
         if subprocess.call(('grep', '/dev/hugepages/huge_2MB', '/proc/mounts'), stdout=devnull, stderr=devnull) == 0:
            hp_2M_mounted = True
         if subprocess.call(('grep', '/dev/hugepages/huge_1GB', '/proc/mounts'), stdout=devnull, stderr=devnull) == 0:
            hp_1G_mounted = True
      if hp_2M_mounted and hp_1G_mounted:
         print_ok('Huge pages are mounted', 'OK')
      elif hp_2M_mounted: # only 2M are mounted
         if nr_hp_2M > 0:
            print_ok('Huge pages of 2MB are mounted', 'OK')
         else:
            print_nok('Huge pages are not mounted', 'NOK')
            err = 1
      elif hp_1G_mounted: # only 1G are mounted
         if nr_hp_1G > 0:
            print_ok('Huge pages of 1GB are mounted', 'OK')
         else:
            print_nok('Huge pages are not mounted', 'NOK')
            err = 1
      else:
         print_nok('Huge pages are not mounted', 'NOK')
         err = 1

   else:
      print_nok('No huge pages configured', 'NOK')
      err = 1

   return err 

def check_kmdrivers():
   err = 0
   lsmod = subprocess.check_output(('cat', '/proc/modules')).decode()
   if (lsmod.find('rte_kni') != -1):
      print_ok('rte_kni', 'loaded')
   else:
      print_nok('rte_kni', 'not loaded')
      err = 1

   if (lsmod.find('igb_uio') != -1):
      print_ok('igb_uio', 'loaded')
   else:
      print_nok('igb_uio', 'not loaded')
      err = 1

   return err

def check_nic_config():
   uio_bound_count = 0
   nic_list = kbtools.get_nic_list()
   for nic in nic_list:
      if 'Driver' in nic and nic['Driver'] == 'igb_uio':
         print_ok('NIC {} bound to {}'.format(nic['Slot'],nic['Driver']), 'OK')
         uio_bound_count += 1
   if uio_bound_count == 0:
      print_nok('No NIC bound to UIO driver', 'NOK')
      return 1
   return 0

def check_iommu_status():
   bootargs = subprocess.check_output(('cat', '/proc/cmdline')).decode()
   if ((bootargs.find('intel_iommu=on') != -1) or 
       (bootargs.find('amd_iommu=on') != -1) or
       (bootargs.find('iommu=pt') != -1) or
       (bootargs.find('amd_iommu=pt') != -1)):
      print_nok('IOMMU is enabled', 'NOK')
      return 1
   else:
      print_ok('IOMMU is not enabled', 'OK')
      return 0


def main(args=None):
   global gSilentMode
   err_cnt = 0

   parser = argparse.ArgumentParser(description=__script_desc__)
   parser.add_argument('-v', '--version', help='show program version', action='store_true')
   parser.add_argument('-s', '--silent', help='Run silently', action='store_true')
   args = parser.parse_args(args)

   if args.version:
      print('{} version {}\n'.format(os.path.basename(__file__), __version__))
      return 0

   if args.silent:
      gSilentMode = True

   kbprint('=================================')
   kbprint('KBDPDK system configuration check')
   kbprint('=================================')
   kbprint('')

   kbprint('Checking presence of mandatory folders and files')
   err_cnt += check_lib_install()
   err_cnt += check_config_files()
   err_cnt += check_boot_script()

   kbprint('')
   kbprint('Checking CPU isolation')
   err_cnt += check_isol_cpu()

   kbprint('')
   kbprint('Checking huge pages configuration')
   err_cnt += check_hugepage_config()

   kbprint('')
   kbprint('Checking km drivers')
   err_cnt += check_kmdrivers()

   kbprint('')
   kbprint('Checking NICs configuration')
   err_cnt += check_nic_config()

   kbprint('')
   kbprint('Checking iommu status')
   err_cnt += check_iommu_status()

   return 1 if err_cnt > 0 else 0


if __name__ == "__main__":
   sys.exit(main())

