#!/bin/sh
###############################################################################
#             IP Virtual Card Linux installation shell script                 #
###############################################################################
check_access_right()
{
	echo "Checking access right..."
	if [ $(id -u) -ne 0 ]; then 
		echo "Please run as root"
		exit 1 
	fi
}

check_package_integrity()
{
	echo "Checking package integrity..."	
	VAR_INTEGRITY=1

	if [ ! -f ./dlmcli.out ]; then VAR_INTEGRITY=0; fi
	if [ ! -f ./dlm.sh ]; then VAR_INTEGRITY=0; fi
	if [ ! -f ./dlm.service ]; then VAR_INTEGRITY=0; fi
	if [ ! -f ./lib/libFlxComm64.so.2020.07 ]; then VAR_INTEGRITY=0; fi
	if [ ! -f ./lib/libFlxCore64.so.2020.07 ]; then VAR_INTEGRITY=0; fi

	if [ $VAR_INTEGRITY = "0" ]; then
		echo "Package integrity KO"
		echo "Error : the shell script should be executed from Deltacast license manager redist folder or the package is corrupted"
		exit 1
	fi
}

set_member_variables()
{
	echo "Setting member variables..."
	VAR_INSTALLDIR=`pwd`

	if [ -d /usr/local/lib64 ]; then
		VAR_LIBDIR=/usr/local/lib64
	else
		VAR_LIBDIR=/usr/local/lib
	fi


	if [ -f /etc/os-release ]; then
		. /etc/os-release
		OS=$ID
	fi


	if [ $OS = "redhat"  ]; then
		PACKAGE_MANAGER=yum
	elif [ $OS = "centos"  ]; then
		PACKAGE_MANAGER=yum
	else
		PACKAGE_MANAGER=apt-get
	fi
}

install_common_tools()
{
	echo "Installing common tools..."
	# Common tools installation
	$PACKAGE_MANAGER -y update
	$PACKAGE_MANAGER -y install openssl binutils

	if [ ! $? -eq 0 ]; then
		echo "ERROR : common tools installation failed"
		exit 1
	fi                                    
}

install_libs()
{  
	echo "Installing libs..."
	for VAR_LIBPATH in "$VAR_INSTALLDIR/lib"/*
	do
		VAR_LIBNAME="${VAR_LIBPATH##*/}"
     
		install -m 644 $VAR_LIBPATH $VAR_LIBDIR;
   
		if [ ! $? -eq 0 ]; then 
			echo "Error : $VAR_LIBNAME library installation error"
			exit 1
		fi

		LIBSONAME=$(objdump -p $VAR_LIBDIR/$VAR_LIBNAME | grep SONAME)
  
		ln -sf ${LIBSONAME#*SONAME} $VAR_LIBDIR/"${VAR_LIBNAME%%.*}".so;
	done
	
	if ! grep -qx "$VAR_LIBDIR" /etc/ld.so.conf.d/libc.conf; then
		echo "$VAR_LIBDIR" >>/etc/ld.so.conf.d/libc.conf
	fi
 
	/sbin/ldconfig -n $VAR_LIBDIR
	/sbin/ldconfig -X
}

uninstall_libs()
{
	echo "Uninstalling libs..."
	for VAR_LIBPATH in "$VAR_INSTALLDIR/lib"/*
	do
		VAR_LIBNAME="${VAR_LIBPATH##*/}"
		 
		rm -rf $VAR_LIBDIR/"${VAR_LIBNAME%.*.*.*.*}"*;         
	done
}

install_dlmcli()
{ 
	echo "Installing dlmcli..."
	install -d /opt/deltacast/dlm/
	install -m 755 ./dlmcli.out  /opt/deltacast/dlm/
	ln -nfrs /opt/deltacast/dlm/dlmcli.out /usr/local/sbin/dlmcli
  
	if [ ! $? -eq 0 ]; then 
		echo "Error : dlmcli installation error"
		exit 1
	fi
} 

uninstall_dlmcli()
{
	echo "Uninstalling dlmcli..."	
	rm -rf /opt/deltacast/dlm
	rm /usr/local/sbin/dlmcli
}

install_services()
{ 
	echo "Installing services..."
	
	install -d /opt/deltacast/dlm/
	install -m 755 ./dlm.sh  /opt/deltacast/dlm/
  
	if [ ! $? -eq 0 ]; then 
		echo "Error : DLM service script installation error"
		exit 1
	fi
	
	cd /lib/systemd/system/   
	
	# DLM service installation				 
	install -m 644 $VAR_INSTALLDIR/dlm.service  /lib/systemd/system/
 	 	
	systemctl enable dlm.service
 
	
	systemctl daemon-reload
	
 
	# DLM service start 	
	systemctl start dlm.service
 
	if [ ! $? -eq 0 ]; then 
		echo "Error : DLM service installation error"
		exit 1
	fi
  	 
 	cd $VAR_INSTALLDIR  
}

uninstall_services()
{
	echo "Uninstalling services..."
	systemctl disable dlm.service
	systemctl stop dlm.service	
	rm -rf /lib/systemd/system/dlm.service
   
	systemctl daemon-reload	
	systemctl reset-failed
	
	rm -rf /opt/deltacast/dlm
} 

stop_services()
{ 
	echo "Stopping services..."
	# Stop services before install               
	systemctl disable dlm.service
	systemctl stop dlm.service	
}   

create_trusted_storage()
{
	echo "Creating trusted storage..."
	
	# Create the trusted storage folder
	install -d -m 777 /var/opt/deltacast/dlm/
	
	# Create trusted storage files by retrieving status
	/opt/deltacast/dlm/dlmcli.out status
	
	# Change trusted storoage files permission
	chmod 777 /var/opt/deltacast/dlm/*
}


case "$1" in 
    clean) 
        check_access_right
		check_package_integrity
		set_member_variables
		uninstall_services
		uninstall_libs
		uninstall_dlmcli
        ;; 
    update) 
		check_access_right
		check_package_integrity
		set_member_variables
		stop_services
      install_libs
		install_dlmcli
		install_services 
		create_trusted_storage
        ;; 
    *) 
		check_access_right
		check_package_integrity
		set_member_variables
		install_common_tools
		install_libs
		install_dlmcli
		install_services
		create_trusted_storage
        ;; 
esac    	   
