#!/bin/sh
###############################################################################
#             IP Virtual Card Linux installation shell script                 #
###############################################################################

check_package_integrity()
{
	VAR_INTEGRITY=1

	if [ ! -d ./kbdpdk ]; then VAR_INTEGRITY=0; fi
	if [ ! -d ./licensing ]; then VAR_INTEGRITY=0; fi
	if [ ! -d ./linuxptp ]; then VAR_INTEGRITY=0; fi
	if [ ! -d ./vcs/ ]; then VAR_INTEGRITY=0; fi
	if [ ! -d ./videomasterip/ ]; then VAR_INTEGRITY=0; fi

	if [ $VAR_INTEGRITY = "0" ]; then
		echo "Package integrity KO"
		echo "Error : the shell script should be executed from VideoMasterIP Linux redist folder or the package is corrupted"
		exit 1
	fi
}

set_member_variables()
{
	VAR_SDKDIR=`pwd` 

	if [ -d /usr/local/lib64 ]; then
		VAR_LIBDIR=/usr/local/lib64
	else
		VAR_LIBDIR=/usr/local/lib
	fi


	if [ -f /etc/os-release ]; then
		. /etc/os-release
		OS=$ID
	fi


	if [ $OS = "redhat"  ]; then
		PACKAGE_MANAGER=yum
	elif [ $OS = "centos"  ]; then
		PACKAGE_MANAGER=yum
	else
		PACKAGE_MANAGER=apt-get
	fi
}

install_common_tools()
{
	# Common tools installation
	$PACKAGE_MANAGER -y update
	$PACKAGE_MANAGER -y install git g++ gcc cmake libwebsocketpp-dev openssl libssl-dev ninja-build libboost-all-dev

	if [ ! $? -eq 0 ]; then
		echo "ERROR : common tools installation failed"
		exit 1
	fi


	# Build and install cpprestsdk package
	CPP_REST_SDK_VERSION=2.10.16

	DIR=$(mktemp -d)
	cd $DIR/
	git clone https://github.com/Microsoft/cpprestsdk.git cpprestsdk

	if [ ! $? -eq 0 ]; then
		echo "ERROR: git cpprestsdk clone failed"
		exit 1
	fi

	cd cpprestsdk
	git checkout -b $CPP_REST_SDK_VERSION v$CPP_REST_SDK_VERSION

	if [ ! $? -eq 0 ]; then
		echo "ERROR: git cpprestsdk checkout failed"
		exit 1
	fi

	cd ./cpprestsdk-$CPP_REST_SDK_VERSION                             
	cmake -G Ninja -DCMAKE_BUILD_TYPE=Release
	ninja
	ninja install

	if [ ! $? -eq 0 ]; then
		echo "ERROR: cpprestsdk installation failed"
		exit 1
	fi

	rm -rf $DIR

 	cd $VAR_SDKDIR 
                                     
}

install_ptp()
{
	# Disabling NTP
	timedatectl set-ntp 0

	if [ ! $? -eq 0 ]; then
		echo "ERROR: NTP disabling failed"
		exit 1
	fi


	# Build+patch linuxptp instead of installing pre-built package
	PTP_VERSION=2.0
	DIR=$(mktemp -d)
	cd $DIR/
	git clone http://git.code.sf.net/p/linuxptp/code linuxptp

	if [ ! $? -eq 0 ]; then
		echo "ERROR: git linuxptp clone failed"
		exit 1
	fi

	cd linuxptp
	git checkout -b $PTP_VERSION v$PTP_VERSION

	if [ ! $? -eq 0 ]; then
		echo "ERROR: git linuxptp checkout failed"
		exit 1
	fi

	patch -p1 < $VAR_SDKDIR/linuxptp/donotrespondst2059_2tlv.patch
	patch -p1 < $VAR_SDKDIR/linuxptp/kni.patch
	make
	make install

	if [ ! $? -eq 0 ]; then
		echo "ERROR: PTP installation failed"
		exit 1
	fi

	make distclean
	rm -rf $DIR

	install -d /etc/linuxptp/ 
	install -m 644 $VAR_SDKDIR/linuxptp/ptp4l.conf     /etc/linuxptp/ptp4l.conf

	if [ ! $? -eq 0 ]; then
		echo "ERROR: PTP .conf file installation failed"
		exit 1
	fi
 
	cd $VAR_SDKDIR                                     
 }   

install_libs()
{  
	for VAR_LIBPATH in "$VAR_SDKDIR/videomasterip"/*
	do
	    [ -f "$VAR_LIBPATH" ] || continue
     	VAR_LIBNAME="${VAR_LIBPATH##*/}"
     
		install -m 644 $VAR_LIBPATH $VAR_LIBDIR;
   
		if [ ! $? -eq 0 ]; then 
			echo "Error : $VAR_LIBNAME library installation error"
			exit 1
		fi

		LIBSONAME=$(objdump -p $VAR_LIBDIR/$VAR_LIBNAME | grep SONAME)
  
		ln -sf ${LIBSONAME#*SONAME} $VAR_LIBDIR/"${VAR_LIBNAME%%.*}".so;
	done
	
	if ! grep -qx "$VAR_LIBDIR" /etc/ld.so.conf.d/libc.conf; then
		echo "$VAR_LIBDIR" >>/etc/ld.so.conf.d/libc.conf
	fi
 
	/sbin/ldconfig -n $VAR_LIBDIR
	/sbin/ldconfig -X
}

install_vcs()
{ 
	install -d /opt/deltacast/videomasterip/vcs/
    #back-up the configuration files
	install -b -m 755 ./vcs/vcsconfig*  /opt/deltacast/videomasterip/vcs/
	install -m 755 ./vcs/*  /opt/deltacast/videomasterip/vcs/
  
	if [ ! $? -eq 0 ]; then 
		echo "Error : VCS installation error"
		exit 1
	fi
} 

stop_services()
{ 
	# Stop services before install               
	systemctl disable vcs.service
	systemctl stop vcs.service
	
	systemctl disable ptp4l.service
	systemctl stop ptp4l.service
	
	systemctl disable network_sysctl.service
	systemctl stop network_sysctl.service
}     

install_licensing()
{
	VAR_LICENSINGDIR=$VAR_SDKDIR/licensing
	
	if [ ! -d "$VAR_LICENSINGDIR/dlmcli" ] 
	then
		mkdir -p $VAR_LICENSINGDIR/dlmcli 
		tar -xvf $VAR_LICENSINGDIR/dlmcli.tar.gz -C $VAR_LICENSINGDIR/dlmcli
	fi

	cd $VAR_LICENSINGDIR/dlmcli/x64

	sh install_dlm.sh

	cd $VAR_SDKDIR
}

install_services()
{ 
	# VCS service installation               
	install -m 644 ./vcs/vcs.service  /lib/systemd/system/
	cd /lib/systemd/system/   
	
	systemctl enable vcs.service

 
	# PTP service installation				 
	install -m 644 $VAR_SDKDIR/linuxptp/ptp4l.service  /lib/systemd/system/
 	 	
	systemctl enable ptp4l.service
 
 
	# Network sysctl service installation				 
	install -m 644 $VAR_SDKDIR/vcs/network_sysctl.service  /lib/systemd/system/
	
	systemctl enable network_sysctl.service
 		
		
	systemctl daemon-reload
	
 
	# VCS service start 	
	systemctl start vcs.service
 
	if [ ! $? -eq 0 ]; then 
		echo "Error : VCS service installation error"
		exit 1
	fi
  	
   
	# PTP service start
	systemctl start ptp4l.service
 
	if [ ! $? -eq 0 ]; then 
		echo "Error : ptp4l service installation error"
		exit 1
	fi


	# Network sysctl service start	
	systemctl start network_sysctl.service
 
	if [ ! $? -eq 0 ]; then 
		echo "Error : Network sysctl service installation error"
		exit 1
	fi
 
	 
 	cd $VAR_SDKDIR  
}

uninstall_libs()
{
	for VAR_LIBPATH in "$VAR_SDKDIR/videomasterip"/*
	do
	    [ -s "$VAR_LIBPATH" ] || continue
		VAR_LIBNAME="${VAR_LIBPATH##*/}"
		 
		rm -rf $VAR_LIBDIR/"${VAR_LIBNAME%.*.*.*.*}"*;         
	done
}

uninstall_licensing()
{
	VAR_LICENSINGDIR=$VAR_SDKDIR/licensing

	echo "${VAR_LICENSINGDIR}/dlmcli"
	if [ ! -d "$VAR_LICENSINGDIR/dlmcli" ] 
	then 
		mkdir -p $VAR_LICENSINGDIR/dlmcli
		tar -xaf $VAR_LICENSINGDIR/dlmcli.tar.gz -C $VAR_LICENSINGDIR/dlmcli
	fi

	cd $VAR_LICENSINGDIR/dlmcli/x64

	sh install_dlm.sh clean

	cd $VAR_SDKDIR
}

update_licensing()
{
	VAR_LICENSINGDIR=$VAR_SDKDIR/licensing
	
	if [ ! -d "$VAR_LICENSINGDIR/dlmcli" ] 
	then
		mkdir -p $VAR_LICENSINGDIR/dlmcli 
		tar -xvf $VAR_LICENSINGDIR/dlmcli.tar.gz -C $VAR_LICENSINGDIR/dlmcli
	fi

	cd $VAR_LICENSINGDIR/dlmcli/x64

	sh install_dlm.sh update

	cd $VAR_SDKDIR
}

uninstall_vcs()
{
	rm -rf /opt/deltacast
}

uninstall_services()
{
	systemctl disable vcs.service
	systemctl stop vcs.service	
	rm -rf /lib/systemd/system/vcs.service
 
	systemctl disable network_sysctl.service
	systemctl stop network_sysctl.service	
	rm -rf /lib/systemd/system/network_sysctl.service
  
	systemctl daemon-reload	
	systemctl reset-failed
} 

check_access_right()
{
	if [ $(id -u) -ne 0 ]; then 
		echo "Please run as root"
		exit 1 
	fi
}

case "$1" in 
    clean) 
      check_access_right
		check_package_integrity
		set_member_variables
		uninstall_services
		uninstall_licensing
		uninstall_libs
		uninstall_vcs
      ;; 
    update) 
		check_access_right
		check_package_integrity
		set_member_variables
		stop_services
		uninstall_libs
      install_libs
		update_licensing
		install_vcs
		install_services
		;; 
    *) 
		check_access_right
		check_package_integrity
		set_member_variables
		install_common_tools
		stop_services
		uninstall_libs
		install_ptp
		install_libs
		install_licensing
		install_vcs
		install_services 
      ;; 
esac    	   
